/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire;


import static fr.histoiremondiale.histoire.utiles.exttypes.Flottants.arrondir;
import fr.histoiremondiale.histoire.donnees.Civilisation;
import fr.histoiremondiale.histoire.donnees.Territoire;
import fr.histoiremondiale.histoire.utiles.igraphique.swing.Observable;
import fr.histoiremondiale.histoire.utiles.math.PlagesNombres;
import fr.histoiremondiale.histoire.utiles.math.PointSphere;


/**
 * Classe rassemblant les informations sur l'état de l'application : les valeurs qui changent
 *   lorsque l'utilisateur manipule l'application.<br>
 * Note : la classe est chargée de s'assurer que les valeurs qu'elle contient sont toujours valides
 *   (y compris si elle est initialisée avec des valeurs invalides).
 *   
 *   
 */
public class EtatAppli extends Observable
{

    // Constantes
    public static final int[]  VALS_PAS_ANNEES  = {1, 5, 10, 25, 50, 100, 200, 500} ;   // Valeurs possibles pour le pas de déplacement de la date (en années)
    public static final int[]  VALS_PAS_ESPACE  = {1, 2, 5, 10, 15, 20} ;               // Valeurs possibles pour le pas de déplacement des longitudes/latitudes (en degrés)
    // (loupe)
    public static final double MINI_LOUPE       = 1.0 ;                                 // Valeur minimale et maximale de la loupe
    public static final double MAXI_LOUPE       = 25.0 ;
    public static final double INCR_LOUPE       = 1.3 ;//Math.exp (Math.log (MAXI_LOUPE/(MAXI_LOUPE - 1))) ;
    // (coordonnées dans l'espace)
    public static final int    PRECISION_ESPACE = 2 ;                                   // Précision des coordonnées spatiales (en nombre de chiffres après la virgule)

    
    // Coordonnées espace/temps
    private PointSphere ptCentralCarte ;        // Coordonnées du point central de la carte
    private int         annee ;                 // Année courante
    private double      loupe ;                 // Valeur de grossissement de la loupe

    // Civilisation sélectionnée/suivie
    private Civilisation civilisationSelectionnee ;         // Civilisation sélectionnée (navigateur html notamment)
    private boolean      suiviCivilisationSelectionnee ;    // Indique si la civilisation sélectionnée est suivie (<=> s'il faut utiliser cette civilisation pour modifier automatiquement le navigateur par exemple)
    
    // Options d'affichage
    private boolean fleuvesAffiches ;               // Indique s'il faut tracer les fleuves sur la carte
    private boolean infobullesCarte ;               // Indique si les infobulles de la carte s'affichent ou non
    private boolean meridiensParallelesAffiches ;   // Indique s'il faut dessiner les méridiens et les parallèles
    private boolean symboleAttenteTracer ;          // Indique s'il faut afficher le symbole indiquant qu'une carte est en cours de tracer
    private boolean texteSurvolFondClair ;          // Indique si le texte concernant le territoire survolé doit être sur fond clair
    private double  posSeparateurCarteNavig ;       // Position du séparateur entre les deux panneaux carte et navigateur
    private int     iDeplAnnees ;                   // Indice du pas de déplacement pour les années
    private int     iDeplEspace ;                   // Indice du pas de déplacement dans l'espace
    
    // Autres optionss de l'interface
    private boolean simpleClicFermerParagraphes ;   // Indique si un simple clic ferme le dialogue des paragraphes
    private boolean moletteGlisseDeUn ;             // Indique si un mouvement de glissière à la molette décale de 1 ou de l'incrément prévu
    private boolean affParagraphesHtmlGeneres ;     // Indique si les pages html qui sont générées doivent être affichées automatiquement dans une fenêtre (pour vérification)
    
    // Infos sur l'affichage temporaire (affichage d'infos en cas de survol de la souris, ...)
    private Territoire territoireSurvole ;          // Territoire en cours de survol par la souris
    
    // Chemins sélectionnés
    private String chemRepSauvCartes ;              // Chemin du répertoire de sauvegarde des cartes
    private String chemRepSauvParagraphes ;         // Chemin du répertoire de sauvegarde des paragraphes html
    private String chemRepVueParagraphes ;          // Chemin du répertoire de visuallisation des paragraphes html sauvegardés

    
    
    // Accesseurs
    public PointSphere ptCentralCarte () { return this.ptCentralCarte ; }
    public double      longitude      () { return this.ptCentralCarte.longitude() ; }
    public double      latitude       () { return this.ptCentralCarte.latitude() ; }
    public double      loupe          () { return this.loupe ; }
    public int         annee          () { return this.annee ; }

    /**
     * Renvoie la civilisation suivie, s'il y en a une.
     * @return La civilisation suivie, null si le suivi est désactivé ou qu'aucune civilisation n'est sélectionnée.
     */
    public synchronized Civilisation civilisationSuivie ()
    {
        return (this.suiviCivilisationSelectionnee ? this.civilisationSelectionnee : null) ;
    }
    /**
     * Renvoie le nom de la civilisation, y compris si le suivi n'est pas actif<br>
     * Renvoie null s'il n'y a pas du tout de civilisation sélectionnée).
     * @return Le nom de la civilisation, peu importe l'état du suivi, null si pas de civilisation
     *         sélectionnée.
     */
    public synchronized String nomCivilisationSelectionnee ()
    {
        return (this.civilisationSelectionnee == null ? null : this.civilisationSelectionnee.nom()) ;
    }
    public synchronized boolean suiviCivilisationSelectionnee ()
    {
        return this.suiviCivilisationSelectionnee ;
    }
    /**
     * Renvoie la civilisation sélectionnée, même si elle n'est pas suivie.<br>
     * Renvoie null si aucune civilisation n'est sélectionnée.
     * @return La civilisation sélectionnée.
     */
    public synchronized Civilisation civilisationSelectionnee ()
    {
        return this.civilisationSelectionnee ;
    }
    
    
    // CODE Il faudrait revoir pourquoi ces méthode sont synrhonisées (et vérifier que c'est aussi le cas des
    //      modificateurs associés) : est-ce que tout ça n'est pas manipulé uniquement dans le fil de Swing ?
    //      (certaines infos sont sans doute récupérées au moment de tracer des cartes)
    public synchronized boolean fleuvesAffiches             () { return this.fleuvesAffiches ; }
    public synchronized boolean infobullesCarte             () { return this.infobullesCarte ; }
    public synchronized boolean meridiensParallelesAffiches () { return this.meridiensParallelesAffiches ; }
    public synchronized boolean symboleAttenteTracer        () { return this.symboleAttenteTracer ; }
    public synchronized boolean texteSurvolFondClair        () { return this.texteSurvolFondClair ; }
    public synchronized double  posSeparateurCarteNavig     () { return this.posSeparateurCarteNavig ; }
    
    public synchronized int     iDeplAnnees                 () { return this.iDeplAnnees ; }
    public synchronized int     deplAnnees                  () { return VALS_PAS_ANNEES[this.iDeplAnnees] ; }
    public synchronized int     iDeplEspace                 () { return this.iDeplEspace ; }
    public synchronized int     deplEspace                  () { return VALS_PAS_ESPACE[this.iDeplEspace] ; }
    
    public synchronized boolean simpleClicFermerParagraphes () { return this.simpleClicFermerParagraphes ; }
    public synchronized boolean moletteGlisseDeUn           () { return this.moletteGlisseDeUn ; }
    public synchronized boolean affParagraphesHtmlGeneres   () { return this.affParagraphesHtmlGeneres ; }
    
    public              Territoire territoireSurvole        () { return this.territoireSurvole ; }

    public              String chemRepSauvCartes            () { return this.chemRepSauvCartes ; }
    public              String chemRepSauvParagraphes       () { return this.chemRepSauvParagraphes ; }
    public              String chemRepVueParagraphes        () { return this.chemRepVueParagraphes ; }

    
    /**
     * Modifie la longitude du point central de la carte.
     * @param longitude La longitude.
     */
    public void modifLongitude (double longitude)
    {
        modifCoordonneesEspaceTemps (longitude, null, null) ;
    }
    
    /**
     * Modifie la latitude du point central de la carte.
     * @param latitude La latitude.
     */
    public void modifLatitude (double latitude)
    {
        modifCoordonneesEspaceTemps (null, latitude, null) ;
    }
    
    /**
     * Modifie le point central de la carte.
     * @param ptCentral Le nouveau point central.
     */
    public void modifPtCentralCarte (PointSphere ptCentral)
    {
        if (ptCentral!=null) modifCoordonneesEspaceTemps (ptCentral.longitude(), ptCentral.latitude(), null) ;
    }
    
    /**
     * Augmente le grossissement (loupe) d'un niveau.
     */
    public void augmenterLoupe()
    {
        modifLoupe (loupe() * INCR_LOUPE) ;
    }
    
    /**
     * Diminue le grossissement (loupe) d'un niveau.
     */
    public void diminuerLoupe ()
    {
        modifLoupe (loupe() / INCR_LOUPE) ;
    }
    
    /**
     * Modifie le grossissement du nombre de niveaux passé en paramètre.
     * @param decalageLoupe Le nombre de niveaux.
     */
    public void modifRelativeLoupe (double decalageLoupe)
    {
        modifLoupe (loupe() * Math.pow (INCR_LOUPE, decalageLoupe)) ;
    }
    
    
    /**
     * Modifie la valeur de la loupe.
     * @param loupe Nouvelle valeur.
     */
    public void modifLoupe (double loupe)
    {
        double loupeAjustee = loupe ;
        if (loupeAjustee > MAXI_LOUPE) loupeAjustee = MAXI_LOUPE ;
        if (loupeAjustee < MINI_LOUPE) loupeAjustee = MINI_LOUPE ;
        
        if (this.loupe != loupeAjustee)
        {
            this.loupe = loupeAjustee ;
            avertirObservateurs ("loupe") ;
        }
        
    }
    
    /**
     * Modifie l'année.
     * @param annee La nouvelle année.
     */
    public void modifAnnee (int annee)
    {
    	modifCoordonneesEspaceTemps (null, null, annee) ;
    }
    
    /**
     * Décrémente l'année du pas de déplacement des années.
     */
    public void decrementerAnnee ()
    {
        modifAnnee (this.annee - VALS_PAS_ANNEES[this.iDeplAnnees]) ;
    }
    /**
     * Incrémente l'année du pas de déplacement des années.
     */
    public void incrementerAnnee ()
    {
        modifAnnee (this.annee + VALS_PAS_ANNEES[this.iDeplAnnees]) ;
    }

    /**
     * Modifie la longitude, la latitude et l'année en même temps.<br>
     * Si certains paramètres valent null, ils ne sont pas modifiés.
     * @param ptCentralCarte Nouveau point central.
     * @param annee          Nouvelle année.
     * @param versLeFutur 
     */
    public void modifCoordonneesEspaceTemps (PointSphere ptCentralCarte, Integer annee, boolean versLeFutur)
    {
        Double longitude = (ptCentralCarte == null ? null : ptCentralCarte.longitude()) ;
        Double latitude  = (ptCentralCarte == null ? null : ptCentralCarte.latitude()) ;
        modifCoordonneesEspaceTemps (longitude, latitude, annee, versLeFutur) ;
    }
    /**
     * Modifie la longitude, la latitude et l'année en même temps.<br>
     * Si certains paramètres valent null, ils ne sont pas modifiés.
     * @param longitude Nouvelle longitude.
     * @param latitude  Nouvelle latitude.
     * @param annee     Nouvelle année.
     */
    public void modifCoordonneesEspaceTemps (Double longitude, Double latitude, Integer annee)
    {
        modifCoordonneesEspaceTemps (longitude, latitude, annee, true) ;
    }
    /**
     * Modifie la longitude, la latitude et l'année en même temps.<br>
     * Si certains paramètres valent null, ils ne sont pas modifiés.
     * @param longitude Nouvelle longitude.
     * @param latitude  Nouvelle latitude.
     * @param annee     Nouvelle année.
     * @param versLeFutur Indique si on décale les années vers le futur (pour le cas année = 0 : indication sur le choix à effectuer (année -1 ou +1)).
     */
    public void modifCoordonneesEspaceTemps (Double longitude, Double latitude, Integer annee, Boolean versLeFutur)
    {
        boolean modifEspace = false ;
        boolean modifAnnees = false ;
//        if (ptCentralCarte == null)
//        {
//        	MethodeCourante.getPileExecutionSimplifiee();
//        }
        //cela pose un problème quand le point central est null
        //il faut vérifier dans quel cas cela arrive !
        double ancLongitude = (this.ptCentralCarte == null ? 0.0 : this.ptCentralCarte.longitude()) ;
        double ancLatitude  = (this.ptCentralCarte == null ? 0.0 : this.ptCentralCarte.latitude()) ;
        int    ancAnnee     = this.annee ;

        // Calculer les valeurs ajustées
        double longitudeAjustee = arrondir ((longitude == null ? ancLongitude : PlagesNombres.dansPlageCyclique (longitude, -179,  +180)), PRECISION_ESPACE) ;
        double latitudeAjustee  = arrondir ((latitude  == null ? ancLatitude  : PlagesNombres.dansPlage         (latitude,   -90,   +90)), PRECISION_ESPACE) ;
        int    anneeAjustee     = (annee     == null ? ancAnnee     : PlagesNombres.dansPlage         (annee,    -3100, +2000)) ;
        if (anneeAjustee == 0)
            anneeAjustee = (versLeFutur ? +1 : -1) ;      // (pas d'année 0)


        // Mettre à jour les valeurs
        if (this.ptCentralCarte == null ||
            // (note : garder les deux versions)
            // longitudeAjustee != ancLongitude ||                // Version qui ne fait pas passer le curseur d'une extrémité à l'autre (valeur cyclique des longitudes)
            (longitude != null && longitude != ancLongitude) ||   // Version qui génére sans doute un peu plus d'événements, mais permet de faire passer le curseur de gauche à droite quand on le colle à gauche (et réciproquement), comme ça le fait avec les flèches aussi
            latitudeAjustee  != ancLatitude)
        {
            this.ptCentralCarte = new PointSphere (longitudeAjustee, latitudeAjustee) ;
            modifEspace = true ;
        }
        if (anneeAjustee != ancAnnee)
        {
            this.annee = anneeAjustee ;
            modifAnnees = true ;
        }

        // Avertir les observateurs
        if (modifEspace)
        {
            avertirObservateurs ("ptCentralCarte") ;
            avertirObservateurs ("longitude") ;
            avertirObservateurs ("latitude") ;
        }
        if (modifAnnees)
            avertirObservateurs ("annee") ;
    }
    
    
    /**
     * Modifie la civilisation sélectionnée.<br>
     * La civilisation peut être null.
     * @param civilisationSelectionnee La nouvelle civilisation sélectionnée.
     */
    public synchronized void modifCivilisationSelectionnee (Civilisation civilisationSelectionnee)
    {
        if (this.civilisationSelectionnee != civilisationSelectionnee)
        {
            this.civilisationSelectionnee = civilisationSelectionnee ;
            avertirObservateurs ("civilisationSelectionnee") ;
        }
    }

    /**
     * Active ou désactive le suivi de la civilisation sélectionnée.
     * @param suiviCivilisationSelectionnee Indique si le suivi est actif ou non.
     */
    public synchronized void modifSuiviCivilisationSelectionnee (boolean suiviCivilisationSelectionnee)
    {
        if (suiviCivilisationSelectionnee != this.suiviCivilisationSelectionnee)
        {
            this.suiviCivilisationSelectionnee = suiviCivilisationSelectionnee ;
            avertirObservateurs ("suiviCivilisationSelectionnee") ;
        }
    }
    /**
     * Inverse le suivi de la civilisation.
     */
    public synchronized void inverserSuiviCivilisationSelectionnee ()
    {
        modifSuiviCivilisationSelectionnee (! suiviCivilisationSelectionnee()) ;
    }
    

    /**
     * Inverse la valeur de l'option "fleuves affichés".
     */
    public synchronized void inverserFleuvesAffiches ()
    {
        modifFleuvesAffiches (! fleuvesAffiches()) ;
    }
    /**
     * Modifie l'option "fleuves affichés sur la carte".
     * @param fleuvesAffiches Indique si les fleuves doivent être affichés.
     */
    public synchronized void modifFleuvesAffiches (boolean fleuvesAffiches)
    {
        if (this.fleuvesAffiches != fleuvesAffiches)
        {
            this.fleuvesAffiches = fleuvesAffiches ;
            avertirObservateurs ("fleuvesAffiches") ;
        }
    }

    /**
     * Inverse la valeur de l'option "infobulles carte".
     */
    public synchronized void inverserInfobullesCarte ()
    {
        modifInfobullesCarte (! infobullesCarte()) ;
    }
    /**
     * Modifie l'option "infobulles carte".
     * @param infobulles Indique si les infobulles doivent être affichées.
     */
    public synchronized void modifInfobullesCarte (boolean infobulles)
    {
        if (this.infobullesCarte != infobulles)
        {
            this.infobullesCarte = infobulles ;
            avertirObservateurs ("infobullesCarte") ;
        }
    }
    
    /**
     * Inverse la valeur de l'option "méridiens et parallèles affichés"
     */
    public synchronized void inverserMeridiensParallelesAffiches ()
    {
        modifMeridiensParallelesAffiches (! meridiensParallelesAffiches()) ;
    }
    /**
     * Modifie la valeur de l'option "méridiens et parallèles affichés".
     * @param meridiensParallelesAffiches Indique si l'option est active ou non.
     */
    public synchronized void modifMeridiensParallelesAffiches (boolean meridiensParallelesAffiches)
    {
        if (this.meridiensParallelesAffiches != meridiensParallelesAffiches)
        {
            this.meridiensParallelesAffiches = meridiensParallelesAffiches ;
            avertirObservateurs ("MeridiensParallelesAffiches") ;
        }
    }

    /**
     * Inverse la valeur de l'option "symbole de tracers en cours".
     */
    public synchronized void inverserSymboleAttenteTracer ()
    {
        modifSymboleAttenteTracer (! symboleAttenteTracer()) ;
    }
    /**
     * Modifie la valeur de l'option "symbole de tracers en cours".
     * @param symboleAttenteTracer Nouvelle valeur.
     */
     public synchronized void modifSymboleAttenteTracer (boolean symboleAttenteTracer)
    {
        if (this.symboleAttenteTracer != symboleAttenteTracer)
        {
            this.symboleAttenteTracer = symboleAttenteTracer ;
            avertirObservateurs ("symboleAttenteTracer") ;
        }
    }
    
    /**
     * Inverse la valeur de l'option "texte territoire survolé sur fond clair".
     */
    public synchronized void inverserTexteSurvolFondClair ()
    {
        modifTexteSurvolFondClair (! texteSurvolFondClair()) ;
    }
    /**
     * Modifie la valeur de l'option "texte territoire survolé sur fond clair".
     * @param texteSurvolFondClair Nouvelle valeur.
     */
    public synchronized void modifTexteSurvolFondClair (boolean texteSurvolFondClair)
    {
        if (this.texteSurvolFondClair != texteSurvolFondClair)
        {
            this.texteSurvolFondClair = texteSurvolFondClair ;
            avertirObservateurs ("texteSurvolFondClair") ;
        }
    }
    
    /**
     * Modifie la proportion entre les panneaux carte et navigateur.<br>
     * Note : Cette propriété est utile pour l'initialisation de l'interface, elle n'a pas besoin d'être mise
     *        à jour dès que l'information est modifiée.
     * @param posSeparateurCarteNavig Nouvelle valeur.
     */
    public synchronized void modifPosSeparateurCarteNavig (double posSeparateurCarteNavig)
    {
        if (this.posSeparateurCarteNavig != posSeparateurCarteNavig)
        {
            this.posSeparateurCarteNavig = posSeparateurCarteNavig ;
            avertirObservateurs ("posSeparateurCarteNavig") ;
        }
    }
    
    
    /**
     * Modifie le pas de déplacement dans les années.
     * @param iDeplAnnees Indice dans le tableau des valeurs possible.
     */
    public void modifIDeplAnnees (int iDeplAnnees)
    {
        int iDeplAnneesAjuste = iDeplAnnees ;
        iDeplAnneesAjuste = Math.min (iDeplAnneesAjuste, VALS_PAS_ANNEES.length - 1) ;
        iDeplAnneesAjuste = Math.max (iDeplAnneesAjuste, 0) ;
        if (this.iDeplAnnees != iDeplAnneesAjuste)
        {
            this.iDeplAnnees = iDeplAnneesAjuste ;
            avertirObservateurs ("iDeplAnnees") ;
            avertirObservateurs ("deplAnnees") ;
        }
    }
    /**
     * Modifie le pas de déplacement dans les années.
     * @param deplAnnees Nouvelle valeur choisie. La valeur la plus proche dans le tableau des valeurs possibles est sélectionnée.
     */
    public void modifDeplAnnees (int deplAnnees)
    {
        
        int iDeplAnnees = -1 ;      // Indice dans le tableau correspondant à la valeur la plus proche trouvée

        // Chercher la valeur la plus proche
        for (int i = 0 ; i < VALS_PAS_ANNEES.length ; i++)
        {
            if (VALS_PAS_ANNEES[i] >= deplAnnees)
            {
                iDeplAnnees = i ;
                break ;
            }
        }
        
        // Si pas de valeur trouvée, prendre la dernière
        if (iDeplAnnees == -1)
            iDeplAnnees = VALS_PAS_ANNEES.length -1 ;
        
        
        // Modifier la valeur
        modifIDeplAnnees (iDeplAnnees) ;
    }

    /**
     * Modifie le pas de déplacement dans l'espace.
     * @param iDeplEspace Indice dans le tableau des valeurs possible.
     */
    public void modifIDeplEspace (int iDeplEspace)
    {
        int iDeplEspaceAjuste = iDeplEspace ;
        iDeplEspaceAjuste = Math.min (iDeplEspaceAjuste, VALS_PAS_ESPACE.length - 1) ;
        iDeplEspaceAjuste = Math.max (iDeplEspaceAjuste, 0) ;
        if (this.iDeplEspace != iDeplEspaceAjuste)
        {
            this.iDeplEspace = iDeplEspaceAjuste ;
            avertirObservateurs ("iDeplEspace") ;
            avertirObservateurs ("deplEspace") ;
        }
    }
    /**
     * Modifie le pas de déplacement dans l'espace.
     * @param deplEspace Nouvelle valeur choisie. La valeur la plus proche dans le tableau des valeurs possibles est sélectionnée.
     */
    public void modifDeplEspace (int deplEspace)
    {
        
        int iDeplEspace = -1 ;      // Indice dans le tableau correspondant à la valeur la plus proche trouvée

        // Chercher la valeur la plus proche
        for (int i = 0 ; i < VALS_PAS_ESPACE.length ; i++)
        {
            if (VALS_PAS_ESPACE[i] >= deplEspace)
            {
                iDeplEspace = i ;
                break ;
            }
        }
        
        // Si pas de valeur trouvée, prendre la dernière
        if (iDeplEspace == -1)
            iDeplEspace = VALS_PAS_ESPACE.length -1 ;
        
        // Modifier la valeur
        modifIDeplEspace (iDeplEspace) ;
    }
    
    
    /**
     * Inverse la valeur de l'option "simple clic ferme le dialogue des paragraphes".
     */
    public synchronized void inverserSimpleClicFermerParagraphes ()
    {
        modifSimpleClicFermerParagraphes (! simpleClicFermerParagraphes()) ;
    }
    /**
     * Modifie la valeur de l'option "simple clic ferme le dialogue des paragraphes".
     * @param simpleClicFermerParagraphes Nouvelle valeur.
     */
    public synchronized void modifSimpleClicFermerParagraphes (boolean simpleClicFermerParagraphes)
    {
        if (this.simpleClicFermerParagraphes != simpleClicFermerParagraphes)
        {
            this.simpleClicFermerParagraphes = simpleClicFermerParagraphes ;
            avertirObservateurs ("simpleClicFermerParagraphes") ;
        }
    }
    
    /**
     * Inverse la valeur de l'option "la molette glisse de 1".
     */
    public synchronized void inverserMoletteGlisseDeUn ()
    {
        modifMoletteGlisseDeUn (! moletteGlisseDeUn()) ;
    }
    /**
     * Modifie la valeur de l'option "la molette glisse de 1".
     * @param moletteGlisseDeUn Nouvelle valeur.
     */
    public synchronized void modifMoletteGlisseDeUn (boolean moletteGlisseDeUn)
    {
        if (this.moletteGlisseDeUn != moletteGlisseDeUn)
        {
            this.moletteGlisseDeUn = moletteGlisseDeUn ;
            avertirObservateurs ("moletteGlisseDeUn") ;
        }
    }
    
    
    /**
     * Inverse la valeur de l'option "les pages html générées sont affichées dans une fenêtre".
     */
    public synchronized void inverserAffParagraphesHtmlGeneres ()
    {
        modifAffParagraphesHtmlGeneres (! affParagraphesHtmlGeneres()) ;
    }
    /**
     * Modifie la valeur de l'option "les pages html générées sont affichées dans une fenêtre".
     * @param affParagraphesHtmlGeneres Nouvelle valeur.
     */
    public synchronized void modifAffParagraphesHtmlGeneres (boolean affParagraphesHtmlGeneres)
    {
        if (this.affParagraphesHtmlGeneres != affParagraphesHtmlGeneres)
        {
            this.affParagraphesHtmlGeneres = affParagraphesHtmlGeneres ;
            avertirObservateurs ("affParagraphesHtmlGeneres") ;
        }
    }
    
    
    /**
     * Modifie le territoire actuellement survolé par la souris.
     * @param territoireSurvole Le territoire survolé.
     */
    public void modifTerritoireSurvole (Territoire territoireSurvole)
    {
        if (this.territoireSurvole != territoireSurvole)
        {
            this.territoireSurvole = territoireSurvole ;
            avertirObservateurs ("territoireSurvole") ;
        }
    }

    /**
     * Modifie les dimensions de la zone disponible pour tracer la carte.<br>
     * (pas de paramètre, mais une notification est envoyée aux observateurs)
     */
    public void modifTailleZoneCarte ()
    {
        // Avertir les observateurs
        // (l'information n'est pas mémorisée).
        avertirObservateurs ("taillePourCarte") ;
    }
    

    /**
     * Modifie le chemin proposé pour sauvegarder les cartes.
     * @param chemRepSauvCartes Le chemin du répertoire dans lequel sauvegarder les cartes.
     */
    public void modifChemRepSauvCartes (String chemRepSauvCartes)
    {
        this.chemRepSauvCartes = chemRepSauvCartes ;
    }

    /**
     * Modifie le chemin proposé pour sauvegarder les paragraphes html.
     * @param chemRepSauvParagraphes Le chemin du répertoire dans lequel sauvegarder les paragraphes html.
     */
    public void modifChemRepSauvParagraphes (String chemRepSauvParagraphes)
    {
        this.chemRepSauvParagraphes = chemRepSauvParagraphes ;
    }

    /**
     * Modifie le chemin proposé pour visualiser les paragraphes html sauvegardés.
     * @param chemRepVueParagraphes Le chemin du répertoire dans lequel aller chercher les paragraphes html.
     */
    public void modifChemRepVueParagraphes (String chemRepVueParagraphes)
    {
        this.chemRepVueParagraphes = chemRepVueParagraphes ;
    }
    
}
