/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.donnees;

import java.util.HashSet ;
import java.util.Set ;



import fr.histoiremondiale.histoire.utiles.math.PointSphere;


/**
 * Représente une civlisation (grande région du monde, culture, ...).<br>
 * Regroupe les différentes formes d'une civilisation au cours du temps (ses territoires).
 */
public class Civilisation
{
    
    private int                     id ;
    private String                  nom ;                       // Nom de la civilisation
    private Set<CentreCivilisation> centres = new HashSet<CentreCivilisation>() ;   // Centres de la civilisation au cours du temps
    
    
    
    /**
     * Constructeur avec les attributs simples.
     * @param id  L'identifiant de la civilisation.
     * @param nom Le nom de la civilisation.
     */
    public Civilisation (int id, String nom)
    {
        this.id  = id ;
        this.nom = nom ;
    }
    
    
    
    /**
     * Renvoie le centre d'une civilisation à une date donnée.<br>
     * Lève une exception s'il n'y a pas de centre à cette époque.
     * Auparavant il fallait lever une exception, maintenant ce cas est accepté 
     * car certains peuples nomades disparaissent pendant quelques années avant 
     * de réapparaitre (les Vandales en 407 et 408)
     * @param annee L'année pour laquelle on veut obtenir le centre.
     * @return Le centre de la civilisation à cette date ou null s'il n'y a pas de centre
     */
    public PointSphere ptCentral (int annee)
    {
        for (CentreCivilisation centre : this.centres)
        {
            if (centre.debut() <= annee && annee <= centre.fin())
                return new PointSphere (centre.longitude(), centre.latitude()) ;
        }
        return null;
//        throw new IllegalArgumentException ("Pas de centre pour la civilisation " + this.nom + " l'année " + annee) ;
    }
    
    
    // Egalité
    public boolean equals (Object o)
    {
        if (o == null)
            return false ;
        if (o == this)
            return true ;
        if (o.getClass() != this.getClass())
            return false ;
        
        // Comparer les objets
        return this.id == ((Civilisation)o).id ;
    }
    
    
    // Hachage
    public int hashCode ()
    {
        return this.id ;
    }
    
    
    // Accesseurs
    public int                     id      () { return this.id ; }
    public String                  nom     () { return this.nom ; }
    public Set<CentreCivilisation> centres () { return this.centres ; }
    
}
