/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.donnees;

import java.util.ArrayList ;
import java.util.List ;

import fr.histoiremondiale.histoire.utiles.math.PointSphere;





/**
 * Représente le contour d'une zone.
 */
public class Pourtour
{

    private int                id ;                     // Identifiant du pourtour
    private List<LimesOrdonne> limesOrdonnes ;          // Limes constituant le pourtour, avec leur sens de parcours
    private PointSphere        centreCercleLimite ;     // Centre du cercle limite contenant le tracé du fleuve
    private double             cosCercleLimite ;        // Cosinus du cercle limite contenant le tracé du fleuve     
    
    
    
    /**
     * Constructeur.
     * @param id                 L'identifiant du pourtour .
     * @param limesOrdonnes      Les limes constituant le pourtour .
     * @param centreCercleLimite Le centre du cercle contenant le pourtour.
     * @param cosCercleLimite    Le cosinus permettant de déterminer le cercle contenant le pourtour.
     */
    public Pourtour (int id, List<LimesOrdonne> limesOrdonnes, PointSphere centreCercleLimite, double cosCercleLimite)
    {
        this.id                 = id ;
        this.limesOrdonnes      = limesOrdonnes ;
        this.centreCercleLimite = centreCercleLimite ;
        this.cosCercleLimite    = cosCercleLimite ;
    }

    
    
    /**
     * Renvoie la liste des points du pourtour, dans l'ordre.
     * @return La liste des points du pourtour.
     */
    public List<PointSphere> points ()
    {
        int               nbPoints ;    // Nombre de points du pourtour
        List<PointSphere> points ;      // Points du pourtour
        
        
        // Compter les points
        nbPoints = 0 ;
        for (LimesOrdonne limesOrdonne : this.limesOrdonnes)
            nbPoints += limesOrdonne.limes().points().size() ;
        
        // Rassembler les points du pourtour
        points = new ArrayList<> (nbPoints) ;
        for (LimesOrdonne limesOrdonne : this.limesOrdonnes)
        {
            List<PointSphere> ptsLimes = limesOrdonne.limes().points() ;
            if (limesOrdonne.sensNormal())
            {
                points.addAll (ptsLimes) ;
            }
            else
            {
                for (int i = ptsLimes.size() - 1 ; i >= 0 ; i--)
                    points.add (ptsLimes.get (i)) ;
            }
        }
        
        // Renvoyer le résultat
        return points ;
    }
    


    // Autres accesseurs
    public int                id                 () { return this.id ; }
    public List<LimesOrdonne> limesOrdonnes      () { return this.limesOrdonnes ; }
    public PointSphere        centreCercleLimite () { return this.centreCercleLimite ; }
    public double             cosCercleLimite    () { return this.cosCercleLimite ; }
    
}
