/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique;

import static fr.histoiremondiale.histoire.RessourcesAppli.ICONE_PLUME;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.util.Observable;
import java.util.Observer;

import javax.swing.JPanel;

import fr.histoiremondiale.histoire.EtatAppli;
import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.RessourcesAppli;
import fr.histoiremondiale.histoire.igraphique.tracercartes.GestTracerCartes;
import fr.histoiremondiale.histoire.igraphique.tracercartes.InfosTracerImage;
import fr.histoiremondiale.histoire.utiles.exceptions.Exceptions;
import fr.histoiremondiale.histoire.utiles.math.CoordSphere;
import fr.histoiremondiale.histoire.utiles.math.PointPlan;
import fr.histoiremondiale.histoire.utiles.math.PointSphere;



/**
 * Panneau pour l'affichage de la carte du monde.
 */
public class PanCarte extends JPanel implements Observer
{

    // Constantes
    // (publiques parce qu'elles servent en dehors du panneau, pour tracer d'autres cartes)
    public static final Color COUL_TERRES  = new Color (0xFFFFC0) ;     // Couleur des terres
    public static final Color COUL_MERS    = new Color (0xC0FFFF) ;     // Couleur des mers
    public static final Color COUL_FLEUVES = new Color (0x000040) ;     // Couleur des fleuves
    public static final int   ANGLE_CARTE  = 60 ;                       // Angle (en degrés) couvert par la carte
    public static final double COEFFT_CARTE = 2*Math.tan (Math.toRadians (ANGLE_CARTE/2));
    
    private GestTracerCartes gestTracerCartes ; // Gestionnaire des tracers de cartes sur le panneau
    
    
    
    /**
     * Constructeur.
     */
    public PanCarte ()
    {
        // Créer l'outil de gestion des tracers de cartes
        this.gestTracerCartes = GestTracerCartes.creerInstance (this) ;
        
        // S'inscrire comme observateur de l'état de l'application
        HistoireMondiale.instance().etat().addObserver              (this) ;
        HistoireMondiale.instance().donneesIGraphique().addObserver (this) ;
    }
    
    
    /**
     * Renvoie le traceur de cartes du panneau.
     * @return Le traceur de cartes du panneau.
     */
    public GestTracerCartes gestTracerCartes ()
    {
        return this.gestTracerCartes ;
    }
    
    
    
    /**
     * Affichage du panneau.
     */
    protected void paintComponent (Graphics g)
    {
        InfosTracerImage infosTracerImage = this.gestTracerCartes.infosImageActuelle() ;
        BufferedImage    img              = infosTracerImage.ancImg.image ;
        if (img == null)
        {
            recreerCarte() ;
        }
        else
        {
            // Tracer la carte centrée sur la zone de dessin
            PointPlan ptPanneau = posImgCarteVersPosPanneau (0, 0) ;
            g.drawImage (img, (int)ptPanneau.x(), (int)ptPanneau.y(), this) ;
            
            // Si une autre carte est en cours de tracé, ajouter le symbole sur la carte
            // Si l'image actuelle est périmée, ajouter le symbole sur la carte
            if (HistoireMondiale.instance().etat().symboleAttenteTracer())
            {
                if (infosTracerImage.perimee())
                {
                    // int tailleSymboleModif = 20 ;
                    // g.setColor (Color.RED) ;
                    //  g.fillOval (0, 0, tailleSymboleModif, tailleSymboleModif) ;
                    g.drawImage (RessourcesAppli.imageIcone (ICONE_PLUME), 0, 0, null) ;
                }
            }
            
            // Si une erreur est survenue en traçant la carte suivante, afficher le message d'erreur
            Throwable erreur = infosTracerImage.ancImg.erreur ;
            if (erreur != null)
            {
                int cptLigne = 0 ;
                g.drawString ("Erreur lors de la génération de la carte",  1, ++cptLigne * 20) ;
                g.drawString (Exceptions.extraireMessagesErreur (erreur),  1, ++cptLigne * 20) ;
            }
        }
    }
    
    
    /**
     * Calcule une position sur l'image de la carte en fonction de sa position sur le panneau (sur sa zone de
     *   dessin en réalité) (l'image n'est pas forcément alignée sur le coin haut-gauche du panneau).
     * @param posXPanneau Abscisse de la position sur le panneau.
     * @param posYPanneau Ordonnée de la position sur le panneau.
     * @return La position correspondante sur l'image.
     */
    public PointPlan posPanneauVersPosImgCarte (int posXPanneau, int posYPanneau)
    {
        // Récupérer les informations sur l'image et le panneau
        InfosTracerImage infosTracerImage = this.gestTracerCartes.infosImageActuelle() ;
        BufferedImage    img              = infosTracerImage.ancImg.image ;
        int              largeurImg       = img.getWidth() ;
        int              hauteurImg       = img.getHeight() ;
        
        // Calculer les coordonnées dans l'image
        int posXImgCarte = posXPanneau + (largeurImg - this.getWidth())  / 2 ;
        int posYImgCarte = posYPanneau + (hauteurImg - this.getHeight()) / 2 ;
        return new PointPlan (posXImgCarte, posYImgCarte) ;
    }
    
    
    /**
     * Calcule une position sur le panneau en fonction de sa position sur l'image de la carte.
     * @param posXImgCarte Abscisse de la position sur la carte.
     * @param posYImgCarte Ordonnée de la position sur la carte.
     * @return La position correspondante sur la carte.
     */
    public PointPlan posImgCarteVersPosPanneau (int posXImgCarte, int posYImgCarte)
    {
        // Récupérer les informations sur l'image et le panneau
        InfosTracerImage infosTracerImage = this.gestTracerCartes.infosImageActuelle() ;
        BufferedImage    img              = infosTracerImage.ancImg.image ;
        int              largeurImg       = img.getWidth() ;
        int              hauteurImg       = img.getHeight() ;
        
        // Calculer les coordonnées dans l'image
        int posXPanneau = posXImgCarte - (largeurImg - this.getWidth())  / 2 ;
        int posYPanneau = posYImgCarte - (hauteurImg - this.getHeight()) / 2 ;
        return new PointPlan (posXPanneau, posYPanneau) ;
    }
    
    
    /**
     * Calcule une position sur la sphère en fonction de la position sur le panneau.
     * @param posXPanneau Abscisse de la position sur le panneau.
     * @param posYPanneau Ordonnée de la position sur le panneau.
     * @return Le point de la phère correspondant (le point projeté à cet endroit sur le panneau).
     */
    public PointSphere posPanneauVersPosSphere (int posXPanneau, int posYPanneau)
    {
        // Récupérer les informations sur l'image
        InfosTracerImage infosTracerImage = this.gestTracerCartes.infosImageActuelle() ;
        BufferedImage    img              = infosTracerImage.ancImg.image ;
        int              largeurPlan      = img.getWidth() ;
        int              hauteurPlan      = img.getHeight() ;
        PointSphere      ptCentralCarte   = infosTracerImage.ancImg.demande.ptCentrePlan ;
        double           loupe            = infosTracerImage.ancImg.demande.loupe ;
        
        // Calculer le point dans la sphère
        PointPlan ptImgCarte = posPanneauVersPosImgCarte (posXPanneau, posYPanneau) ;
        return CoordSphere.coordPlanVersSphere (ptImgCarte, ptCentralCarte,
                                                CoordSphere.matChgtPlanVersSphere (ptCentralCarte),
                                                largeurPlan, hauteurPlan,
                                                loupe) ;
    }
    
    
    /**
     * Renvoie la carte actuellement affichée.
     * @return l'image actuellement affichée.
     */
    public BufferedImage imageCarteActuelle ()
    {
        return this.gestTracerCartes.infosImageActuelle().ancImg.image ;
    }
    
    
    /**
     * Demande au traceur de redessiner la carte à afficher. La méthode se termine une fois la demande
     *   envoyée, alors que le tracé n'est pas encore prêt.<br>
     * Le traceur modifiera l'image affichée par le panneau quand le tracé sera prêt.<br>
     * (méthode publique pour être appelée depuis les objets d'action par exemple)
     */
    public void recreerCarte ()
    {
        // Initialisations
        HistoireMondiale     application = HistoireMondiale.instance() ;
        EtatAppli            etatAppli   = application.etat() ;
        
        // Demander la création de la carte
        this.gestTracerCartes.demanderNouvelleCarte (etatAppli.annee(),
                                                     etatAppli.fleuvesAffiches(),
                                                     etatAppli.meridiensParallelesAffiches(),
                                                     etatAppli.ptCentralCarte(),
                                                     this.getWidth(), this.getHeight(), etatAppli.loupe(),
                                                     COUL_MERS, COUL_TERRES, COUL_FLEUVES) ;
    }


    /**
     * Méthode de traitement des modifications de l'état de l'appli.
     */
    public void update (Observable o, Object params)
    {
        // Redessiner la carte quand l'état de l'appli change
        // (à la méthode de gestion des demandes de tracé de ne pas recréer l'image si elle convient déjà)
        recreerCarte() ;
    }

}
