/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique.actions;

import static fr.histoiremondiale.histoire.utiles.es.fichiers.Fichiers.urlLocaleDeNomFichier;
import static fr.histoiremondiale.histoire.utiles.exttypes.Flottants.formater;
import static fr.histoiremondiale.histoire.utiles.igraphique.dessin.Images.copie;

import java.awt.Image;
import java.awt.Point;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

import javax.imageio.ImageIO;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;

import fr.histoiremondiale.histoire.DonneesIGraphique;
import fr.histoiremondiale.histoire.EtatAppli;
import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.donnees.Civilisation;
import fr.histoiremondiale.histoire.donnees.Territoire;
import fr.histoiremondiale.histoire.igraphique.FenPageHTML;
import fr.histoiremondiale.histoire.igraphique.FenPrincipale;
import fr.histoiremondiale.histoire.igraphique.PanCarte;
import fr.histoiremondiale.histoire.igraphique.composants.FiltreSelectionFichiersPNG;
import fr.histoiremondiale.histoire.igraphique.composants.FiltreSelectionRepertoire;
import fr.histoiremondiale.histoire.igraphique.donnees.InfosParagrapheNavig;
import fr.histoiremondiale.histoire.igraphique.tracercartes.DemandeTracerCarte;
import fr.histoiremondiale.histoire.igraphique.tracercartes.InfosTracerImage;
import fr.histoiremondiale.histoire.stockage.ExporteurHTML;
import fr.histoiremondiale.histoire.stockage.ExporteurPNG;
import fr.histoiremondiale.histoire.utiles.es.fichiers.Fichiers;
import fr.histoiremondiale.histoire.utiles.exceptions.Exceptions;
import fr.histoiremondiale.histoire.utiles.exttypes.Chaines;
import fr.histoiremondiale.histoire.utiles.math.CalculsPlan;
import fr.histoiremondiale.histoire.utiles.math.CoordSphere;
import fr.histoiremondiale.histoire.utiles.math.Matrice;
import fr.histoiremondiale.histoire.utiles.math.PointPlan;
import fr.histoiremondiale.histoire.utiles.math.PointSphere;



/**
 * Actions sur la carte et son panneau.
 */
public class ActionsCarte
{

    private static Point       pointDepartGlissementSouris ;    // Point de départ du glissement de la souris dans le plan
    private static PointSphere pointDepartCentreTexte ;         // Point de départ du centre du texte déplacé (en coordonnées sphériques)
    private static Territoire  territoireTexteEnDeplacement ;   // Territoire dont le texte est déplacé
    
    private static JFileChooser selectFicSauvCarte ;            // Sélectionneur de répertoire pour la sauvegarde des cartes
    private static JFileChooser selectFicSauvCarteEtParagraphe ;// Sélectionneur de répertoire pour la sauvegarde des cartes + paragraphe
    private static JFileChooser selectFicParagrapheSauvegarde ;     // Sélectionneur de fichier pour le chargement de cartes/paragraphes sauvegadés

    
    
    /**
     * Commence le déplacement d'un texte de territoire sur la carte.
     * @param x Abscisse du point de départ du déplacement.
     * @param y Ordonnée du point de départ du déplacement.
     */
    public static void initierDeplacementTexteTerritoireEn (int x, int y)
    {
        HistoireMondiale application = HistoireMondiale.instance() ;
        
        
        // Récupérer le territoire dont le texte est déplacé
        Territoire territoire = texteEn (x, y) ;
        
        // Initialiser les attributs de déplacement si besoin
        if (territoire == null)
        {
            ActionsCarte.pointDepartGlissementSouris  = null ;
            ActionsCarte.pointDepartCentreTexte       = null ;
            ActionsCarte.territoireTexteEnDeplacement = null ;
        }
        else
        {
            DonneesIGraphique donneesIGraphique = application.donneesIGraphique() ;
            ActionsCarte.pointDepartGlissementSouris  = new Point (x, y) ;
            ActionsCarte.pointDepartCentreTexte       = donneesIGraphique.centreTexteTerritoire (territoire) ;
            ActionsCarte.territoireTexteEnDeplacement = territoire ;
        }
    }
    
    
    /**
     * Déplace le texte du territoire en cours de déplacement à la position données sur le panneau de la
     *   carte.
     * @param x Abscisse du point d'arrivée du déplacement.
     * @param y Ordonnée du point d'arrivée du déplacement.
     */
    public static void deplacerTexteTerritoire (int x, int y)
    {
        HistoireMondiale   application       = HistoireMondiale.instance() ;
        DonneesIGraphique  donneesIGraphique = application.donneesIGraphique() ;
        PanCarte           p_carte           = application.fenPrincipale().p_carte ;
        InfosTracerImage   infosTracerImage  = p_carte.gestTracerCartes().infosImageActuelle() ;
        DemandeTracerCarte demandeTracer     = infosTracerImage.ancImg.demande ;
        
        
        // Si le déplacement s'effectue bien à l'intérieur du panneau
        if (x >= 0 && x < p_carte.getWidth() && y >= 0 && y < p_carte.getHeight())
        {
            if (pointDepartGlissementSouris != null)
            {
                // Récupérer les informations sur l'image tracée
                PointSphere centre = demandeTracer.ptCentrePlan ;
                double     loupe   = demandeTracer.loupe ;
                
                int largeurCarte   = demandeTracer.largeurCarte ;
                int hauteurCarte   = demandeTracer.hauteurCarte ;
                
                // Calculer le décalage par rapport à la position d'origine
                int decalageX = x - (int) pointDepartGlissementSouris.getX() ;
                int decalageY = y - (int) pointDepartGlissementSouris.getY() ;
                
                int a = Math.max (largeurCarte, hauteurCarte);
                Matrice matrice = CoordSphere.matChgtSphereVersPlan2 (centre) ;
                
                PointPlan   ancCentreTextePlan = CalculsPlan.changementOrigineCentreVersCoinHautGauche (CoordSphere.projection (pointDepartCentreTexte, matrice, loupe, a), largeurCarte, hauteurCarte);
                PointPlan Pt = new PointPlan(ancCentreTextePlan.x() + decalageX, 
                                             ancCentreTextePlan.y() + decalageY);
                PointSphere nouvCentreTexteSphere = CoordSphere.InvProjection (Pt, centre, largeurCarte, 
                                                                               hauteurCarte, loupe) ;
                
                donneesIGraphique.modifCentreTexteTerritoire (territoireTexteEnDeplacement, nouvCentreTexteSphere) ;
            }
        }
    }
    
    
    /**
     * Centre la carte sur le point du panneau dont les coordonnées sont passées en paramètre.
     * @param x L'abscisse du point sur lequel centrer la carte.
     * @param y L'ordonnée du point sur lequel centrer la carte.
     */
    public static void centrerCarteSur (int x, int y)
    {
        HistoireMondiale application = HistoireMondiale.instance() ;
        PointSphere centre           = application.etat ().ptCentralCarte ();
        double loupe                 = application.etat().loupe ();
        
        int largeurCarte          = application.fenPrincipale().p_carte.getWidth ();
        int hauteurCarte          = application.fenPrincipale().p_carte.getHeight ();
        
        // Calculer le nouveau centre
        // TODO La deuxième ligne ne sert manifestement à rien. Elle a servi un jour quand on créait un
        //      nouveau point avec des longitude/latitude arrondie parce qu'on ne manipulait que des
        //      coordonnées entières. A supprimer.
        PointSphere ptSphere = CoordSphere.InvProjection (new PointPlan(x,y), centre,
                                                          largeurCarte, hauteurCarte, 
                                                          loupe);
//        ptSphere = new PointSphere (ptSphere.longitude(), ptSphere.latitude()) ;
        
        // Ajuster les coordonnées
        ActionsEspaceTemps.centrerCarteSurPosition (ptSphere) ;
    }
    
    /**
     * Centre la carte sur le point du panneau dont les coordonnées sont passées en paramètre s'il n'y a pas
     *   de texte à l'endroit indiqué.
     * @param x L'abscisse du point sur lequel centrer la carte.
     * @param y L'ordonnée du point sur lequel centrer la carte.
     */
    public static void centrerCarteSiNonTexteSur (int x, int y)
    {
        if (texteEn (x, y) == null)
            centrerCarteSur (x, y) ;
    }
    
    /**
     * Centre la carte sur le point du panneau dont les coordonnées sont passées en paramètre si on a cliqué
     *   sur un texte à l'origine (s'il y avait possibilité de déplacement de texte, pour être rigoureux).
     * @param x L'abscisse du point sur lequel centrer la carte.
     * @param y L'ordonnée du point sur lequel centrer la carte.
     */
    public static void centrerCarteSiOrigineTexteSur (int x, int y)
    {
        if (ActionsCarte.pointDepartGlissementSouris != null)
            centrerCarteSur (x, y) ;
    }
    
    /**
     * Centre la carte sur le centre de la civilisation pour l'année indiquée.
     * @param civilisation La civilisation concernée.
     * @param annee        L'année donnée.
     */
    public static void centrerCarteSurCivilisation (Civilisation civilisation, int annee)
    {
        HistoireMondiale.instance().etat().modifPtCentralCarte (civilisation.ptCentral (annee)) ;
    }
    
    
    /**
     * Affiche le paragraphe correspondant au territoire se trouvant aux coordonnées données sur l'image de
     *   la carte.
     * @param x Abscisse.
     * @param y Ordonnée.
     */
    public static void affParagrapheTerritoireEn (int x, int y)
    {
        HistoireMondiale application = HistoireMondiale.instance() ;
        EtatAppli        etatAppli   = application.etat() ;
        
        // Afficher le Paragraphe correspondant
        Territoire territoireClique = territoireOuTexteEn (x, y) ;
        if (territoireClique != null)
        {
            Civilisation civilisation = territoireClique.civilisation() ;
            int annee = etatAppli.annee() ;
            if (civilisation != null)
            {
                // Afficher le Paragraphe correspondant à l'année courante et au territoire cliqué
                InfosParagrapheNavig paragraphe = application.donneesIGraphique().paragrapheParCriteres (civilisation, annee) ;
                ActionsNavigateur.afficherParagraphe (paragraphe) ;

                // Sélectionner cette civilisation
                etatAppli.modifCivilisationSelectionnee (civilisation);
            }
        }
    }
    
    
    /**
     * Affiche la boîte de dialogue de sélection de la limite de visibilité du territoire aux coordonnées en
     *   paramètre sur l'image de la carte.
     * @param x Abscisse.
     * @param y Ordonnée.
     */
    public static void affDialVisibiliteTerritoireEn (int x, int y)
    {
        Territoire territoireClique = territoireOuTexteEn (x, y) ;
        if (territoireClique != null)
            ActionsFenetres.affDialVisibiliteTerritoire (territoireClique) ;
    }
    
    
    /**
     * Réinitialise le territoire actuellement survolé par la souris (souris en (x,y)).<br>
     * Aucun territoire n'est plus en cours de survol.
     */
    public static void reinitialiserTerritoireSurvole ()
    {
        HistoireMondiale.instance().etat().modifTerritoireSurvole (null) ;
    }

    
    /**
     * Note le territoire actuellement survolé par la souris (souris en (x,y)).
     * @param x Abscisse.
     * @param y Ordonnée.
     */
    public static void definirTerritoireSurvole (int x, int y)
    {
        HistoireMondiale application = HistoireMondiale.instance() ;
        PanCarte         carte       = application.fenPrincipale().p_carte ;
        InfosTracerImage infosImage  = carte.gestTracerCartes().infosImageActuelle() ;

        
        // Mettre à jour le territoire survolé si une carte est affichée
        if (infosImage.imageDisponible())
            application.etat().modifTerritoireSurvole (territoireEn (x, y)) ;
    }

    
    /**
     * Réinitialise les infobulles de la carte.
     */
    public static void reinitialiserInfobullesPanneau ()
    {
        HistoireMondiale.instance().fenPrincipale().p_carte.setToolTipText (null) ;
    }
    
    /**
     * Modifie les infobulles de la carte en fonction de la position de la souris sur le panneau.
     * @param x Abscisse de la position de la souris.
     * @param y Ordonnée de la position de la souris.
     */
    public static void ajusterInfobullesPanneau (int x, int y)
    {
        HistoireMondiale application = HistoireMondiale.instance() ;
        EtatAppli        etatAppli   = application.etat() ;
        PanCarte         p_carte     = application.fenPrincipale().p_carte ;
        InfosTracerImage infosImage  = p_carte.gestTracerCartes().infosImageActuelle() ;
        
        
        // Mettre à jour les infobulles sur la carte si une image est affichée
        if (infosImage.imageDisponible())
        {
            Territoire territoire      = territoireEn  (x, y) ;
            Territoire texteTerritoire = texteEn (x, y) ;
            
            // Modifier les infobulles
            if      (! etatAppli.infobullesCarte()) reinitialiserInfobullesPanneau() ;
            else if (texteTerritoire != null)       p_carte.setToolTipText ("Clic gauche (glisser) pour déplacer le texte,\n clic droit pour afficher le Paragraphe, maj + clic droit pour régler le niveau de visibilité du texte.") ;
            else if (territoire      != null)       p_carte.setToolTipText ("Clic gauche pour centrer la carte sur ce point, clic droit pour afficher le Paragraphe, maj + clic droit pour régler le niveau de visibilité du texte.") ;
            else                                    p_carte.setToolTipText ("Clic gauche pour centrer la carte sur ce point.") ;
        }
    }
    
    
    /**
     * Modifie la taille disponible pour tracer la carte.<br>
     * (les valeurs n'étant pas stockées, cette méthode ne prend plus la largeur et la hauteur en paramètre)
     */
    public static void modifierTailleDispoPourCarte ()
    {
        HistoireMondiale.instance().etat().modifTailleZoneCarte() ;
    }
    
    
    
    /**
     * Sauvegarde la carte affichée.
     */
    public static void sauvegarderCarte ()
    {
        HistoireMondiale     application ;              // Instance de l'application
        EtatAppli            etatAppli ;                // Etat de l'application
        PanCarte             p_carte ;                  // Panneau d'affichage de la carte
        BufferedImage        imgCarte ;                 // Carte à exporter
        File                 ficChoisi ;                // Répertoire dans lequel exporter les données
        

        // Initialisations
        application = HistoireMondiale.instance() ;
        etatAppli   = application.etat() ;
        p_carte     = application.fenPrincipale().p_carte ;

        
        // Récupérer l'image courante
        imgCarte = p_carte.imageCarteActuelle() ;
        if (imgCarte == null)
        {
            JOptionPane.showMessageDialog (p_carte, "Pas de carte à sauvegarder : aucune carte affichée", "Erreur de sauvegarde de la carte", JOptionPane.ERROR_MESSAGE) ;
            return ;
        }
        
        // Récupérer le sélectionneur de répertoires
        if (selectFicSauvCarte == null)
        {
            selectFicSauvCarte = new JFileChooser() ;
            selectFicSauvCarte.setFileFilter              (new FiltreSelectionFichiersPNG()) ;
            selectFicSauvCarte.setAcceptAllFileFilterUsed (false) ;
            selectFicSauvCarte.setDialogTitle ("Choix d'un nom de fichier") ;
        }
        
        // Demander à l'utilisateur de choisir le fichier
        do
        {
            // Proposer un nom de fichier
            String chemRep = (etatAppli.chemRepSauvCartes() == null ?
                                 System.getProperty ("user.home") :
                                 etatAppli.chemRepSauvCartes()) ;
            String nomFic  = (etatAppli.civilisationSelectionnee() == null ?
                                 "histoiremondiale " :
                                 etatAppli.civilisationSelectionnee().nom () + " ") +
                             "(" +
                             formater (etatAppli.ptCentralCarte().longitude()) + ";" +
                             formater (etatAppli.ptCentralCarte().latitude())  + ")" +
                             " " +
                             formater (etatAppli.loupe()) + "x" +
                             " en " +
                             etatAppli.annee() +
                             ".png" ;
            selectFicSauvCarte.setSelectedFile (new File (chemRep + File.separator + nomFic)) ;
            
            // Laisser l'utilisateur choisir
            int resChoixFic = selectFicSauvCarte.showSaveDialog (p_carte) ;       
            if (resChoixFic != JFileChooser.APPROVE_OPTION)
                return ;

            ficChoisi = selectFicSauvCarte.getSelectedFile() ;
            
            // Arranger le nom (extension)
            if (! Fichiers.extensionEgale (ficChoisi.getName(), "png"))
                ficChoisi = new File (ficChoisi.getAbsolutePath() + ".png") ;
        
            // Demander confirmation si le fichier existe déjà
            if (ficChoisi.isDirectory())
            {
                JOptionPane.showMessageDialog (p_carte,
                                               "Vous devez choisir un fichier et non un répertoire.",
                                               "Erreur de sauvegarde de la carte",
                                               JOptionPane.ERROR_MESSAGE) ;    
                ficChoisi = null ;
            }
            else if (ficChoisi.exists() &&
                     JOptionPane.showConfirmDialog (p_carte,
                                                   "Le fichier existe déjà. Voulez-vous l'écraser ?",
                                                   "Confirmation de remplacement du fichier",
                                                   JOptionPane.WARNING_MESSAGE)
                     != JOptionPane.YES_OPTION)
            {
                ficChoisi = null ;
            }

        } while (ficChoisi == null) ;

        
        // Exporter les données
        try
        {
            new ExporteurPNG().exporter (copie (imgCarte, p_carte), ficChoisi) ;
        }
        catch (Throwable e)
        {
            e.printStackTrace() ;
            JOptionPane.showMessageDialog (p_carte,
                                           "Erreur lors de la sauvegarde de la carte :\n" +
                                             Exceptions.extraireMessagesErreur (e),
                                           "Erreur de sauvegarde",
                                           JOptionPane.ERROR_MESSAGE) ;
        }
        
        // Mémoriser le chemin pour la prochaine fois
        etatAppli.modifChemRepSauvCartes (ficChoisi.getParent()) ;
    }
    
    
    /**
     * Sauvegarde la carte affichée, avec le paragraphe correspondant à la date courante et la civilisation
     *   sélectionnée.
     */
    public static void sauvegarderCarteEtParagraphe ()
    {
        HistoireMondiale     application ;              // Instance de l'application
        EtatAppli            etatAppli ;                // Etat de l'application
        PanCarte             p_carte ;                  // Panneau d'affichage de la carte
        BufferedImage        imgCarte ;                 // Carte à exporter
        Civilisation         civilisationSelectionnee ; // Civilisation sélectionnée
        int                  anneeCourante ;            // Année courante sur la carte
        PointSphere          ptCentre ;                 // Coordonnées du point au centre de la carte
        InfosParagrapheNavig paragraphe ;               // Paragraphe à exporter
        File                 repChoisi ;                // Répertoire dans lequel exporter les données
        

        // Initialisations
        application = HistoireMondiale.instance() ;
        etatAppli   = application.etat() ;
        p_carte     = application.fenPrincipale().p_carte ;

        
        // Récupérer le Paragraphe courant
        civilisationSelectionnee = etatAppli.civilisationSelectionnee() ;
        anneeCourante  = etatAppli.annee() ;
        ptCentre       = etatAppli.ptCentralCarte() ;
        if (civilisationSelectionnee == null)
        {
            JOptionPane.showMessageDialog (p_carte, "Aucune civilisation sélectionnée", "Erreur de sauvegarde de la carte", JOptionPane.ERROR_MESSAGE) ;
            return ;
        }
        
        // Récupérer le Paragraphe correspondant aux critères
        // (s'il n'y a pas de Paragraphe tant pis, on peut toujours exporter la carte)
        paragraphe = application.donneesIGraphique().paragrapheParCriteres (civilisationSelectionnee, anneeCourante) ;

        // Récupérer l'image courante
        imgCarte = p_carte.imageCarteActuelle() ;
        if (imgCarte == null)
        {
            JOptionPane.showMessageDialog (p_carte, "Pas de carte à sauvegarder : aucune carte affichée", "Erreur de sauvegarde de la carte", JOptionPane.ERROR_MESSAGE) ;
            return ;
        }
        
        // Récupérer le sélectionneur de répertoires
        if (selectFicSauvCarteEtParagraphe == null)
        {
            selectFicSauvCarteEtParagraphe = new JFileChooser() ;
            selectFicSauvCarteEtParagraphe.setFileSelectionMode       (JFileChooser.DIRECTORIES_ONLY) ;
            selectFicSauvCarteEtParagraphe.setFileFilter              (new FiltreSelectionRepertoire()) ;
            selectFicSauvCarteEtParagraphe.setAcceptAllFileFilterUsed (false) ;
            selectFicSauvCarteEtParagraphe.setDialogTitle ("Choix d'un répertoire pour générer la page") ;
        }
        
        // Demander à l'utilisateur de choisir le répertoire
        do
        {
            // Proposer un répertoire
            String chemRep = (etatAppli.chemRepSauvParagraphes() == null ?
                                 System.getProperty ("user.home") :
                                 etatAppli.chemRepSauvParagraphes()) ;
            selectFicSauvCarteEtParagraphe.setCurrentDirectory (new File (chemRep + File.separator)) ;

            // Laisser l'utilisateur choisir
            int resChoixFic = selectFicSauvCarteEtParagraphe.showSaveDialog (p_carte) ;       
            if (resChoixFic != JFileChooser.APPROVE_OPTION)
                return ;
            
            // Vérifier le répertoire choisi
            repChoisi = selectFicSauvCarteEtParagraphe.getSelectedFile() ;
            // (ne pas créer des répertoires comme ça, pour éviter les conneries : demander à l'utilisateur
            //  d'en choisir/créer un autre)
            //    => Si, laissons faire
            // if (!repChoisi.exists())
            // {
            //     JOptionPane.showMessageDialog (p_carte,
            //                                    "Ce répertoire n'existe pas.\nChoisissez un répertoire dans lequel générer les fichiers.",
            //                                    "Erreur de sauvegarde",
            //                                    JOptionPane.ERROR_MESSAGE) ;
            //     repChoisi = null ;
            // }
            // else
            if ((repChoisi.exists()) && (!repChoisi.isDirectory()))
            {
                JOptionPane.showMessageDialog (p_carte,
                                               "Ce n'est pas un répertoire.\nChoisissez un répertoire dans lequel générer les fichiers.",
                                               "Erreur de sauvegarde de la carte",
                                               JOptionPane.ERROR_MESSAGE) ;    
                repChoisi = null ;
            }
//            else
//            {
//                // Si tout s'est bien passé, placer le sélecteur dans le bon état pour la prochaine fois
//                // (si on ne précise pas le répertoire courant, soit on était dedans quand on l'a choisi
//                //  et on se retrouve dehors sans qu'il soit sélectionné, soit on l'a sélectionné sans
//                //  entrer dedans et ça fonctionne bien [on se retrouve dehors avec le répertoire sélectionné])
//                //      => Là on se retrouve toujours dedans
//                selectFicSauvCarteEtParagraphe.setCurrentDirectory (repChoisi) ;
//            }

        } while (repChoisi == null) ;

        
        // Exporter les données
        String chemFicHtml = null ;
        try
        {
            chemFicHtml = new ExporteurHTML().exporter (paragraphe, anneeCourante, ptCentre, copie (imgCarte, p_carte), repChoisi) ;
        }
        catch (Throwable e)
        {
            e.printStackTrace() ;
            JOptionPane.showMessageDialog (p_carte,
                                           "Erreur lors de la sauvegarde de la carte :\n" +
                                             Exceptions.extraireMessagesErreur (e),
                                           "Erreur de sauvegarde",
                                           JOptionPane.ERROR_MESSAGE) ;
        }
        
        // Mémoriser le chemin pour la prochaine fois
        etatAppli.modifChemRepSauvParagraphes (repChoisi.getAbsolutePath()) ;
        
        // Afficher la page générée
        if (etatAppli.affParagraphesHtmlGeneres() && chemFicHtml != null)
            afficherFenParagraphe (chemFicHtml, application.fenPrincipale()) ;
    }
    
    
    /**
     * Affiche une carte précédemment sauvegardée.
     */
    public static void visualiserParagraphe ()
    {
        HistoireMondiale application = HistoireMondiale.instance() ;
        EtatAppli        etatAppli   = application.etat() ;
        PanCarte         p_carte     = application.fenPrincipale().p_carte ;
        
        
        // Récupérer le sélectionneur de répertoires
        if (selectFicParagrapheSauvegarde == null)
        {
            selectFicParagrapheSauvegarde = new JFileChooser() ;
            selectFicParagrapheSauvegarde.setMultiSelectionEnabled (false) ;                  // Sélection d'un seul fichier
            selectFicParagrapheSauvegarde.setFileSelectionMode (JFileChooser.FILES_ONLY) ;    // Pas de répertoire
            FileFilter filtre = new FileNameExtensionFilter("Fichiers html", "html") ;    // Fichiers *.html
            selectFicParagrapheSauvegarde.addChoosableFileFilter (filtre) ;
        }
          
        // Proposer un répertoire
        String chemRep ;
        if      (etatAppli.chemRepVueParagraphes()  != null) chemRep = etatAppli.chemRepVueParagraphes() ;
        else if (etatAppli.chemRepSauvParagraphes() != null) chemRep = etatAppli.chemRepSauvParagraphes() ;
        else                                                 chemRep = System.getProperty ("user.home") ;
        selectFicParagrapheSauvegarde.setCurrentDirectory (new File (chemRep + File.separator)) ;
        
        // Afficher la page choisie
        if (selectFicParagrapheSauvegarde.showOpenDialog (p_carte) == JFileChooser.APPROVE_OPTION)
        {
            afficherFenParagraphe (selectFicParagrapheSauvegarde.getSelectedFile().getAbsolutePath(),application.fenPrincipale()) ;
            // Mémoriser le chemin pour la prochaine fois
            etatAppli.modifChemRepVueParagraphes (selectFicParagrapheSauvegarde.getSelectedFile().getParent()) ;
        }
    }
    
    
    /**
     * Affiche une page html dans une fenêtre.
     * @param chemFicHtml Le chemin du fichier html.
     * @param parent la fenêtre parent
     */
    private static void afficherFenParagraphe (String chemFicHtml, FenPrincipale parent)
    {
        try
        {
        	Image img = ImageIO.read(new File(determinerChemImage(chemFicHtml)));
        	//on surcharge le constructeur de FenPageHtml pour utiliser les dimensions intérieures de celle-ci
        	//un booléen interieur fera l'affaire
        	FenPageHTML fenHtml = new FenPageHTML ("Carte", urlLocaleDeNomFichier (chemFicHtml), img.getWidth(null), img.getHeight(null), parent) ;
        	
//        	/*
//        	 * Je ne trouve pas très propre ces divers appels à getInsets.
//        	 * Pour la fenêtre parent encore ça va mais créer un ScrollBar et un JEditorPane
//        	 * pour connaître leur largeur et marges par défaut revient à savoir que notre fenêtre
//        	 * contiendra un JEditorPane et donc un JScrollPane.
//        	 * Ne vaudrait-i pas mieux passer la taille souhaitée pour le contenu et faire calculer
//        	 * la taille réelle par FenPageHTML ?
//        	 */
//        	
//        	Insets marges = new JEditorPane().getInsets(); 
//        	// +1 pour être sûr qu'il n'y aura pas un scrollbar en bas de la fenêtre
//        	int margesEditorPane = marges.left + marges.right + 1;
//        	int epaisseurScrollBar = new JScrollBar(JScrollBar.VERTICAL).getPreferredSize().width;
//        	int margesHorizontales = parent.getInsets().left + parent.getInsets().right;
//        	int l = Math.min(img.getWidth(null) + margesHorizontales + epaisseurScrollBar + margesEditorPane,
//        					 parent.getWidth() - margesHorizontales);
//        	int margesVerticales = parent.getInsets().top + parent.getInsets().bottom + parent.getJMenuBar().getHeight();
//        	int h = parent.getHeight()- margesVerticales;
//            FenPageHTML fenHtml = new FenPageHTML ("Carte", urlLocaleDeNomFichier (chemFicHtml), l, h) ;
            
            fenHtml.setVisible (true) ;
            //fenHtml.setExtendedState (JFrame.MAXIMIZED_BOTH) ;
        }
        catch (Throwable e)
        {
            e.printStackTrace() ;
            JOptionPane.showMessageDialog (null, "Erreur lors de l'affichage de la carte.", 
                                           "Erreur d'affichage de la page", JOptionPane.ERROR_MESSAGE) ;
        }
    }
    
    
    
// *****************************************************************************
//                          Méthodes internes
// *****************************************************************************
    
    
    private static String determinerChemImage(String fichier)
	{
		
    	//le contenu du fichier sous forme de texte
    	String S = "";
    	try
		{
			S = Fichiers.chargerContenuTexte(new File(fichier));
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		
		//retrouver la partie correspondant à <img src=" ... jusqu'à .png"
		S = Chaines.entreMarqueurs(S, "<img src=\"", "\"");
		
		
		String image = S.replace("/", File.separator);
		//maintenant S contient le nom complet de l'image, le chemin étant normalement en forme relative
		//si on commence par remonter d'un répertoire
		if (image.startsWith(".."))
		{
			while (image.startsWith(".."))
			{
				image = image.substring(3);
				int n = fichier.lastIndexOf(File.separator);
				fichier = fichier.substring(0,n+1);
			}
			image = fichier + image;
		}
		//si on utilise le répertoire courant
		else if (S.startsWith("."))
		{
			int n = fichier.lastIndexOf(File.separator);
			fichier = fichier.substring(0,n+1);
			image = fichier + image.replace("." + File.separator, "");
		}
		//pas de répertoire indiqué (équivalent à ".")
		else if (! image.contains(File.separator))
		{
			int n = fichier.lastIndexOf(File.separator);
			fichier = fichier.substring(0,n+1);
			image = fichier + image;
		}
		
		return image;
	}


	// Renvoie le territoire correspondant au texte de territoire/roi se trouvant aux coordonnées en paramètre
    //   sur le panneau.
    // Si aucune image n'est tracée, null est renvoyé.
    // Si on a cliqué sur un texte de territoire/roi, le territoire correspondant est renvoyé.
    // Sinon null est renvoyé.
    private static Territoire texteEn (int x, int y)
    {
        PanCarte         p_carte    = HistoireMondiale.instance().fenPrincipale().p_carte ;
        InfosTracerImage infosImage = p_carte.gestTracerCartes().infosImageActuelle();

        // Vérifier la présence d'un texte de territoire aux coordonnées indiquées
        if (infosImage.imageDisponible())
        {
            PointPlan ptImage = p_carte.posPanneauVersPosImgCarte (x, y) ;
            return infosImage.territoireTexteAuxCoordonnees (ptImage.x(), ptImage.y()) ;
        }
        else
        {
            return null ;
        }
    }

    
    // Renvoie le territoire se trouvant aux coordonnées en paramètre sur le panneau.
    // Si aucune image n'est tracée, null est renvoyé.
    // Si on a cliqué sur un territoire, il est renvoyé.
    // Sinon null est renvoyé.
    private static Territoire territoireEn (int x, int y)
    {
        PanCarte         p_carte    = HistoireMondiale.instance().fenPrincipale().p_carte ;
        InfosTracerImage infosImage = p_carte.gestTracerCartes().infosImageActuelle();

        // Vérifier la présence d'un texte de territoire aux coordonnées indiquées
        if (infosImage.imageDisponible())
        {
            PointPlan ptImage = p_carte.posPanneauVersPosImgCarte (x, y) ;
            return infosImage.territoireAuxCoordonnees (ptImage.x(), ptImage.y()) ;
        }
        else
        {
            return null ;
        }
    }
    
    
    // Renvoie le territoire se trouvant (ou dont le texte se trouve) aux coordonnées en paramètre sur le panneau.
    // Si aucune image n'est tracée, null est renvoyé.
    // Si on a cliqué sur un territoire, il est renvoyé.
    // Si on a cliqué sur un texte de territoire/roi, le territoire correspondant est renvoyé.
    // Sinon null est renvoyé.
    private static Territoire territoireOuTexteEn (int x, int y)
    {
        // Vérifier la présence d'un territoire aux coordonnées indiquées
        Territoire territoireClique ;
        territoireClique = texteEn (x, y) ;       // Texte territoire/roi
        if (territoireClique == null)
            territoireClique = territoireEn (x, y) ;    // Territoire lui-même
        
        // Renvoyer le territoire
        return territoireClique ;
    }
    
}
