/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique.actions;


import fr.histoiremondiale.histoire.EtatAppli;
import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.donnees.Territoire;
import fr.histoiremondiale.histoire.igraphique.DialAnnee ;
import fr.histoiremondiale.histoire.igraphique.DialCentre;
import fr.histoiremondiale.histoire.igraphique.DialLoupe;
import fr.histoiremondiale.histoire.utiles.math.PointSphere;



/**
 * Actions sur la localisation du point central de la carte, l'année courante, ...
 */
public class ActionsEspaceTemps
{

    /**
     * Centre la carte sur le point désigné par les coordonnées en paramètre dans le panneau de dessin.
     * @param ptSphere Le point qui doit être affiché au centre de la carte.
     */
    public static void centrerCarteSurPosition (PointSphere ptSphere)
    {
        HistoireMondiale.instance().etat().modifPtCentralCarte (ptSphere) ;
    }
    
    
    /**
     * Modifie la longitude du point central de la carte.
     * @param nouvLongitude La nouvelle longitude.
     */
    public static void modifierLongitude (double nouvLongitude)
    {
        HistoireMondiale.instance().etat().modifLongitude (nouvLongitude) ;
    }
    
    
    /**
     * Modifie la latitude du point central de la carte.
     * @param nouvLatitude La nouvelle latitude.
     */
    public static void modifierLatitude (double nouvLatitude)
    {
        HistoireMondiale.instance().etat().modifLatitude (nouvLatitude) ;
    }

    
    /**
     * Modifie le niveau de grossissement de la loupe.
     * @param nouvLoupe Nouvelle valeur pour la loupe.
     */
    public static void modifierLoupe (double nouvLoupe)
    {
        HistoireMondiale.instance ().etat ().modifLoupe (nouvLoupe) ;        
    }
    
    /**
     * Augmente le niveau de grossissement.
     */
    public static void augmenterLoupe ()
    {
        HistoireMondiale.instance().etat().augmenterLoupe() ;
    }
    
    /**
     * Diminue le niveau de grossissement.
     */
    public static void diminuerLoupe ()
    {
        HistoireMondiale.instance().etat().diminuerLoupe() ;
    }
    
    
    /**
     * Affiche la boîte de dialogue de choix du niveau de grossissement de la loupe.
     */
    public static void affDialChoixLoupe ()
    {
        ActionsFenetres.affDialLoupe() ;
    }
    
    /**
     * Affiche la boîte de dialogue de choix de l'année.
     */
    public static void affDialChoixAnnee ()
    {
        ActionsFenetres.affDialChoixDate() ;
    }
    
    /**
     * Affiche la boîte de dialogue de choix du centre de la carte.
     */
    public static void affDialChoixCentre ()
    {
        ActionsFenetres.affDialCentre() ;
    }

    
    /**
     * Valide le changement de valeur en cours de la loupe (par la boîte de dialogue).
     */
    public static void validerChangementLoupe ()
    {
        ActionsFenetres.masquerDialLoupe() ;
    }
    
    
    /**
     * Annule le changement de valeur en cours de la loupe (par la boîte de dialogue).
     */
    public static void annulerChangementLoupe()
    {
        DialLoupe dialLoupe      = HistoireMondiale.instance().dialLoupe() ;
		double    valeurInitiale = dialLoupe.valeurInitiale() ;
		EtatAppli etatAppli      = HistoireMondiale.instance().etat() ;
    	etatAppli.modifLoupe (valeurInitiale) ;
    	ActionsFenetres.masquerDialLoupe () ;
    }


    /**
     * Modifie le niveau de grossissement du nombre de niveaux passé en paramètre.
     * @param decalageLoupe Le nombre de niveaux.
     */
    public static void decalerLoupe (double decalageLoupe)
    {
        HistoireMondiale.instance().etat().modifRelativeLoupe (decalageLoupe) ;
    }
    
    
    /**
     * Valide le changement de valeur en cours de la date (par la boîte de dialogue).
     */
    public static void validerChangementDate ()
    {
        DialAnnee dialAnnee    = HistoireMondiale.instance().dialAnnee() ;
        int       anneeChoisie = dialAnnee.annee() ;
        EtatAppli etatAppli    = HistoireMondiale.instance().etat() ;
        etatAppli.modifAnnee (anneeChoisie) ;
        ActionsFenetres.masquerDialChoixAnnee() ;
    }
    
    
    /**
     * Annule le changement de valeur en cours de la date (par la boîte de dialogue).
     */
    public static void annulerChangementDate()
    {
        ActionsFenetres.masquerDialChoixAnnee() ;
    }


    
    
    /**
     * Modifie l'année courante.
     * @param nouvAnnee La nouvelle année.
     */
    public static void modifierAnnee (int nouvAnnee)
    {
        HistoireMondiale.instance().etat().modifAnnee (nouvAnnee)  ;
    }
    
    /**
     * Diminue l'année courante du pas de déplacement des années.
     */
    public static void decrementerAnnee ()
    {
        HistoireMondiale.instance().etat().decrementerAnnee()  ;
    }
    
    /**
     * Augmente l'année courante du pas de déplacement des années.
     */
    public static void incrementerAnnee ()
    {
        HistoireMondiale.instance().etat().incrementerAnnee()  ;
    }
    
    
    /**
     * Modifie la valeur limite de grossissement auquel le texte du territoire est affiché.
     * @param territoire Le territoire concerné.
     * @param loupe      Le nouveau niveau limite de grossissement.
     */
    public static void modifierLoupeTerritoire (Territoire territoire, double loupe)
    {
        HistoireMondiale.instance().donneesIGraphique().modifLoupeTerritoire (territoire, loupe) ;
    }
    
	public static void annulerChangementCentre() 
	{
        DialCentre 	dialCentre     = HistoireMondiale.instance().dialCentre() ;
		PointSphere valeurInitiale = dialCentre.getCentreInitial() ;
		EtatAppli etatAppli        = HistoireMondiale.instance().etat() ;
    	etatAppli.modifPtCentralCarte(valeurInitiale) ;
    	ActionsFenetres.masquerDialCentre () ;
	}

	public static void validerChangementCentre() 
	{
        ActionsFenetres.masquerDialCentre() ;
	}
    
}
