/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 * 
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou le modifier sous les
 * termes de la licence Affero GPL publiée par la Fondation pour le logiciel libre (Free
 * Software Foundation), en choisissant la version 3 de cette licence ou n'importe quelle
 * version ultérieure, à votre convenance.
 * 
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE D'AUCUNE SORTE
 * : y compris d'être vendable ou de pouvoir servir un but donné. Voir le texte de la
 * licence AGPL pour plus de détails.
 * 
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire. Si ce n'est pas le
 * cas, regardez à cette adresse : <http://www.gnu.org/licenses/>.
 * 
 */
package fr.histoiremondiale.histoire.igraphique.tracercartes;

import static fr.histoiremondiale.histoire.EtatAppli.MAXI_LOUPE;
import static java.lang.Math.toDegrees;
import static java.lang.Math.toRadians;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Stroke;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import fr.histoiremondiale.histoire.DonneesIGraphique;
import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.donnees.Fleuve;
import fr.histoiremondiale.histoire.donnees.Limes;
import fr.histoiremondiale.histoire.donnees.Mer;
import fr.histoiremondiale.histoire.donnees.Pourtour;
import fr.histoiremondiale.histoire.donnees.Terre;
import fr.histoiremondiale.histoire.donnees.Territoire;
import fr.histoiremondiale.histoire.donnees.TexteTerritoire;
import fr.histoiremondiale.histoire.igraphique.PanCarte;
import fr.histoiremondiale.histoire.utiles.exttypes.Chaines;
import fr.histoiremondiale.histoire.utiles.igraphique.dessin.hachures.PeintureHachures;
import fr.histoiremondiale.histoire.utiles.math.CalculsPlan;
import fr.histoiremondiale.histoire.utiles.math.CoordSphere;
import fr.histoiremondiale.histoire.utiles.math.Matrice;
import fr.histoiremondiale.histoire.utiles.math.PointPlan;
import fr.histoiremondiale.histoire.utiles.math.PointSphere;

/**
 * La classe permettant de tracer les cartes.
 */
public class TraceurCarte
{
	public static boolean TEST = false;
	//la police pour le nom des fleuves
	private static Font fonteEcolier = new Font("Ecolier",Font.PLAIN,36);
	
	// Constantes de style
	// (hachures)
	private static final int HACHURES_AUCUNE = 0;
	/*
	 * Note : ici pour documentation, pas utilisées directement : les nombres issus des
	 * données sont directement utilisés comme indices de tableaux. private static final
	 * int HACHURES_HORIZONTALES = 1 ; private static final int HACHURES_VERTICALES = 2 ;
	 * private static final int HACHURES_DIAG_HG_BD = 3 ; private static final int
	 * HACHURES_DIAG_BG_HD = 4 ; private static final int HACHURES_CROISEES_DROITES = 5 ;
	 * private static final int HACHURES_CROISEES_DIAG = 6 ;
	 */
	// (frontières, fleuves, ...)
	private static final int LIGNE_AUCUNE = 0;
	private static final int LIGNE_PLEINE_1 = 1;
	/*
	 * Note : ici pour documentation, pas utilisées directement : les nombres issus des
	 * données sont directement utilisés comme indices de tableaux. private static final
	 * int LIGNE_PLEINE_2 = 2 ; private static final int LIGNE_PLEINE_3 = 3 ; private
	 * static final int LIGNE_PLEINE_4 = 4 ; private static final int LIGNE_PLEINE_5 = 5 ;
	 * private static final int LIGNE_PLEINE_6 = 6 ; private static final int
	 * LIGNE_POINTILLES_LONGS = 7 ; private static final int LIGNE_POINTILLES_COURTS = 8 ;
	 * private static final int LIGNE_POINTILLES_MIXTES_1 = 9 ; private static final int
	 * LIGNE_POINTILLES_MIXTES_2 = 10 ;
	 */

	// Paramètres et outils de tracé des hachures
	private static final Color COUL_DEFAUT_HACHURES = Color.BLACK;
	private static final double ESPACE_HACHURES = 10;
	private static final double EPAISSEUR_HACHURES = 2;
	private static final int LARGEUR_FLOU_HACHURES = 25;
	private static final PeintureHachures[][] PEINTURES_HACHURES = {
			{},
			{ new PeintureHachures(COUL_DEFAUT_HACHURES, ESPACE_HACHURES, 0, EPAISSEUR_HACHURES, LARGEUR_FLOU_HACHURES) },
			{ new PeintureHachures(COUL_DEFAUT_HACHURES, ESPACE_HACHURES, Math.PI / 2, EPAISSEUR_HACHURES,
					LARGEUR_FLOU_HACHURES) },
			{ new PeintureHachures(COUL_DEFAUT_HACHURES, ESPACE_HACHURES, Math.PI * 3 / 4, EPAISSEUR_HACHURES,
					LARGEUR_FLOU_HACHURES) },
			{ new PeintureHachures(COUL_DEFAUT_HACHURES, ESPACE_HACHURES, Math.PI * 1 / 4, EPAISSEUR_HACHURES,
					LARGEUR_FLOU_HACHURES) },
			{
					new PeintureHachures(COUL_DEFAUT_HACHURES, ESPACE_HACHURES, 0, EPAISSEUR_HACHURES,
							LARGEUR_FLOU_HACHURES),
					new PeintureHachures(COUL_DEFAUT_HACHURES, ESPACE_HACHURES, Math.PI / 2, EPAISSEUR_HACHURES,
							LARGEUR_FLOU_HACHURES) },
			{
					new PeintureHachures(COUL_DEFAUT_HACHURES, ESPACE_HACHURES, Math.PI * 3 / 4, EPAISSEUR_HACHURES,
							LARGEUR_FLOU_HACHURES),
					new PeintureHachures(COUL_DEFAUT_HACHURES, ESPACE_HACHURES, Math.PI * 1 / 4, EPAISSEUR_HACHURES,
							LARGEUR_FLOU_HACHURES) } };

	// Paramètres et outils de tracé des frontières
	// (le tableau de flottants dans le constructeur de BasicStroke correspond au motif
	// d'une ligne
	// discontinue : nb pixels remplis, nb pixels blancs, nb pixels remplis, nb pixels
	// blancs, ...)
	private static final int FORME_BOUT_FRONTIERE = BasicStroke.CAP_ROUND;
	// Forme du bout du pinceau (pour les largeurs de trait > 1)
	private static final int FORME_ANGLE_FRONTIERE = BasicStroke.JOIN_ROUND;
	// Forme du prolongement des angles des tracés (pour les largeurs de trait > 1)
	private static final float ANGLE_MIN_BOUT_TRIANGLE = 1.0f;
	// Un paramètre pour les angles prolongés un triangle
	private static final float DECALAGE_MOTIF = 0;
	// Décalage entre le début du tracé et la pose du pinceau (pour commencer par un
	// blanc quand un motif est fourni (ligne non pleine))
	private static final Stroke[] STYLES_FRONTIERES = {
			null,
			new BasicStroke(1.0f, FORME_BOUT_FRONTIERE, FORME_ANGLE_FRONTIERE, ANGLE_MIN_BOUT_TRIANGLE, new float[] {
					1, 0 }, DECALAGE_MOTIF),
			new BasicStroke(2.0f, FORME_BOUT_FRONTIERE, FORME_ANGLE_FRONTIERE, ANGLE_MIN_BOUT_TRIANGLE, new float[] {
					1, 0 }, DECALAGE_MOTIF),
			new BasicStroke(3.0f, FORME_BOUT_FRONTIERE, FORME_ANGLE_FRONTIERE, ANGLE_MIN_BOUT_TRIANGLE, new float[] {
					1, 0 }, DECALAGE_MOTIF),
			new BasicStroke(4.0f, FORME_BOUT_FRONTIERE, FORME_ANGLE_FRONTIERE, ANGLE_MIN_BOUT_TRIANGLE, new float[] {
					1, 0 }, DECALAGE_MOTIF),
			new BasicStroke(5.0f, FORME_BOUT_FRONTIERE, FORME_ANGLE_FRONTIERE, ANGLE_MIN_BOUT_TRIANGLE, new float[] {
					1, 0 }, DECALAGE_MOTIF),
			new BasicStroke(6.0f, FORME_BOUT_FRONTIERE, FORME_ANGLE_FRONTIERE, ANGLE_MIN_BOUT_TRIANGLE, new float[] {
					1, 0 }, DECALAGE_MOTIF),
			new BasicStroke(1.0f, FORME_BOUT_FRONTIERE, FORME_ANGLE_FRONTIERE, ANGLE_MIN_BOUT_TRIANGLE, new float[] {
					15, 5 }, DECALAGE_MOTIF),
			new BasicStroke(1.0f, FORME_BOUT_FRONTIERE, FORME_ANGLE_FRONTIERE, ANGLE_MIN_BOUT_TRIANGLE, new float[] {
					3, 3 }, DECALAGE_MOTIF),
			new BasicStroke(1.0f, FORME_BOUT_FRONTIERE, FORME_ANGLE_FRONTIERE, ANGLE_MIN_BOUT_TRIANGLE, new float[] {
					3, 5, 8, 5 }, DECALAGE_MOTIF),
			new BasicStroke(1.0f, FORME_BOUT_FRONTIERE, FORME_ANGLE_FRONTIERE, ANGLE_MIN_BOUT_TRIANGLE, new float[] {
					3, 5, 3, 5, 8, 5 }, DECALAGE_MOTIF), };

	/************************************************************************************
	 * on trace une carte en tenant compte du fait que le panneau sur lequel ira le dessin
	 * est rectangulaire
	 * 
	 * @param terres la liste des terres, il faut filtrer celle-ci pour ne calculer que ce
	 * qui est nécessaire (voir ci-dessous)
	 * @param mersInterieures la liste des mers, il faut filtrer celle-ci pour ne calculer
	 * que ce qui est nécessaire
	 * @param fleuves la liste des fleuves qu'il faut filtrer
	 * @param territoiresATracer la liste des territoires à tracer filtrés par l'année.
	 * C'est une liste de territoires. Il faut filtrer géographiquement celle-ci pour ne
	 * calculer que ce qui est nécessaire
	 * @param annee l'année est utilisée pour les textes à tracer (le filtrage des
	 * territoires par année est fait avant l'appel à cette méthode)
	 * @param fleuvesAffiches pour savoir si on affiche les fleuves
	 * @param meridiensParallelesAffiches pour savoir si on affiche les méridiens et
	 * parallèles
	 * @param centreCarte le point central de la carte. C'est une instance de la classe
	 * PointSphere, c'est à dire un point géographique (longitude, latitude)
	 * @param largeurCarte la largeur de la carte (largeur du panneau sur lequel elle sera
	 * dessinée)
	 * @param hauteurCarte la hauteur de la carte (hauteur du panneau sur lequel elle sera
	 * dessinée)
	 * @param loupe loupe la valeur du grossissement de la loupe.
	 * @param coulMers la couleur des mers (0xC0FFFF)
	 * @param coulTerres la couleur des terres (0xFFFFC0)
	 * @param coulFleuves la couleur des fleuves (0x000040)
	 * @param g un contexte graphique, on dessine via ce contexte graphique
	 * @param infosTraceImage une instance de InfosTracerImage. Elle permet de stocker
	 * certains calculs de points pour économiser du temps processeur en ne recalculant
	 * pas une deuxième fois ce qui vient d'être calculé.
	 * 
	 * Le filtrage géographique des données ne se fera que dans le dessin des diverses
	 * couches (il est inutile de filtrer les fleuves si on ne doit pas les dessiner). Le
	 * filtrage géographique consiste en l'opération décrite ci-après Deux points A et B
	 * sur la terre déterminent un angle entre eux, angle(AB) : soit T le centre de la
	 * terre, dans le triangle TAB, angle(AB) est l'angle opposé au segment AB. On
	 * s'intéresse au cosinus de cet angle. Il se calcule par la formule : cos(angle(AB))
	 * = cos(LatA)*cos(latB)*cos(LongA-LongB) + sin(LatA)*sin(LatB) Toute partie de la
	 * terre est décrite par un ensemble de points géographiques de coordonnées longitude
	 * et latitude exprimées en degrés décimaux. On détermine l'isobarycentre (I) de cet
	 * ensemble de points. Pour chaque point de l'ensemble, le calcul ci-dessus nous donne
	 * le cosinus de l'angle entre ce point et I. Le cosinus le plus faible correspond à
	 * l'angle le plus grand (alpha). Pour chaque point M on a angle(MI)<= alpha car
	 * cosinus(MI) >= cos(alpha). J'appelle cosinus de l'ensemble des points cette valeur,
	 * je l'appelle aussi improprement distance. J'appelle angle du territoire l'angle
	 * alpha. La carte est une partie de la terre. On choisit comme centre non pas
	 * l'isobarycentre mais naturellement le centre de la carte (C). Le point en haut à
	 * gauche de la carte correspond à un point du globe (H), le cercle de centre C et
	 * passant par H recouvre la partie du globe représentée par la carte, on prend donc
	 * naturellement le cosinus de l'angle(CH) comme cosinus de la carte. L'angle(CH) est
	 * obtenu par la fonction arcCosinus. Soit maintenant un territoire de centre A et
	 * d'angle alpha, la carte de centre C et d'angle beta. Pour que le territoire et la
	 * carte aient des points en commun il faut que angle(AC) <= alpha + beta Attention
	 * cependant, il ne s'agit que d'une condition nécessaire et pas suffisante. Cette
	 * condition peut être remplie et cependant le territoire et la carte n'ont aucun
	 * point en commun. C'est le cas en particulier si le territoire englobe la carte, ou
	 * s'il est en forme de croissant et que la carte est dans la partie évidée.
	 * 
	 * Pour le tracé d'une région, on commence par vérifier si elle est concernée par la
	 * carte (si elle a des points communs avec la carte. Son pourtour est une liste de
	 * points géographiques. Puis on détermine les projections des points à l'intérieur du
	 * cercle englobant la carte. En même temps on détermine les points qui font que le
	 * tracé entre ou sort du cercle. On détermine ensuite à l'aide des projections
	 * (contours) des entrées et des sorties l'ensemble des polygones qui représentent la
	 * région sur la carte (un territoire d'un seul tenant peut se retrouver découper en
	 * plusieurs polygones pour son tracé). Ce sont des points du plan, coordonnées X et
	 * Y, origine du plan au centre de l'écran, axe des abscisses horizontal, orienté de
	 * gauche à droite, axes des ordonnées vertical, orienté de bas en haut. Enfin on
	 * change de repère du plan pour qu'il corresponde à l'écran et on dessine les
	 * polygones par les méthodes de dessin de swing (ou de AWT).
	 ******************************************************************************************************************/

	public void tracerCarte(List<Terre> terres, List<Mer> mersInterieures, List<Fleuve> fleuves,
			List<Territoire> territoiresATracer, int annee, boolean fleuvesAffiches,
			boolean meridiensParallelesAffiches, PointSphere centreCarte, int largeurCarte, int hauteurCarte,
			double loupe, Color coulMers, Color coulTerres, Color coulFleuves, Graphics g,
			InfosTracerImage infosTraceImage)
	{
		// On sauvegarde la couleur du pinceau pour la restituer après
		Color coulInitPinceau = g.getColor();

		// la nouvelle image que l'on va construire
		InfosImage nouvImg = infosTraceImage.nouvImg;

		if (nouvImg.imgCompleteSansTextes == null)
		// si la carte n'existe pas, il faut la créer
		{
			// Création d'une "page blanche" sur laquelle dessiner
			nouvImg.imgCompleteSansTextes = new BufferedImage(largeurCarte, hauteurCarte, BufferedImage.TYPE_4BYTE_ABGR);

			// le contexte graphique, intermédiaire indispensable pour dessiner
			Graphics gCouche = nouvImg.imgCompleteSansTextes.getGraphics();

			// On dessine le fond de la carte (couleur de la mer)
			this.tracerMersPrincipales(coulMers, gCouche, largeurCarte, hauteurCarte);

			// Dessin des terres
			this.tracerTerres(terres, coulTerres, gCouche, largeurCarte, hauteurCarte, loupe, centreCarte,
					infosTraceImage);
			
			// Dessin des territoires
			this.tracerFondTerritoires(territoiresATracer, gCouche, largeurCarte, hauteurCarte, loupe, centreCarte,
					infosTraceImage);

			// Dessin des mers
			this.tracerMers(mersInterieures, coulMers, gCouche, largeurCarte, hauteurCarte, loupe, centreCarte,
					infosTraceImage);
			
			// Dessin des fleuves
			if (fleuvesAffiches)
			{
				this.tracerFleuves(fleuves, coulFleuves, gCouche, largeurCarte, hauteurCarte, loupe, centreCarte,
						infosTraceImage);
			}

			// Dessin des méridiens
			if (meridiensParallelesAffiches)
			{
				this
						.tracerMeridiensParalleles(gCouche, largeurCarte, hauteurCarte, loupe, centreCarte,
								infosTraceImage);
			}
		}
		// Tracer tout sauf les textes
		g.drawImage(nouvImg.imgCompleteSansTextes, 0, 0, null);

		// Les textes sont systématiquement retracés
		this.tracerTextesTerritoires(territoiresATracer, annee, g, largeurCarte, hauteurCarte, loupe, centreCarte,
				infosTraceImage);

		// restituer le pinceau initial
		g.setColor(coulInitPinceau);
	}

	 /******************************************************************************************************************
	 * on trace le fond de la carte (la première couche : le bleu de la mer)
	 *
	 * @param largeurCarte : la largeur de la carte (largeur du panneau sur lequel elle
	 * sera dessinée)
	 * @param hauteurCarte : la hauteur de la carte (hauteur du panneau sur lequel elle
	 * sera dessinée)
	 * @param coulMers : la couleur des mers (Color COUL_MERS = new Color (0xC0FFFF))
	 * @param g : un contexte graphique, on dessine via ce contexte graphique
	 *
	 * Il s'agit de remplir la carte de la couleur de la mer
	 ******************************************************************************************************************/

	public void tracerMersPrincipales(Color coulMers, Graphics g, int largeurCarte, int hauteurCarte)
	{
		// Sauver la couleur du pinceau
		Color coulInitPinceau = g.getColor();

		// Tracer le fond de la carte
		g.setColor(coulMers);
		g.fillRect(0, 0, largeurCarte, hauteurCarte);

		// Restaurer la couleur du pinceau
		g.setColor(coulInitPinceau);
	}

	/******************************************************************************************************************
	 * On trace les terres. Ce tracé ne dépend que du centre de la carte et de la loupe,
	 * pas du changement d'année
	 * 
	 * @param terres la liste des terres,
	 * @param coulTerres la couleur des terres
	 * @param g un contexte graphique, on dessine via ce contexte graphique
	 * @param largeurCarte la largeur de la carte en pixels
	 * @param hauteurCarte la hauteur de la carte en pixels
	 * @param loupe la loupe (1 à 25)
	 * @param centre le centre de la carte
	 * @param infosTracerImage le stockage des calculs réutilisables On commence par
	 * filtrer les terres en éliminant celles qui ne peuvent avoir d'intersection avec la
	 * carte. Pour chaque terre restante on va déterminer une liste de polygones
	 * permettant de la représenter. On vérifie si le pourtour de la terre a déjà fait
	 * l'objet d'un calcul avec ce centre, cette loupe et les dimensions de la carte. Si
	 * c'est le cas on récupère ce calcul, sinon on construit la liste des polygones, on
	 * stocke ce calcul pour une utilisation future éventuelle. Enfin on dessine la terre
	 * en question sur l'image.
	 ******************************************************************************************************************/

	public void tracerTerres(List<Terre> terres, Color coulTerres, Graphics g, int largeurCarte, int hauteurCarte,
			double loupe, PointSphere centre, InfosTracerImage infosTracerImage)
	{
		double rayonCarte = rayonCarte(largeurCarte, hauteurCarte, loupe, centre);
		for (Terre terre : terres)
		{
			/*
			 * filtrer les terres : on ne calcule quelque chose que si l'angle du centre
			 * de la carte au centre du pourtour est inférieur à la somme de l'angle du
			 * cercle limite de la carte et de l'angle du cercle limite du pourtour.
			 */

			if (this.proche(terre.pourtour(), centre, rayonCarte))
			// on ne traite que les terres qui peuvent avoir des points en commun avec la
			// carte
			{
				// Si le pourtour est dans InfosTraceTmage on le récupère
				// InfosTraceImages contient une liste associative récupérée via la
				// méthode
				// polygonesTraces() dont la clé est l'identifiant du pourtour, la valeur
				// est
				// une liste de polygones (voir determinerContours pour la construction de
				// cette liste)
				List<Polygon> polygones = infosTracerImage.polygonesTraces(terre.pourtour().id());
				if (polygones == null)
				// sinon on le construit
				{
					// La projection des points du pourtour d'une région détermine une
					// liste de contours, chaque contour est une liste de points du plan
					List<List<PointPlan>> contours = this.determinerContours(terre.pourtour().points(), largeurCarte,
							hauteurCarte, loupe, centre, infosTracerImage);
					// on détermine les contours (les projections des points du pourtour)
					if (!contours.isEmpty())
					// S'il y a des contours (au moins un)
					{

						polygones = new ArrayList<>();
						for (List<PointPlan> contour : contours)
						{
							// pour chaque contour, on détermine le polygone
							polygones.add(this.polygonePourTrace(contour));
						}
						// on sauvegarde les polygones liés au pourtour
						infosTracerImage.nouvImg.multiPolygonesPourtoursTraces.put(terre.pourtour().id(), polygones);
					}
				}

				if (polygones != null)
				{
					// S'il y a des polygones à tracer, on les trace
					for (Polygon polygone : polygones)
					{
						this.tracerRegion(polygone, coulTerres, null, 0, g);
					}
				}
			}
		}
	}

	 /******************************************************************************************************************
	 * On trace les Territoires. Ce tracé dépend du centre de la carte, de la loupe et
	 * aussi de l'année (la liste des territoires a été filtrée selon l'année avant
	 * l'appel)
	 *
	 * @param listeTerritoiresATracer : la liste des territoires, il faudra filtrer
	 * géographiquement celle-ci pour ne calculer que ce qui est nécessaire
	 * @param g : un contexte graphique, on dessine via ce contexte graphique
	 * @param centre le centre de la carte
	 * @param largeurCarte la largeur de la carte en pixels
	 * @param hauteurCarte la hauteur de la carte en pixels
	 * @param loupe la loupe (1 à 25)
	 * @param infosTracerImage stockage des calculs réutilisables On commence par
	 filtrer
	 * les territoires en éliminant ceux qui ne peuvent avoir d'intersection avec la
	 * carte. Pour chaque territoire restant on va déterminer une liste de polygones
	 * permettant de le représenter. On vérifie si le pourtour du territoire a déjà fait
	 * l'objet d'un calcul avec ce centre, cette loupe, les dimensions de la carte. Si
	 * c'est le cas on récupère ce calcul, sinon on construit la liste des polygones, on
	 * stocke ce calcul pour une utilisation future éventuelle (en particulier le clic
	 * droit sur la représentation graphique du territoire). Enfin on dessine le
	 * territoire en question sur l'image.
	 ******************************************************************************************************************/

	public void tracerFondTerritoires(List<Territoire> listeTerritoiresATracer, Graphics g, int largeurCarte,
			int hauteurCarte, double loupe, PointSphere centre, InfosTracerImage infosTracerImage)
	{
		double rayonCarte = rayonCarte(largeurCarte, hauteurCarte, loupe, centre);
		for (Territoire territoire : listeTerritoiresATracer)
		{
			if (this.proche(territoire.pourtour(), centre, rayonCarte))
			// si le territoire a une intersection possible avec la carte
			{
				List<Polygon> polygones = infosTracerImage.polygonesTraces(territoire.pourtour().id());
				if (polygones == null)
				// Si les polygones n'existent pas déjà, il faut les construire
				{
					// on commence par récupérer les contours (points dans le plan,
					// origine au centre)
					List<List<PointPlan>> contours = this.determinerContours(territoire.pourtour().points(),
							largeurCarte, hauteurCarte, loupe, centre, infosTracerImage);
					if (!contours.isEmpty())
					// s'il y a des contours
					{
						polygones = new ArrayList<>();
						for (List<PointPlan> contour : contours)
						{
							
							// pour chaque contour on construit le polygone
							// correspondant
							polygones.add(this.polygonePourTrace(contour));
						}
						// Sauvegarder les polygones du pourtour
						// Voir la remarque dans tracerTerres
						infosTracerImage.nouvImg.multiPolygonesPourtoursTraces.put(territoire.pourtour().id(),
								polygones);
						// Sauvegarder le territoire
						infosTracerImage.nouvImg.territoiresTraces.put(territoire.id(), territoire);
					}
				}
				// Sauvegarder le territoire s'il ne l'est pas déjà. En effet le pourtour
				// peut être déjà connu mais pas le territoire.
				// par exemple le pourtour de la Crète est connu par le dessin des terres
				// (avant le dessin des territoires) mais
				// la crète en tant que territoire est inconnue donc il faut la rajouter.
				if (!infosTracerImage.territoireConnu(territoire.id()))
				{
					// Si le territoire n'est pas stocké, le stocker
					infosTracerImage.nouvImg.territoiresTraces.put(territoire.id(), territoire);
				}
				if (polygones != null)
				{
					// S'il y a des polygones à tracer (au moins un)
					for (Polygon polygone : polygones)
					{
						// on trace le fond du territoire et ses hachures
						this.tracerRegion(polygone, territoire.coulInterieur(), territoire.coulFrontieres(), territoire
								.styleHachures(), g);
						// on trace la frontière ...
						if (territoire.styleFrontieres() > 0)
						{
							// ... si c'est nécessaire
							this.tracerFrontieres(polygone, territoire.coulFrontieres(), territoire.styleFrontieres(),
									g);
						}
					}
				}
			}
		}
	}

	 /******************************************************************************************************************
	 * On trace les mers intérieures. Ce tracé ne dépend que du centre de la carte et de
	 * la loupe.
	 * 
	 * @param mersInterieures les mers
	 * @param coulMers la couleur des mers (Color COUL_MERS = new Color (0xC0FFFF))
	 * @param g un contexte graphique, on dessine via ce contexte graphique
	 * @param centre le centre de la carte
	 * @param largeurCarte la largeur de la carte en pixels
	 * @param hauteurCarte la hauetur d ela carte en pixels
	 * @param loupe la loupe de 1 à 25
	 * @param infosTracerImage les informations pour tracer l'image
	 * On commence par filtrer les mers en éliminant celles qui ne peuvent avoir
	 * d'intersection avec la carte. Pour chaque mer restante on va déterminer une liste
	 * de polygones permettant de la représenter. On vérifie si le pourtour de la mer a
	 * déjà fait l'objet d'un calcul avec ce centre, cette loupe et les dimensions de la
	 * carte. Si c'est le cas on récupère ce calcul, sinon on construit la liste des
	 * polygones, on stocke ce calcul pour une utilisation future éventuelle. Enfin on
	 * dessine la mer en question sur l'image.
	 *
	 ******************************************************************************************************************/

	public void tracerMers(List<Mer> mersInterieures, Color coulMers, Graphics g, int largeurCarte, int hauteurCarte,
			double loupe, PointSphere centre, InfosTracerImage infosTracerImage)
	{
		double rayonCarte = rayonCarte(largeurCarte, hauteurCarte, loupe, centre);
		for (Mer mer : mersInterieures)
		{
			/*
			 * filtrer les mers : on ne calcule quelque chose que si l'angle du centre de
			 * la carte au centre du pourtour est inférieur à la somme de l'angle du
			 * cercle limite de la carte et de l'angle du cercle limite du pourtour.
			 */

			if (this.proche(mer.pourtour(), centre, rayonCarte))

			{
				// Si le pourtour est dans InfosTraceTmage on le récupère
				List<Polygon> polygones = infosTracerImage.polygonesTraces(mer.pourtour().id());
				if (polygones == null)
				// sinon on le construit
				{
					// La projection des points du pourtour d'une région détermine une
					// liste de contour
					// chaque contour est une liste de points du plan
					List<List<PointPlan>> contours = this.determinerContours(mer.pourtour().points(), largeurCarte,
							hauteurCarte, loupe, centre, infosTracerImage);
					// on détermine les contours (les projections des points du pourtour)
					if (!contours.isEmpty())
					// S'il y a des contours
					{
						polygones = new ArrayList<>();
						for (List<PointPlan> contour : contours)
						{
							// pour chaque contour, on détermine le polygone
							polygones.add(this.polygonePourTrace(contour));
						}
						// on sauvegarde les polygones liés au pourtour
						infosTracerImage.nouvImg.multiPolygonesPourtoursTraces.put(mer.pourtour().id(), polygones);
					}
				}
				if (polygones != null)
				{
					// S'il y a des polygones à tracer
					for (Polygon polygone : polygones)
					{
						this.tracerRegion(polygone, coulMers, null, 0, g);
					}
				}
			}
		}
	}

	/******************************************************************************************************************
	 * On trace les fleuves. Ce tracé ne dépend que du centre de la carte et de la loupe.
	 * Il est différent des autres tracés car on trace une ligne (et même plusieurs
	 * parfois) et on a pas un polygone On commence par filtrer les fleuves en éliminant
	 * ceux qui ne peuvent avoir d'intersection avec la carte. Pour chaque fleuve restant
	 * on va déterminer une liste de lignes permettant de la représenter. Enfin on dessine
	 * le fleuve en question sur l'image.
	 * 
	 * @param fleuves : la liste des fleuves qu'il faudra filtrer géographiquement pour ne
	 * calculer que ce qui est nécessaire
	 * @param coulFleuves : la couleur des fleuves (COUL_FLEUVES = new Color (0x000040))
	 * @param g : un contexte graphique, on dessine via ce contexte graphique
	 * @param largeurCarte la largeur de la carte en pixels
	 * @param hauteurCarte la hauetur de la carte en pixels
	 * @param centre le centre de la carte
	 * @param loupe la loupe (1 à 25)
	 * @param infosTracerImage les informations pour tracer l'image
	 ******************************************************************************************************************/

	public void tracerFleuves(List<Fleuve> fleuves, Color coulFleuves, Graphics g, int largeurCarte, int hauteurCarte,
			double loupe, PointSphere centre, InfosTracerImage infosTracerImage)
	{
		double rayonCarte = rayonCarte(largeurCarte, hauteurCarte, loupe, centre);
		for (Fleuve fleuve : fleuves)
		{
			if (this.proche(fleuve, centre, rayonCarte))
			{
				Limes limes = fleuve.lignes();
				// les projections sont représentées par plusieurs lignes brisées
				List<PointPlan> ligneBrisee = this.determinerLigneBrisee(limes.points(), largeurCarte,
							hauteurCarte, loupe, centre, infosTracerImage);
				if (!ligneBrisee.isEmpty())
				{
					// si la ligne n'est pas vide, on la trace
					this.tracerLigneBrisee(CalculsPlan.changementOrigineCentreVersCoinHautGauche(ligneBrisee,
								largeurCarte, hauteurCarte), coulFleuves, LIGNE_PLEINE_1, g);
				}
					
				// et maintenant le nom du fleuve
				if (fleuve.nom().isEmpty())	continue;
				Graphics2D g2 = (Graphics2D) g;
				Font oldFonte = g2.getFont();
				Color oldColor = g2.getColor();
				AffineTransform oldTransf = g2.getTransform();
				
				g2.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
				//le nom à écrire
				String s = fleuve.nom();
				//la taille de la police
				float taille = (float) (16.0f + (5.0f)*(loupe-1.0f));
				//la police utilisée
				Font fonte = fonteEcolier.deriveFont(taille);
				g2.setFont(fonte);
				//la couleur
				g2.setColor(coulFleuves);
				//les paramètres de passage des PointSphere aux PointPlan
				Matrice matriceChgt = CoordSphere.matChgtSphereVersPlan2(centre);
				int coteCarte = Math.max(largeurCarte, hauteurCarte);
				//le début du nom
				PointPlan P1 = CoordSphere.projection(fleuve.debut(), matriceChgt, loupe, coteCarte);
				P1 = CalculsPlan.changementOrigineCentreVersCoinHautGauche(P1, largeurCarte, hauteurCarte);
				int x1 = (int) P1.x(), y1 = (int) P1.y();  
				//la fin
				PointPlan P2 = CoordSphere.projection(fleuve.fin(), matriceChgt, loupe, coteCarte);
				P2 = CalculsPlan.changementOrigineCentreVersCoinHautGauche(P2, largeurCarte, hauteurCarte);
				int x2 = (int) P2.x(), y2 = (int) P2.y();
				//détermination de l'angle de rotation pour le nom
				int a = (x2-x1), b = (y2-y1), a2 = a*a, b2 = b*b;
				double d = Math.sqrt(a2+b2);
				double angleEcriture = Math.asin(b/d);
				if (a < 0) angleEcriture = Math.PI - angleEcriture;
		        //dessin du nom
				g2.rotate(angleEcriture, x1, y1);
				g2.drawString(s, x1, y1);
				
				//retour aux conditions initiales
				g2.setTransform(oldTransf);
				g2.setFont(oldFonte);
				g2.setColor(oldColor);
			}
		}
	}

	/******************************************************************************************************************
	 * On détermine une ligne brisée (une liste de points du plan) à partir d'une liste de
	 * points
	 * 
	 * @param points la liste des points géographiques représentant une ligne
	 * @param centreCarte le centre de la carte
	 * @param largeurCarte la largeur de la carte
	 * @param hauteurCarte la hauteur de la carte
	 * @param loupe la loupe (1 à 25)
	 * @param infosTracerImage les informatiosn pour tracer l'image
	 * @return une liste de points du plan (origine au centre de la carte)
	 * 
	 ******************************************************************************************************************/

	private List<PointPlan> determinerLigneBrisee(List<PointSphere> points, int largeurCarte, int hauteurCarte,
			double loupe, PointSphere centreCarte, InfosTracerImage infosTracerImage)
	{

		List<PointPlan> ligneBrisee = new ArrayList<>();
		// la projection du point courant
		PointPlan projectionPointCourant;
		// la projection du point précédent
		PointPlan ptPlanPrecedent = null;
		// le point est-il dans le cache de InfosTraceImage
		boolean enCache;
		// infosTraceImage
		// InfosTracerImage infosTracerImage = dc.getITI();
		Matrice matriceChgt = CoordSphere.matChgtSphereVersPlan2(centreCarte);
		int coteCarte = Math.max(largeurCarte, hauteurCarte);

		for (PointSphere pointCourant : points)
		{
			// le point courant est-il dans le cache
			enCache = (infosTracerImage != null) && (infosTracerImage.pointPlanConnu(pointCourant));
			// si oui on récupère sa projection, si non on construit la projection
			projectionPointCourant = (enCache ? infosTracerImage.pointPlan(pointCourant) : CoordSphere.projection(
					pointCourant, matriceChgt, loupe, coteCarte));
			// si le point n'est pas dans le cache on l'y met
			if (!enCache)
			{
				infosTracerImage.nouvImg.cachePoints.put(pointCourant, projectionPointCourant);
			}
			if (ptPlanPrecedent == null || !ptPlanPrecedent.confonduCoordEntieres(projectionPointCourant))
			// si la nouvelle projection n'est pas égale à la précédente on l'ajoute
			{
				ptPlanPrecedent = projectionPointCourant;
				ligneBrisee.add(projectionPointCourant);
			}
		}
		return ligneBrisee;
	}

	/**
	 * On détermine les contours (une liste de polygones) à partir d'une liste de points
	 * 
	 * On considère le cercle qui recouvre la région du globe représentées sur la carte
	 * (ce cercle est déterminé par la méthode rayonCarte à partir des dimensions de la
	 * carte et de la valeur de la loupe). On va déterminer une liste de points du plan
	 * qui représentent la projection du contour sur le plan d'origine le centre de la
	 * carte. Pour chaque point du pourtour on regarde si celui-ci est dans le cercle ou
	 * non, ainsi que son successeur (le successeur du dernier point est le premier car le
	 * pourtour est fermé). On distingue quatre cas.
	 * 
	 * Le point courant et son suivant sont dans le cercle : on ajoute la projection du
	 * point courant à la liste des points (sous la condition que ce nouveau point du plan
	 * n'est pas identique au précédent).
	 * 
	 * Le point courant est dans le cercle et le point suivant est en dehors : le point
	 * suivant est une sortie du cercle, on ajoute la projection du point courant à la
	 * liste, on calcule l'intersection du cercle avec le segment [point courant, point
	 * suivant], on ajoute cette intersection à la liste, on crée une sortie avec ce point
	 * et on la stocke dans la liste des entrées et sorties.
	 * 
	 * Le point courant est hors du cercle et le point suivant est dedans : le point
	 * courant est une entrée dans le cercle, on calcule l'intersection du cercle avec le
	 * segment [point courant, point suivant], on ajoute cette intersection à la liste, on
	 * cére une entrée avec ce point et on la stocke dans la liste des entrées et sorties.
	 * 
	 * Le point courant et le point suivant sont hors du cercle : rien à faire
	 * 
	 * @param points la liste des points géographiques représentant un pourtour
	 * @param centre le centre de la carte
	 * @param largeurCarte la largeur de la carte
	 * @param hauteurCarte la hauteur de la carte
	 * @param loupe la loupe
	 * @param infosTracerImage un objet pour conserver les calculs liés aux points
	 * 
	 * @return une liste de liste de points
	 */

	private List<List<PointPlan>> determinerContours(List<PointSphere> points, int largeurCarte, int hauteurCarte,
			double loupe, PointSphere centre, InfosTracerImage infosTracerImage)
	{
		// On détermine en une seule passe les projections, les ES, le polygone des
		// coordonnées géographiques
		List<PointPlan> ptsPlan = new ArrayList<>();
		/*
		 * on ne stocke pas plusieurs fois le même point donc on doit savoir si la
		 * projection du point courant est identique à celle du point précédent. Pour
		 * savoir cela il faut conserver la projection du point précédent.
		 */
		PointPlan ptPlanPrecedent = null;
		// Le point courant (point géographique). Au début c'est le premier
		PointSphere pointCourant = points.get(0);
		// La projection du point courant (point plan)
		PointPlan projectionPointCourant;
		// Pour savoir si le point courant est dans le cercle
		double rayonCarte = rayonCarte(largeurCarte, hauteurCarte, loupe, centre);
		boolean pointCourantDansLeCercle = centre.DansLeCercle(pointCourant, rayonCarte);
		
		// Le point suivant du pourtour
		PointSphere pointSuivant;
		// La projection du point suivant (point plan)
		PointPlan projectionPointSuivant;
		// Pour savoir si le point suivant est dans le cercle
		boolean pointSuivantDansLeCercle;
		// La liste des entrées et des sorties
		List<EntreeSortie> entreesEtSorties = new LinkedList<EntreeSortie>();
		// le polygone du pourtour des coordonnées géographiques (cas où il n'y a aucun
		// point dans le cercle)
		Polygon pourtour = new Polygon();
		Matrice matriceChgt = CoordSphere.matChgtSphereVersPlan2(centre);
		int a = Math.max(largeurCarte, hauteurCarte);
		double rayonPanneauCarte = Math.sqrt(largeurCarte * largeurCarte + hauteurCarte * hauteurCarte) / 2;
		for (int i = 0; i < points.size(); i++)
		{
			pointSuivant = points.get((i + 1) % points.size());
			pointSuivantDansLeCercle = centre.DansLeCercle(pointSuivant, rayonCarte);
			if (pointCourantDansLeCercle)
			// Le point courant est dans le cercle
			{
				// la projection est-elle déjà calculée ?
				if ((infosTracerImage != null) && (infosTracerImage.pointPlanConnu(pointCourant)))
				// si oui la récupérer,
				{
					projectionPointCourant = infosTracerImage.pointPlan(pointCourant);
				}
				else
				// si non la calculer et la sauvegarder
				{
					projectionPointCourant = CoordSphere.projection(pointCourant, matriceChgt, loupe, a);
					infosTracerImage.nouvImg.cachePoints.put(pointCourant, projectionPointCourant);
				}
				if (ptPlanPrecedent == null || !ptPlanPrecedent.confonduCoordEntieres(projectionPointCourant))
				// Si la nouvelle projection est distincte de la précédente
				{
					ptPlanPrecedent = projectionPointCourant;
					// Ajouter la projection du point courant à la liste des points du
					// plan
					ptsPlan.add(projectionPointCourant);
				}

				if (pointSuivantDansLeCercle)
				// Le point suivant est dans le cercle, il devient le point courant
				{
					pointCourant = pointSuivant;
					pointCourantDansLeCercle = true;
				}
				else
				// Le point suivant n'est pas dans le cercle : une sortie
				{
					// la projection est-elle déjà calculée ?
					if ((infosTracerImage != null) && (infosTracerImage.pointPlanConnu(pointSuivant)))
					// si oui la récupérer,
					{
						projectionPointSuivant = infosTracerImage.pointPlan(pointSuivant);
					}
					else
					// si non la calculer et la sauvegarder
					{
						projectionPointSuivant = CoordSphere.projection(pointSuivant, matriceChgt, loupe, a);
						infosTracerImage.nouvImg.cachePoints.put(pointSuivant, projectionPointSuivant);
					}

					// on détermine l'intersection entre le segment et le cercle
					// ATTENTION pour cette fonction, le premier point est dans le cercle
					// le deuxième est hors du cercle
					PointPlan pointSurLeCercle = CalculsPlan.intersectionSegmentCercle(projectionPointCourant,
							projectionPointSuivant, rayonPanneauCarte);
					//si l'une des coordonnées du point retourné est Double.NaN (Not a Number), c'est qu'un 
					//problème est arrivé pour le calcul. 
					//Si c'est l'abscisse on peut prendre le point courant, sinon on prend le point suivant
					if (pointSurLeCercle.x() == Double.NaN || pointSurLeCercle.y() == Double.NaN) 
					{
						System.out.println("carte (lxh) " + largeurCarte + "x" + hauteurCarte);
						System.out.println("rayon = " + rayonPanneauCarte);
						System.out.println("loupe = " + loupe);
						System.out.println("centre = " + centre.enChaineElaboree());
						System.out.println("point dans le cercle " + pointCourant.enChaineElaboree());
						System.out.println("point hors du cercle " + pointSuivant.enChaineElaboree());
						pointSurLeCercle = pointSurLeCercle.x() == Double.NaN ?
										   projectionPointCourant :
										   projectionPointSuivant;
//						throw new RuntimeException("Erreur sur la détermination du point sur le cercle");
					}
					// on ajoute cette intersection aux points qui vont représenter le
					// pourtour dans le plan
					ptsPlan.add(pointSurLeCercle);
					// le point précédent est celui que l'on vient d'ajouter
					ptPlanPrecedent = pointSurLeCercle;
					// c'est une sortie
					entreesEtSorties.add(new EntreeSortie(EntreeSortie.SORTIE, pointSurLeCercle, ptsPlan.size() - 1));
					// le point suivant devient le point courant
					pointCourant = pointSuivant;
					// il n'est pas dans le cercle
					pointCourantDansLeCercle = false;
				}
			}
			else
			// le point courant n'est pas dans le cercle
			{
				if (pointSuivantDansLeCercle)
				// si le point suivant est dans le cercle : une entrée
				{
					// la projection est-elle déjà calculée ?
					if ((infosTracerImage != null) && (infosTracerImage.pointPlanConnu(pointCourant)))
					// si oui la récupérer,
					{
						projectionPointCourant = infosTracerImage.pointPlan(pointCourant);
					}
					else
					// si non la calculer et la sauvegarder
					{
						projectionPointCourant = CoordSphere.projection(pointCourant, matriceChgt, loupe, a);
						infosTracerImage.nouvImg.cachePoints.put(pointCourant, projectionPointCourant);
					}

					// même chose pour le point suivant
					if ((infosTracerImage != null) && (infosTracerImage.pointPlanConnu(pointSuivant)))
					{
						projectionPointSuivant = infosTracerImage.pointPlan(pointSuivant);
					}
					else
					{
						projectionPointSuivant = CoordSphere.projection(pointSuivant, matriceChgt, loupe, a);
						infosTracerImage.nouvImg.cachePoints.put(pointSuivant, projectionPointSuivant);
					}

					// on détermine l'intersection entre le segment et le cercle
					// ATTENTION pour cette fonction, le premier point est dans le cercle
					// le deuxième est hors du cercle
					PointPlan pointSurLeCercle = CalculsPlan.intersectionSegmentCercle(projectionPointSuivant,
							projectionPointCourant, rayonPanneauCarte);
					if (pointSurLeCercle.x() == Double.NaN || pointSurLeCercle.y() == Double.NaN) 
					{
						System.out.println("carte (lxh) " + largeurCarte + "x" + hauteurCarte);
						System.out.println("rayon = " + rayonPanneauCarte);
						System.out.println("loupe = " + loupe);
						System.out.println("centre = " + centre.enChaineElaboree());
						System.out.println("point dans le cercle " + pointCourant.enChaineElaboree());
						System.out.println("point hors du cercle " + pointSuivant.enChaineElaboree());
						pointSurLeCercle = pointSurLeCercle.x() == Double.NaN ?
										   projectionPointSuivant :
										   projectionPointCourant;
//						throw new RuntimeException("Erreur sur la détermination du point sur le cercle");
					}
					
					// on ajoute cette intersection aux points qui vont représenter le
					// pourtour dans le plan
					ptsPlan.add(pointSurLeCercle);
					// le point précédent est celui que l'on vient d'ajouter
					ptPlanPrecedent = pointSurLeCercle;
					// c'est une entrée
					entreesEtSorties.add(new EntreeSortie(EntreeSortie.ENTREE, pointSurLeCercle, ptsPlan.size() - 1));
					// le point suivant devient le point courant
					pointCourant = pointSuivant;
					// il est dans le cercle
					pointCourantDansLeCercle = true;
				}
				else
				// Le point suivant n'est pas non plus dans le cercle
				{
					// s'il n'y a pas de points dans le cercle, on définit un polygone
					// dont les sommets sont les points
					// du pourtour dont les coordonnées géographiques sont multipliées par
					// 100
					// le méridien de longitude -168° peut être pris comme partage des
					// terres
					// Les longitudes vont de -180 à 180 alors que pour traiter facilement
					// le cas où il n'y a pas de
					// points dans le cercle il faut prendre -168 à 192
					if (ptsPlan.isEmpty())
					{
						double x = pointCourant.longitude();
						if (x < -168)
						{
							x = x + 360;
						}
						double y = pointCourant.latitude();
						pourtour.addPoint((int) (100 * x), (int) (100 * y));
					}
					// le point suivant devient le point courant
					pointCourant = pointSuivant;
					// il n'est pas dans le cercle
					pointCourantDansLeCercle = false;
				}
			}
		}
		// on renvoie une liste de polygones qui vont représenter la région dans le cercle
		return this.ConstruireContours(ptsPlan, entreesEtSorties, pourtour, centre, largeurCarte, hauteurCarte);
	}

	/******************************************************************************************************************
	 * On construit les contours à partir d'une liste de points du plan et d'une liste
	 * d'entrées et de sorties On considère le cercle qui recouvre la région du globe
	 * représentées sur la carte (ce cercle est déterminé dans la classe DessinCarte à
	 * partir des dimensions de la carte et de la valeur de la loupe). On doit construire
	 * la liste des points du plan à partir de la liste des points à tracer et des entrées
	 * sorties. En particulier on doit passer d'une sortie à une entrée en tournant sur le
	 * cercle dans le sens trigonométrique, et d'une entrée à une sortie en ajoutant les
	 * points entre les deux. S'il n'y a pas de points à tracer alors il faut vérifier si
	 * le centre est dans le pourtour, si oui la région remplit le cercle, si non la
	 * région n'est pas concernée
	 * 
	 * @param ptsATracer : la liste des points du plan à tracer
	 * @param entreesEtSorties : la liste ordonnée des entrées et des sorties
	 * @param pourtour : pourtour est un polygone construit à partir des points du poutour par le dessin.
	 * @param centre le centre de la carte
	 * @param largeurCarte la largeur de la carte
	 * @param hauteurCarte la hauteur de la carte
	 * @return une liste de listes de points du plan (origine en haut à gauche)
	 ******************************************************************************************************************/

	private List<List<PointPlan>> ConstruireContours(List<PointPlan> ptsATracer, List<EntreeSortie> entreesEtSorties,
			Polygon pourtour, PointSphere centre, int largeurCarte, int hauteurCarte)
	{
		List<List<PointPlan>> pointsContours = new ArrayList<List<PointPlan>>();
		// si pas de points à tracer, on regarde si le centre de la carte est à
		// l'intérieur du pourtour
		if (ptsATracer.isEmpty())
		{
			// X et Y sont les coordonnées géographiques du centre multipliées par 100 et
			// ramenées à [-168 , 192]
			double X = 100 * centre.longitude();
			if (X < -168)
			{
				X = X + 360;
			}
			double Y = 100 * centre.latitude();

			if (pourtour.contains(X, Y))
			// si le centre est à l'intérieur du pourtour, le contour recouvre toute la
			// carte
			// (un peu plus à cause de la division par deux)
			{
				List<PointPlan> pointsContour = new ArrayList<>();
				pointsContour.add(new PointPlan(0, 0));
				pointsContour.add(new PointPlan(0, hauteurCarte));
				pointsContour.add(new PointPlan(largeurCarte, hauteurCarte));
				pointsContour.add(new PointPlan(largeurCarte, 0));
				pointsContours.add(pointsContour);
			}
			// on renvoie les points du contours (éventuellement une liste vide)
			return pointsContours;
		}

		double rayonPanneauCarte = Math.sqrt(largeurCarte * largeurCarte + hauteurCarte * hauteurCarte) / 2;
		// à partir d'ici il y a des points à traiter
		if (entreesEtSorties.isEmpty())
		{
			// il n'y a pas d'entrées-sorties, les ptsATracer sont exactement le pourtour
			pointsContours.add(ptsATracer);
		}
		else
		{
			// il y a des entrées et des sorties, il faut générer les points à tracer sur
			// le cercle
			pointsContours = this.genererContoursATracer(ptsATracer, entreesEtSorties, rayonPanneauCarte);
		}

		// on renvoie les points du contours (éventuellement une liste vide)
		// il faut changer l'origine du plan avant de retourner le résultat
		// origine au centre --> origine en haut à gauche
		// axe des ordonnées vers le haut --> vers le bas

		return CalculsPlan
				.changementOrigineCentreVersCoinHautGauchePourTout(pointsContours, largeurCarte, hauteurCarte);
	}

	/******************************************************************************************************************
	 * On génère les contours quand il y a des entrées et des sorties du cercle On
	 * considère le cercle qui recouvre la région du globe représentées sur la carte (ce
	 * cercle est déterminé dans la classe DessinCarte à partir des dimensions de la carte
	 * et de la valeur de la loupe). On doit construire la liste des points du plan à
	 * partir de la liste des points à tracer et des entrées sorties. En particulier on
	 * doit passer d'une sortie à une entrée en tournant sur le cercle dans le sens
	 * trigonométrique, et d'une entrée à une sortie en ajoutant les points entre les
	 * deux. Lorsque l'on revient à une entrée déjà traitée cela forme un contour. S'il
	 * reste des entrées non traitées il faut recommencer.
	 * 
	 * @param ptsPlan : la liste des points du plan à tracer
	 * @param ES la liste des entrées et des sorties
	 * @param R rayon du cercle sur lequel on projette les entrées-sorties
	 * @return une liste de listes de points du plan (origine centre de la carte)
	 * 
	 ******************************************************************************************************************/

	private List<List<PointPlan>> genererContoursATracer(List<PointPlan> ptsPlan, List<EntreeSortie> ES, double R)
	{
		// Le premier point doit-être une entrée, si c'est une sortie on la déplace à la fin
		if (ES.get(0).isSortie())
		{
			ES.add(ES.remove(0));
			// TODO il faudrait ici tester les conditions suivantes et si
			// l'une n'est pas remplie déclencher une erreur : Le nombre d'éléments de ES
			// est pair et chaque entrée est suivie d'une sortie. Voir avec Sylvain
			// comment faire cela
		}

		// dans cette liste on élimine au fur et à mesure les ES traitées
		// il y a sûrement une manière plus "économique" de faire
		// cela mais beaucoup moins "lisible". C'est déjà assez compliqué comme cela.
		List<EntreeSortie> ESrestantes = new ArrayList<>(ES);
		// la liste dans l'ordre de traitement des ES
		List<List<EntreeSortie>> ESordonnees = new ArrayList<List<EntreeSortie>>();
		// construire la liste ordonnée des ES. L'ordre de traitement des ES est différent
		// de l'ordre dans la liste
		while (!ESrestantes.isEmpty())
		// tant qu'il reste des ES à traiter
		{
			// et sa sortie associée
			EntreeSortie sortieCourante = ESrestantes.get(1);

			// on va créer la liste des ES jusqu'à ce que l'on revienne à l'ES de départ
			List<EntreeSortie> ESbis = new ArrayList<>();
			// pour vérifier si on est revenu à la première entrée
			boolean contourFerme = false;

			while (!contourFerme)
			// tant que le contour n'est pas fermé
			{
				// la sortie courante sous forme d'un angle en degrés
				double angleSortieCourante = toDegrees(sortieCourante.getAngle());
				// récupérer l'entrée suivante c'est à dire
				// la première entrée rencontrée en parcourant le cercle dans le sens
				// trigonométrique
				// à partir de la sortie courante (on ne tient pas compte des ES déjà
				// traitées)
				int iEntreeSuivante = this.indiceEntreeSuivante(ESrestantes, angleSortieCourante);
				// on extrait l'entrée suivante de la liste et on l'ajoute à la liste des
				// ESbis
				EntreeSortie entreeSuivante = ESrestantes.remove(iEntreeSuivante);
				ESbis.add(entreeSuivante);
				// l'entrée suivante étant extraite on trouve la sortie suivante au même
				// index
				EntreeSortie sortieSuivante = ESrestantes.remove(iEntreeSuivante);
				ESbis.add(sortieSuivante);

				// Si on est revenu à la première, la liste ESbis pour le contour est
				// terminée
				contourFerme = (iEntreeSuivante == 0);
				sortieCourante = sortieSuivante;
			}
			ESordonnees.add(ESbis);
		}

		// on a maintenant une liste ordonnées des ES qu'il faut rassembler en plusieurs
		// contours
		// la nouvelle liste des contours
		List<List<PointPlan>> contours = new ArrayList<List<PointPlan>>();
		for (List<EntreeSortie> listeES : ESordonnees)
		// pour chaque liste d'entrées et de sorties, on construit le contour aassocié
		{
			// la nouvelle liste des points de contour (avec les points sur le cercle)
			List<PointPlan> ptsContour = new ArrayList<>();
			for (int i = 0; i < listeES.size(); i = i + 2)
			{
				// Récupérer les entrées/sorties
				EntreeSortie entreeCourante = listeES.get(i);
				EntreeSortie sortieCourante = listeES.get(i + 1);
				// Ajouter les points de l'entrée courante à la sortie courante
				int iPtEntree = entreeCourante.getIndex();
				int iPtSortie = sortieCourante.getIndex();
				// point de départ
				int iPt = iPtEntree;
				// on l'ajoute aux ptsContour
				ptsContour.add(ptsPlan.get(iPt));
				// comme le parcours des points se fait d'une manière cyclique (une fois
				// rendu à la fin on recommence au début) je vais utiliser plutôt une 
				//boucle do while pour parcourir les points
				do
				{
					// parcours cyclique des indices des points
					iPt = (iPt + 1) % ptsPlan.size();
					// on ajoute le point
					ptsContour.add(ptsPlan.get(iPt));
				}
				while (iPt != iPtSortie);

				// l'entrée suivante (c'est la première si on est en train de traiter la
				// dernière)
				EntreeSortie entreeSuivante = listeES.get((i + 2) % listeES.size());
				// Ajouter les points sur le cercle (de la sortie courante à l'entrée
				// suivante)
				double angleSortieCourante = toDegrees(sortieCourante.getAngle());
				
				double angleEntreeSuivante = toDegrees(entreeSuivante.getAngle());
				
				// l'angle d'entrée doit être supérieur à l'angle de sortie
				if (angleEntreeSuivante < angleSortieCourante)
				{
					angleEntreeSuivante += 360;
				}
				// aller de la sortie à l'entrée (tous les points de 10 degrés en 10
				// degrés entre la sortie
				// et l'entrée
				for (double angle = multipleDe10Sup(angleSortieCourante); angle < angleEntreeSuivante; angle += 10)
				{
					ptsContour.add(CalculsPlan.creerPointCoordAngulaires(R, toRadians(angle % 360)));
				}
			}
			contours.add(ptsContour);
		}
		return contours;
	}

	/******************************************************************************************************************
	 * On cherche la première entrée qui suit la sortie en tournant dans le sens
	 * trigonométrique. Il ne reste que des entrées non déjà traitées. On considère le
	 * cercle qui recouvre la région du globe représentées sur la carte (ce cercle est
	 * déterminé dans la classe DessinCarte à partir des dimensions de la carte et de la
	 * valeur de la loupe). Parmi toutes les entrées on repère celle dont l'angle est
	 * minimum.
	 * 
	 * @param es : une liste d'entrées et de sorties, en commençant par une entrée
	 * @param angleSortie : la sortie exprimée sous forme d'un angle
	 * @return l'indice de l'entrée suivante
	 ******************************************************************************************************************/

	private int indiceEntreeSuivante(List<EntreeSortie> es, double angleSortie)
	{
		double angleMin = 1000; // valeur de l'angle minimum trouvé, initialement une
		// valeur supérieure à 360°
		int iAngleMin = -1; // Indice de l'entrée présentant le plus petit angle
		// Chercher le point d'entrée présentant le plus petit angle par rapport à l'angle
		// du point de sortie
		for (int i = 0; i < es.size(); i = i + 2)
		{
			EntreeSortie entree = es.get(i);
			double angle = toDegrees(entree.getAngle()) - angleSortie;
			// Si l'angle est négatif, le ramener dans l'intervalle [0 , 360]
			if (angle < 0)
			{
				angle += 360;
			}

			// Regarder si on a trouvé un plus petit angle
			if (angle < angleMin)
			{
				angleMin = angle;
				iAngleMin = i;
			}
		}
		// Renvoie l'indice de la prochaine entrée/sortie dans la liste
		return iAngleMin;
	}

	/******************************************************************************************************************
	 * Tracer une ligne brisée
	 * 
	 * @param ptsPlan : les points à tracer
	 * @param coulLigne : la couleur du trait
	 * @param styleLigne : l'épaisseur du trait
	 * @param g : le contexte graphique
	 * 
	 ******************************************************************************************************************/

	private void tracerLigneBrisee(List<PointPlan> ptsPlan, Color coulLigne, int styleLigne, Graphics g)
	{
		int[] coordX; // Coordonnées X des points dans le plan
		int[] coordY; // Coordonnées Y des points dans le plan

		// S'il n'y a pas de points, rien à tracer (ça arrive à cause du filtrage des
		// points)
		// ou si le style rend la ligne invisible
		if (ptsPlan.isEmpty() || styleLigne == LIGNE_AUCUNE)
		{
			return;
		}

		// Préparer les outils
		Graphics2D g2 = (Graphics2D) g;

		// Préparer les coordonnées
		int[][] coord = this.coordPourTrace(ptsPlan);
		coordX = coord[0];
		coordY = coord[1];

		// Tracer la ligne
		g2.setStroke(STYLES_FRONTIERES[styleLigne]);
		g2.setColor(coulLigne);
		g2.drawPolyline(coordX, coordY, coordX.length);
	}

	/******************************************************************************************************************
	 * Tracer un polygone
	 * 
	 * @param contours : le polygone à tracer
	 * @param coulRegion : la couleur de l'intérieur
	 * @param coulHachures : la couleur des hachures
	 * @param styleHachures : le style des hachures
	 * @param g : le contexte graphique
	 * 
	 ******************************************************************************************************************/

	private void tracerRegion(Polygon contours, Color coulRegion, Color coulHachures, int styleHachures, Graphics g)
	{
		// S'il n'y a pas de points, rien à tracer (ça arrive à cause du filtrage des
		// points)
		// En plus ça fait planter le hachurage quand il n'y a aucune valeur dans les
		// tableaux de coordonnées
		// Attention, contrairement aux méthodes voisines on ne peut pas faire le test des
		// hachures
		// maintenant : dans tous les cas le fond doit être peint .
		if (contours.npoints == 0)
		{
			return;
		}

		// Préparer les outils
		Graphics2D g2 = (Graphics2D) g;

		// tracer les hachures en même temps que le fond
		// peindre la région
		g2.setColor(coulRegion);
		g2.fillPolygon(contours);

		// Peindre les hachures s'il y en a
		if (styleHachures != HACHURES_AUCUNE)
		{
			// Mémoriser le style de remplissage
			Paint sauvPeinture = g2.getPaint();
			// Peindre les hachures
			for (PeintureHachures peinture : PEINTURES_HACHURES[styleHachures])
			{
				peinture.modifCoulHachures(coulHachures);
				g2.setPaint(peinture);
				g2.fillPolygon(contours);
			}
			// Rétablir le style de remplissage d'origine
			g2.setPaint(sauvPeinture);
		}
	}

	/******************************************************************************************************************
	 * Renvoie un polygone à la place d'une liste de points du plan
	 * 
	 * @param ptsPlan : la liste des points du plan
	 * @return un polygone
	 ******************************************************************************************************************/

	private Polygon polygonePourTrace(List<PointPlan> ptsPlan)
	{
		int[][] coordonnees = this.coordPourTrace(ptsPlan);
		return new Polygon(coordonnees[0], coordonnees[1], coordonnees[0].length);
	}

	/******************************************************************************************************************
	 * détermine les éléments du polygone à partir de la liste des points.
	 * 
	 * @param ptsPlan : la liste des points du plan
	 * @return un tableau d'entiers sur deux colonnes
	 ******************************************************************************************************************/

	private int[][] coordPourTrace(List<PointPlan> ptsPlan)
	{
		int[] coordX; // Coordonnées X des points dans le plan
		int[] coordY; // Coordonnées Y des points dans le plan

		// Préparer les coordonnées
		coordX = new int[ptsPlan.size()];
		coordY = new int[ptsPlan.size()];
		for (int i = 0; i < ptsPlan.size(); i++)
		{
			PointPlan ptPlan = ptsPlan.get(i);
			coordX[i] = (int) ptPlan.x();
			coordY[i] = (int) ptPlan.y();
		}

		// Renvoyer les coordonnées calculées
		return new int[][] { coordX, coordY };
	}

	/******************************************************************************************************************
	 * Renvoie le premier multiple de 10 supérieur à la valeur en paramètre
	 * 
	 * @param val : la valeur pour laquelle on calcule la fonction
	 * @return un multiple de 10
	 * 
	 ******************************************************************************************************************/

	private static double multipleDe10Sup(double val)
	{
		return (10 + 10 * Math.floor(val / 10));
	}

	/******************************************************************************************************************
	 * Trace les contours d'une région
	 * 
	 * @param contours : le polygone à tracer
	 * @param coulFrontieres la couleur de la frontiere
	 * @param styleFrontieres l'épaisseur de la frontière
	 * @param g : le contexte graphique
	 * 
	 ******************************************************************************************************************/

	private void tracerFrontieres(Polygon contours, Color coulFrontieres, int styleFrontieres, Graphics g)
	{
		// S'il n'y a pas de points (ça arrive à cause du filtrage des points) ou pas de
		// ligne de frontière
		// à tracer, rien à tracer
		if (contours.npoints == 0 || styleFrontieres == LIGNE_AUCUNE)
		{
			return;
		}

		// Préparer les outils
		Graphics2D g2 = (Graphics2D) g;

		// Tracer la frontière
		g2.setStroke(STYLES_FRONTIERES[styleFrontieres]);
		g2.setColor(coulFrontieres);
		g2.drawPolygon(contours);
	}

	/******************************************************************************************************************
	 * Trace les textes d'un territoire
	 * 
	 * @param listeTerritoiresATracer : le polygone à tracer
	 * @param annee : l'année
	 * @param g : le contexte graphique
	 * @param loupe la loupe (1 à 25)
	 * @param centre le centre de la carte
	 * @param largeurCarte la largeur de la carte
	 * @param hauteurCarte la hauteur de la carte
	 * @param infosTracerImage les informations du tracé
	 ******************************************************************************************************************/

	public void tracerTextesTerritoires(List<Territoire> listeTerritoiresATracer, int annee, Graphics g,
			int largeurCarte, int hauteurCarte, double loupe, PointSphere centre, InfosTracerImage infosTracerImage)
	{
		Matrice matrice = CoordSphere.matChgtSphereVersPlan2(centre);
		int coteCarte = Math.max(largeurCarte, hauteurCarte);
		double rayonCarte = rayonCarte(largeurCarte, hauteurCarte, loupe, centre);
		// double distanceLimite = dc.getCosLimiteCarte ();
		int fontTailleMaxi = 25;
		int fontTailleMini = 11;
		float fontTaille = (float) (fontTailleMini + (fontTailleMaxi - fontTailleMini) * (loupe - 1) / MAXI_LOUPE);
		Font policeNormale = g.getFont().deriveFont(Font.BOLD, fontTaille);
		Font policeItalique = g.getFont().deriveFont(Font.BOLD + Font.ITALIC, fontTaille);
		DonneesIGraphique dIG = HistoireMondiale.instance().donneesIGraphique();

		for (Territoire territoire : listeTerritoiresATracer)
		{
			if (this.proche(territoire.pourtour(), centre, rayonCarte))
			{
				if (!Chaines.estVide(territoire.nom()))
				{

					if (dIG.loupeTerritoire(territoire) < loupe)
					{
						PointSphere ptCentreTexteTerritoire = dIG.centreTexteTerritoire(territoire);

						if (centre.DansLeCercle(ptCentreTexteTerritoire, rayonCarte))
						{
							PointPlan ptCentreTerritoire = CalculsPlan.changementOrigineCentreVersCoinHautGauche(
									CoordSphere.projection(ptCentreTexteTerritoire, matrice, loupe, coteCarte),
									largeurCarte, hauteurCarte);
							Graphics2D g2 = (Graphics2D) g;
							g.setColor(Color.BLACK);
							String nomTerritoire = territoire.nom();
							TexteCarte txtNomTerritoire = new TexteCarte(nomTerritoire.split("\\$"), policeNormale);
							TexteTerritoire infoTerritoire = territoire.texteEn(annee);
							TexteCarte txtInfoTerritoire;
							boolean tracerCadre;
							if (infoTerritoire == null)
							{
								txtInfoTerritoire = new TexteCarte();
								tracerCadre = false;
							}
							else
							{
								String[] texte = infoTerritoire.texte().split("&");
								boolean italique = infoTerritoire.italique();
								txtInfoTerritoire = new TexteCarte(texte, (italique ? policeItalique : policeNormale));
								tracerCadre = !(italique || texte[0].equals(""));
							}
							int yCoinHGTextes = (int) ptCentreTerritoire.y()
									- TexteCarte.hauteurTextes(txtNomTerritoire, txtInfoTerritoire, g2) / 2;
							int xTxtTerritoire = (int) ptCentreTerritoire.x() - txtNomTerritoire.largeur(g2) / 2;
							int yTxtTerritoire = yCoinHGTextes;
							int xTxtInfo = (int) ptCentreTerritoire.x() - txtInfoTerritoire.largeur(g2) / 2;
							int yTxtInfo = yTxtTerritoire + txtNomTerritoire.hauteur(g2);
							txtNomTerritoire.tracerLignesCentrees(g2, xTxtTerritoire, yTxtTerritoire, false);
							txtInfoTerritoire.tracerLignesCentrees(g2, xTxtInfo, yTxtInfo, tracerCadre);
							// Mémoriser la position du texte
							int xTextes = Math.min(xTxtTerritoire, xTxtInfo);
							int yTextes = yCoinHGTextes;
							int largeurTextes = Math.max(txtNomTerritoire.largeur(g2), txtInfoTerritoire.largeur(g2));
							int hauteurTextes = (yTxtInfo + txtInfoTerritoire.hauteur(g2)) - yTextes;
							Rectangle rectTextes = new Rectangle(xTextes, yTextes, largeurTextes, hauteurTextes);
							infosTracerImage.nouvImg.textesTerritoiresTraces.put(rectTextes, territoire);

						}
					}
				}
			}
		}
	}

	/******************************************************************************************************************
	 * Tracer les méridiens et les parallèles (à améliorer)
	 * 
	 * @param largeurCarte : la largeur de la carte (largeur du panneau sur lequel elle
	 * sera dessinée)
	 * @param hauteurCarte : la hauteur de la carte (hauteur du panneau sur lequel elle
	 * sera dessinée)
	 * @param g : un contexte graphique, on dessine via ce contexte graphique
	 * @param centre centre de la carte
	 * @param largeurCarte largeur de la carte
	 * @param hauteurCarte hauteur de la carte
	 * @param loupe la loupe (1 à 25)
	 * @param infosTracerImage les informations pour tracer l'image
	 ******************************************************************************************************************/

	public void tracerMeridiensParalleles(Graphics g, int largeurCarte, int hauteurCarte, double loupe,
			PointSphere centre, InfosTracerImage infosTracerImage)
	{
		this.tracerMeridiens(g, largeurCarte, hauteurCarte, loupe, centre, infosTracerImage);
		this.tracerParalleles(g, largeurCarte, hauteurCarte, loupe, centre, infosTracerImage);
	}

	/**
	 * determinerPasLongitude
	 * 
	 * @param hemisphereNord indique si le centre est dans l'hémisphère nord ou pas
	 * @param centre centre de la carte
	 * @param largeurCarte largeur de la carte
	 * @param hauteurCarte hauteur de la carte
	 * @param loupe la loupe (1 à 25)
	 * @return le pas en longitude qui indique quels méridiens vont être tracés
	 ****************************************************************************************/

	private int determinerPasLongitude(boolean hemisphereNord, int largeurCarte, int hauteurCarte, double loupe,
			PointSphere centre)
	{
		double amplitude;
		// si on est dans l'hémisphère nord l'amplitude de longitude se trouve en bas de
		// la carte
		if (hemisphereNord)
		{
			PointSphere BD = CoordSphere.InvProjection(new PointPlan(largeurCarte, hauteurCarte), centre, largeurCarte,
					hauteurCarte, loupe);

			PointSphere BG = CoordSphere.InvProjection(new PointPlan(0, hauteurCarte), centre, largeurCarte,
					hauteurCarte, loupe);
			amplitude = BD.longitude() - BG.longitude();
			if (amplitude < 0)
			{
				amplitude += 360;
			}
		}
		// si on est dans l'hémisphère sud l'amplitude de longitude se trouve en haut de
		// la carte
		else
		{
			PointSphere HG = CoordSphere.InvProjection(new PointPlan(0, 0), centre, largeurCarte, hauteurCarte, loupe);
			PointSphere HD = CoordSphere.InvProjection(new PointPlan(largeurCarte, 0), centre, largeurCarte,
					hauteurCarte, loupe);
			amplitude = HD.longitude() - HG.longitude();
			if (amplitude < 0)
			{
				amplitude += 360;
			}
		}

		// suivant le nombre de pixels par degré on détermine le pas (tous les combien
		// on trace les parallèles)
		double pixelsParDegre = largeurCarte / amplitude;
		if (pixelsParDegre > 150)
		{
			return 1;
		}
		if (pixelsParDegre > 75)
		{
			return 2;
		}
		if (pixelsParDegre > 30)
		{
			return 5;
		}
		return 10;
	}

	/**
	 * renvoie le pas en latitude, la latitude maximum, la latitude minimum
	 * 
	 * @param centre centre de la carte
	 * @param largeurCarte largeur de la carte
	 * @param hauteurCarte hauteur de la carte
	 * @param loupe la loupe (1 à 25)
	 * @return un tableau de 3 nombres
	 ****************************************************************************************/

	private double[] determinerPasLatitude(int largeurCarte, int hauteurCarte, double loupe, PointSphere centre)
	{
		Matrice matrice = CoordSphere.matChgtSphereVersPlan2(centre);
		int a = Math.max(largeurCarte, hauteurCarte);

		PointSphere HG = CoordSphere.InvProjection(new PointPlan(0, 0), centre, largeurCarte, hauteurCarte, loupe);
		PointSphere HM = CoordSphere.InvProjection(new PointPlan(largeurCarte / 2, 0), centre, largeurCarte,
				hauteurCarte, loupe);
		PointSphere BG = CoordSphere.InvProjection(new PointPlan(0, hauteurCarte), centre, largeurCarte, hauteurCarte,
				loupe);
		PointSphere BM = CoordSphere.InvProjection(new PointPlan(largeurCarte / 2, hauteurCarte), centre, largeurCarte,
				hauteurCarte, loupe);

		// l'ordonnée du point de latitude maxi et celle du point de latitude mini
		double yH = 0, yB = hauteurCarte;
		// la latitude en haut et en bas
		double LH, LB;

		// si le point en haut à gauche est plus au nord que celui du milieu (hémisphère
		// nord)
		// la latitude maxi est en haut à gauche, sinon au milieu
		LH = ((HG.latitude() > HM.latitude()) ? HG.latitude() : HM.latitude());
		// si le point en bas à gauche est plus au sud que celui du milieu (hémisphère
		// sud)
		// la latitude mini est en bas à gauche, sinon au milieu
		LB = ((BG.latitude() < BM.latitude()) ? BG.latitude() : BM.latitude());

		// si le Pôle nord est dans la carte, la latitude maxi est 90°
		// le point haut est l'ordonnées du pôle nord
		// l'angle pour les marques est obtenu en fonction du PN
		if (centre.latitude() >= (90 - PanCarte.ANGLE_CARTE / 2))
		{
			PointPlan PN = CalculsPlan.changementOrigineCentreVersCoinHautGauche(CoordSphere.projection(
					new PointSphere(0, 90), matrice, loupe, a), largeurCarte, hauteurCarte);
			if (PN.y() > 0)
			{
				LH = 90;
				yH = PN.y();
			}
		}
		// sinon si le Pôle sud est dans la carte, la latitude mini est -90°
		// le point bas est l'ordonnées du pôle sud
		else if (centre.latitude() <= (-90 + PanCarte.ANGLE_CARTE / 2))
		{
			PointPlan PS = CalculsPlan.changementOrigineCentreVersCoinHautGauche(CoordSphere.projection(
					new PointSphere(0, -90), matrice, loupe, a), largeurCarte, hauteurCarte);
			if (PS.y() < hauteurCarte)
			{
				LB = -90;
				yB = PS.y();
				// angle = toDegrees (atan (dc.getLargeurCarte ()/(2.0*yB)));
			}
		}

		// suivant le nombre de pixels par degré on détermine tous les combien on dessine
		// les parallèles
		double pixelsParDegre = (yB - yH) / (LH - LB);
		if (pixelsParDegre > 150)
		{
			return new double[] { 1.0, LH, LB };
		}
		else if (pixelsParDegre > 75)
		{
			return new double[] { 2.0, LH, LB };
		}
		else if (pixelsParDegre > 30)
		{
			return new double[] { 5.0, LH, LB };
		}
		else
		{
			return new double[] { 10.0, LH, LB };
		}
	}

	/**
	 * tracerMeridiens *****************************************************************
	 * 
	 * @param g un contexte graphique
	 * @param centre centre de la carte
	 * @param largeurCarte largeur de la carte
	 * @param hauteurCarte hauteur de la acrte
	 * @param loupe la loupe (1 à 25)
	 * @param infosTracerImage les informations pour tracer la carte
	 */

	private void tracerMeridiens(Graphics g, int largeurCarte, int hauteurCarte, double loupe, PointSphere centre,
			InfosTracerImage infosTracerImage)
	{
		// un booléen qui indique si on est dans l'hémisphère Nord
		boolean hemisphereNord = (centre.latitude() >= 0);
		// un rectangle représentant la carte
		Rectangle carte = new Rectangle(0, 0, largeurCarte, hauteurCarte);
		// le pas qui indique quels méridiens sont tracés
		int pas = this.determinerPasLongitude(hemisphereNord, largeurCarte, hauteurCarte, loupe, centre);
		double rayonCarte = rayonCarte(largeurCarte, hauteurCarte, loupe, centre);
		int coteCarte = Math.max(largeurCarte, hauteurCarte);
		Matrice matriceChgt = CoordSphere.matChgtSphereVersPlan2(centre);

		// pour toutes les longitudes de pas en pas
		for (int longitude = 180; longitude > -180; longitude -= pas)
		{
			// la liste des pointSphères représentant le méridien
			List<PointSphere> ptsSphere = new ArrayList<>();
			// recherche du premier point du méridien dans le cercle
			int lat = 0;
			if (hemisphereNord)
			{
				for (int latitude = -89; latitude < 89; latitude++)
				// si on est dans l'hémisphère nord on va du sud au nord
				{
					PointSphere pointSphere = new PointSphere(longitude, latitude);
					if (centre.DansLeCercle(pointSphere, rayonCarte))
					{
						lat = latitude;
						ptsSphere.add(new PointSphere(longitude, lat - 1));
						ptsSphere.add(pointSphere);
						break;
					}
				}
			}
			else
			{
				for (int latitude = 89; latitude > -89; latitude--)
				// si on est dans l'hémisphère sud on va du nord au sud.
				{
					PointSphere pointSphere = new PointSphere(longitude, latitude);
					if (centre.DansLeCercle(pointSphere, rayonCarte))
					{
						lat = latitude;
						ptsSphere.add(new PointSphere(longitude, lat + 1));
						ptsSphere.add(pointSphere);
						break;
					}
				}
			}

			// recherche des points suivants du méridien dans le cercle
			if (!ptsSphere.isEmpty())
			{
				if (hemisphereNord)
				{
					for (int latitude = lat + 1; latitude < 89; latitude++)
					{
						PointSphere pointSphere = new PointSphere(longitude, latitude);
						ptsSphere.add(pointSphere);
						// on arrête dés qu'on sort du cercle
						if (!(centre.DansLeCercle(pointSphere, rayonCarte)))
						{
							break;
						}
					}
				}
				else
				{
					for (int latitude = lat - 1; latitude > -89; latitude--)
					{
						PointSphere pointSphere = new PointSphere(longitude, latitude);
						ptsSphere.add(pointSphere);
						// on arrête dés qu'on sort du cercle
						if (!(centre.DansLeCercle(pointSphere, rayonCarte)))
						{
							break;
						}
					}
				}

				// on crée leur projection et la ligne brisée représentant le méridien
				PointPlan projectionPointCourant;
				PointPlan ptPlanPrecedent = null;
				boolean enCache;
				List<PointPlan> ligneBrisee = new ArrayList<>();
				for (PointSphere pointCourant : ptsSphere)
				{
					enCache = (infosTracerImage != null) && (infosTracerImage.pointPlanConnu(pointCourant));
					projectionPointCourant = (enCache ? infosTracerImage.pointPlan(pointCourant) : CoordSphere
							.projection(pointCourant, matriceChgt, loupe, coteCarte));
					if (!enCache)
					{
						infosTracerImage.nouvImg.cachePoints.put(pointCourant, projectionPointCourant);
					}
					if (ptPlanPrecedent == null || !ptPlanPrecedent.confonduCoordEntieres(projectionPointCourant))
					{
						ptPlanPrecedent = projectionPointCourant;
						ligneBrisee.add(projectionPointCourant);
					}
				}

				if (!ligneBrisee.isEmpty())
				// si la ligne brisée n'est pas vide
				// (à priori c'est le cas mais on ne sait jamais)
				{
					// on trace le méridien
					List<PointPlan> ligneBriseeAjustee = CalculsPlan.changementOrigineCentreVersCoinHautGauche(
							ligneBrisee, largeurCarte, hauteurCarte);
					this.tracerLigneBrisee(ligneBriseeAjustee, Color.LIGHT_GRAY, LIGNE_PLEINE_1, g);

					// on cherche l'endroit où écrire la marque
					PointPlan precedent = null;
					PointPlan courant = null;
					// on cherche l'intersection entre le méridien et la carte
					// courant est dans la carte, précédent est en dehors
					for (PointPlan P : ligneBriseeAjustee)
					{
						if (carte.contains(P.x(), P.y()))
						{
							courant = P;
							break;
						}
						precedent = P;
					}

					// la marque ne s'écrit que si le méridien traverse le bas du cadre
					// pour l'hémisphère nord et le haut pour l'hémisphère sud.
					if (precedent != null && courant != null)
					{
						double alpha, y0 = 0, x0, // les coorconnées de l'intesection
						x1 = courant.x(), y1 = courant.y(), // les coordonnées du point
						// courant
						x2 = precedent.x(), y2 = precedent.y(); // celles du point
						// précédent
						// pour éviter de coller la marque au bas de la carte (hémisphère
						// nord)
						if (hemisphereNord && y2 > hauteurCarte)
						{
							y0 = hauteurCarte - 10;
						}
						// pour éviter de coller la marque au haut de la carte (hémisphère
						// sud)
						if (!hemisphereNord && y2 < 0)
						{
							y0 = 10;
						}
						// si courant et précédent ont même abscisse alors l'intersection
						// aussi
						if (x2 == x1)
						{
							x0 = x1;
							// sinon il faut calculer a (y = ax + b) et en déduire
							// l'abscisse de l'intersection
						}
						else
						{
							alpha = (y2 - y1) / (x2 - x1);
							x0 = x1 + (y0 - y1) / alpha;
						}
						// si cette abscisse est dans la carte, on écrit la marque
						if (x0 > 0 && x0 < largeurCarte)
						{
							g.drawString("" + longitude, (int) x0, (int) y0);
						}
					}
				}
			}
		}
	}

	/**
	 * tracerParalleles ****************************************************************
	 * 
	 * @param g un contexte graphique
	 * @param centre centre de la carte
	 * @param largeurCarte largeur de la carte
	 * @param hauteurCarte hauteur de la acrte
	 * @param loupe la loupe (1 à 25)
	 * @param infosTracerImage les informations pour tracer la carte
	 */
	private void tracerParalleles(Graphics g, int largeurCarte, int hauteurCarte, double loupe, PointSphere centre,
			InfosTracerImage infosTracerImage)
	{
		// un booléen qui indique si on est dans l'hémisphère Nord
		boolean hemisphereNord = (centre.latitude() >= 0);
		// calcul du pas, de la latitude maxi et la latitude mini
		double[] tableau = this.determinerPasLatitude(largeurCarte, hauteurCarte, loupe, centre);
		// le pas qui indique quels parallèles sont tracés
		int pas = (int) tableau[0];
		// les latitudes minimales et maximales
		double latMaxi = tableau[1], latMini = tableau[2];
		// la longitude de la marque, par défaut la longitude du coin en bas à gauche
		PointSphere PS = CoordSphere.InvProjection(new PointPlan(0, hauteurCarte), centre, largeurCarte, hauteurCarte,
				loupe);
		double longMarque = PS.longitude();

		// ajustement de la latitude maxi en fonction du pas et de la présence du pôle
		// nord
		// sur la carte.
		if (latMaxi == 90)
		{
			latMaxi = 90 - pas;
		}
		else
		{
			latMaxi = Math.floor(latMaxi / pas) * pas;
		}

		// ajustement de la latitude mini en fonction du pas et de la présence du pôle sud
		// sur la carte.
		if (latMini == -90)
		{
			latMini = -90 + pas;
		}
		else
		{
			latMini = Math.ceil(latMini / pas) * pas;
		}

		if (hemisphereNord)
		{
			for (int latitude = (int) latMini; latitude <= latMaxi; latitude += pas)
			{
				longMarque = this.traiterParallele(latitude, longMarque, g, largeurCarte, hauteurCarte, loupe, centre,
						infosTracerImage);
			}
		}
		else
		{
			for (int latitude = (int) latMaxi; latitude >= latMini; latitude -= pas)
			{
				longMarque = this.traiterParallele(latitude, longMarque, g, largeurCarte, hauteurCarte, loupe, centre,
						infosTracerImage);
			}
		}
	}

	/**
	 * renvoie la longitude de la marque, soit la longitude du point où le parallèle coupe
	 * le bord gauche, soit la valeur d'entrée si le parallèle ne coupe pas le bord
	 * 
	 * @param latitude du parallèle à traiter
	 * @param longMarque la longitude de la marque à tracer (c'est la longitude de la marque du dernier parallèle coupant le bord gauche)
	 * @param g le contexte graphiqque
	 * @param centre centre de la carte
	 * @param largeurCarte largeur de la carte
	 * @param hauteurCarte hauteur de la acrte
	 * @param loupe la loupe (1 à 25)
	 * @param infosTracerImage les informations pour tracer la carte
	 * @return la longitude de la marque
	 */

	private double traiterParallele(int latitude, double longMarque, Graphics g, int largeurCarte, int hauteurCarte,
			double loupe, PointSphere centre, InfosTracerImage infosTracerImage)
	{
		double marqueLongitude = longMarque;
		List<PointSphere> ptsSphere = new ArrayList<>();
		// on ajoute tous les points en commençant à -90° de la longitude du
		// centre et en allant jusqu'à 270°
		for (int longitude = (int) (centre.longitude() - 90); longitude < centre.longitude() + 270; longitude++)
		{
			ptsSphere.add(new PointSphere(longitude, latitude));
		}
		// on détermine la projection du parallèle
		List<List<PointPlan>> parallele = this.determinerContours(ptsSphere, largeurCarte, hauteurCarte, loupe, centre,
				infosTracerImage);

		// s'il y a un parallèle à tracer
		if (!parallele.isEmpty())
		{
			// la liste des polygones à tracer
			List<Polygon> polygones = new ArrayList<>();

			// pour chaque morceau on construit le polygone correspondant
			for (List<PointPlan> morceau : parallele)
			{
				polygones.add(this.polygonePourTrace(morceau));
			}

			// S'il y a des polygones à tracer (au moins un), on les trace
			if (polygones != null)
			{
				for (Polygon polygone : polygones)
				{
					this.tracerFrontieres(polygone, Color.lightGray, LIGNE_PLEINE_1, g);
				}

				// les marques
				// seul le premier morceau du parallèle est nécessaire
				marqueLongitude = this.tracerMarque(parallele.get(0), marqueLongitude, g, latitude, largeurCarte,
						hauteurCarte, centre, loupe);
			}
		}

		return marqueLongitude;
	}

	/**
	 * tracerMarque
	 * 
	 * @param list la liste des points du plan qui représentent le parallèle
	 * @param longMarque la longitude de la marque précédente
	 * @param g le contexte graphique
	 * @param latitude la latitude du parallèle
	 * @param centre centre de la carte
	 * @param largeurCarte largeur de la carte
	 * @param hauteurCarte hauteur de la acrte
	 * @param loupe la loupe (1 à 25)
	 * @return la longitude de la marque, c'est la longitude du point d'intersection entre
	 * le bord gauche de la carte et le parallèle
	 */

	private double tracerMarque(List<PointPlan> list, double longMarque, Graphics g, int latitude, int largeurCarte,
			int hauteurCarte, PointSphere centre, double loupe)
	{
		// le rectangle dans lequel est dessiné la carte
		Rectangle carte = new Rectangle(0, 0, largeurCarte, hauteurCarte);

		int coteCarte = Math.max(largeurCarte, hauteurCarte);
		Matrice matriceChgt = CoordSphere.matChgtSphereVersPlan2(centre);

		double marqueLongitude = longMarque;
		// l'endroit où l'on va écrire la marque du parallèle
		int xMarque = -1, yMarque = -1;
		// la longitude 90° à l'ouest du centre
		double longitude = centre.longitude() - 90;
		// le pointSphère correspondant
		PointSphere P = new PointSphere(longitude, latitude);
		// le point du plan correspondant
		PointPlan ptPlan = CalculsPlan.changementOrigineCentreVersCoinHautGauche(CoordSphere.projection(P, matriceChgt,
				loupe, coteCarte), largeurCarte, hauteurCarte);
		// si le premier point n'est pas sur l'écran c'est que ce parallèle
		// peut rencontrer le bord gauche
		if (!carte.contains(ptPlan.x(), ptPlan.y()))
		{
			// on cherche deux points consécutifs tels que le premier est hors de la
			// carte et le deuxième est intérieur à la carte.
			// les PointPlans pour déterminer l'intersection du parallèle et du bord
			// gauche
			PointPlan precedent = ptPlan;
			PointPlan courant = null;
			for (PointPlan Pt : list)
			{
				if (carte.contains(Pt.x(), Pt.y()))
				{
					courant = Pt;
					break;
				}
				precedent = Pt;
			}

			// le point courant est sur la carte, le précédent en dehors
			// si le point courant est null, c'est que tous les points sont hors carte
			// il n'y a pas de marque à tracer

			if (courant != null)
			{
				double alpha, y0, x0 = 0, // les coordonnées de l'intersection
				x1 = courant.x(), y1 = courant.y(), // les coordonnées du point courant
				x2 = precedent.x(), y2 = precedent.y(); // les coordonnées du précédent
				// si le parallèle coupe le bord gauche de la carte
				if (x2 != x1)
				{
					alpha = (y2 - y1) / (x2 - x1);
					y0 = y1 - alpha * x1;
					if (y0 > 0 && y0 < hauteurCarte)
					{
						// les coordonnées de la marque
						xMarque = (int) x0;
						yMarque = (int) y0;
						// on récupère la longitude de la marque
						PointSphere PS = CoordSphere.InvProjection(new PointPlan(x0, y0), centre, largeurCarte,
								hauteurCarte, loupe);
						marqueLongitude = PS.longitude();
					}
					else
					{
						for (int lo = (int) marqueLongitude; lo != centre.longitude(); lo++)
						{
							if (lo == 181)
							{
								lo = -179;
							}
							PointSphere PS = new PointSphere(lo, latitude);
							PointPlan PP = CalculsPlan.changementOrigineCentreVersCoinHautGauche(CoordSphere
									.projection(PS, matriceChgt, loupe, coteCarte), largeurCarte, hauteurCarte);
							if (carte.contains(PP.x(), PP.y()))
							{
								xMarque = (int) PP.x();
								yMarque = (int) PP.y();
								if (yMarque < 10)
								{
									yMarque = 10;
								}
								if (yMarque > hauteurCarte - 10)
								{
									yMarque = hauteurCarte - 10;
								}
								break;
							}
						}
					}
				}
				else
				{
					System.out.println("Alerte : le segment courant-précédent est parallèle au bord gauche");
					System.out.println("ce cas ne devrait jamais arriver");
				}
			}
		}
		// sinon le bord gauche du parallèle est sur la carte
		else
		{
			for (int lo = (int) marqueLongitude; lo != centre.longitude(); lo++)
			{
				if (lo == 181)
				{
					lo = -179;
				}
				PointSphere PS = new PointSphere(lo, latitude);
				PointPlan PP = CalculsPlan.changementOrigineCentreVersCoinHautGauche(CoordSphere.projection(PS,
						matriceChgt, loupe, coteCarte), largeurCarte, hauteurCarte);
				if (carte.contains(PP.x(), PP.y()))
				{
					xMarque = (int) PP.x();
					yMarque = (int) PP.y();
					if (yMarque < 10)
					{
						yMarque = 10;
					}
					if (yMarque > hauteurCarte - 10)
					{
						yMarque = hauteurCarte - 10;
					}
					break;
				}
			}
		}
		if (xMarque > -1)
		{
			g.setColor(Color.magenta);
			Font ancFont = g.getFont();
			g.setFont(ancFont.deriveFont(Font.ITALIC));
			g.drawString("" + latitude, xMarque, yMarque);
			g.setFont(ancFont);
		}
		return marqueLongitude;
	}

	/**
	 * renvoie le cosinus de l'angle au centre du cercle terrestre qui recouvre la carte
	 * 
	 * @param centre le centre de la carte
	 * @param largeurCarte la largeur du panneau
	 * @param hauteurCarte la hauteur du panneau
	 * @param loupe la loupe
	 * @return le cosinus de l'angle au centre
	 * 
	 */
	public static double rayonCarte(int largeurCarte, int hauteurCarte, double loupe, PointSphere centre)
	{
		// les coordonnées géographiques du coin en haut à gauche
		PointSphere P = CoordSphere.InvProjection(new PointPlan(0, 0), centre, largeurCarte, hauteurCarte, loupe);
		// Le cosinus de l'angle limite
		return centre.cosArcAB(P);
	}

	/**
	 * @param pourtour le pourtour à tester
	 * @param rayonCarte le rayon osculateur de la carte
	 * @param centre le centre de la carte
	 * @return vrai si le pourtour est proche de la carte, faux sinon
	 * 
	 */
	public boolean proche(Pourtour pourtour, PointSphere centre, double rayonCarte)
	{
		PointSphere B = pourtour.centreCercleLimite();
		double cosB = pourtour.cosCercleLimite();
		return centre.voisin(rayonCarte, B, cosB);
	}

	/**
	 * @param fleuve le fleuve à tester
	 * @param centre le centre de la carte
	 * @param rayonCarte le rayon du cercle osculateur de la carte
	 * @return vrai si le fleuve est proche de la carte, faux sinon
	 */
	public boolean proche(Fleuve fleuve, PointSphere centre, double rayonCarte)
	{
		PointSphere B = fleuve.centreCercleLimite();
		double cosB = fleuve.cosCercleLimite();
		return centre.voisin(rayonCarte, B, cosB);
	}
}
