/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.stockage;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import fr.histoiremondiale.histoire.donnees.Fleuve;
import fr.histoiremondiale.histoire.donnees.Limes;
import fr.histoiremondiale.histoire.donnees.LimesOrdonne;
import fr.histoiremondiale.histoire.donnees.Mer;
import fr.histoiremondiale.histoire.donnees.Pourtour;
import fr.histoiremondiale.histoire.donnees.Terre;
import fr.histoiremondiale.histoire.utiles.math.PointSphere;



/**
 * Classe d'accès aux données géographiques.
 */
public class AccesDonneesGeographiques extends AccesDonneesBase
{
 
    
    /**
     * Constructeur.
     * @param chemRepDonnees Chemin du répertoire des données de l'application.
     */
    public AccesDonneesGeographiques (String chemRepDonnees)
    {
        super (chemRepDonnees) ;
    }
    
    
    
    /**
     * Charge et indexe les limes et renvoie une table d'association id -> limes.
     * @return La table d'association.
     */
    public Map<Integer,Limes> chargerLimesIndexes ()
    {
        try
        {
            String txtRequeteIdPointMax = "SELECT MAX(points.id)" +
            		                      "  FROM points" ;
            String txtRequetePoints     = "SELECT points.id, points.longitude, points.latitude" +
                                          "  FROM points" +
                                          " ORDER BY points.id" ;
            String txtRequeteLimes      = "SELECT limes.id, limes.id_premier_point, limes.id_dernier_point" +
                                          "  FROM limes" ;
            
            
            // Charger les données
            Map<Integer,Limes> indexLimes      = new HashMap<>() ;
            ResultSet          resIdPointsMax  = connexionCourante().prepareStatement(txtRequeteIdPointMax).executeQuery() ;
            ResultSet          resPoints       = connexionCourante().prepareStatement(txtRequetePoints).executeQuery() ;
            ResultSet          resLimes        = connexionCourante().prepareStatement(txtRequeteLimes).executeQuery() ;
            // (points)
            resIdPointsMax.next() ;
            int           idPointMax = resIdPointsMax.getInt (1) ;
            PointSphere[] points     = new PointSphere[idPointMax+1] ;
            while (resPoints.next())
            {
                int    idPoint   = resPoints.getInt    (1) ;
                double longitude = resPoints.getDouble (2) ;
                double latitude  = resPoints.getDouble (3) ;
                points[idPoint] = new PointSphere (longitude, latitude) ;
            }
            // (limes)
            while (resLimes.next())
            {
                int idLimes        = resLimes.getInt (1) ;
                int idPremierPoint = resLimes.getInt (2) ;
                int idDernierPoint = resLimes.getInt (3) ;
                List<PointSphere> pointsLimes = new ArrayList<>() ;
                for (int i = idPremierPoint ; i <= idDernierPoint ; i++)
                {
                    if (points[i] != null)
                        pointsLimes.add (points[i]) ;
                }
                indexLimes.put (idLimes, new Limes (pointsLimes)) ;
            }

            // Fermer les objets de résultat
            resIdPointsMax.close() ;
            resPoints.close() ;
            resLimes.close() ;
            
            
            // Renvoyer le résultat
            return indexLimes ;
        }
        catch (SQLException e)
        {
            throw new RuntimeException ("Erreur lors du chargement des données", e) ;
        }
        
    }
    

    /**
     * Charge les définitions des terres et les renvoie.
     * @param indexLimes Table d'association id -> limes.
     * @return La liste des terres.
     */
    public List<Terre> chargerTerres (Map<Integer,Limes> indexLimes)
    {
        try
        {
            String txtRequete = "SELECT terres.id, terres.nom, terres.id_pourtour," +
            		            "       pourtours.lat_centre, pourtours.long_centre, pourtours.cos_cercle_limite," +
            		            "       pourtours_limes.id_limes, pourtours_limes.sens" +
                                "  FROM terres" +
                                "  JOIN pourtours       ON terres.id_pourtour = pourtours.id" +
                                "  JOIN pourtours_limes ON pourtours.id       = pourtours_limes.id_pourtour" +
                                " ORDER BY terres.id, pourtours_limes.rang" ;
            
            // Charger les données
            List<Terre>        terres                     = new ArrayList<>() ;
            ResultSet          res                        = connexionCourante().prepareStatement(txtRequete).executeQuery() ;
            Integer            idTerreCourante            = null ;
            List<LimesOrdonne> limesOrdonnesTerreCourante = null ;
            Terre              terreCourante              = null ;
            while (res.next())
            {
                // Lire la ligne
                int     idTerre                 = res.getInt     ("terres.id") ;
                String  nomTerre                = res.getString  ("terres.nom") ;
                int     idPourtour              = res.getInt     ("terres.id_pourtour");
                double  latitudeCentrePourtour  = res.getDouble  ("pourtours.lat_centre") ;
                double  longitudeCentrePourtour = res.getDouble  ("pourtours.long_centre") ;
                double  cosCercleLimitePourtour = res.getDouble  ("pourtours.cos_cercle_limite") ;
                int     idLimes                 = res.getInt     ("pourtours_limes.id_limes") ;
                boolean sensLimes               = res.getBoolean ("pourtours_limes.sens") ;
                
                // Créer un nouvel objet si nécessaire
                if (terreCourante == null || idTerre != idTerreCourante)
                {
                    idTerreCourante            = idTerre ;
                    limesOrdonnesTerreCourante = new ArrayList<>() ;
                    terreCourante              = new Terre (idTerre,
                                                            nomTerre,
                                                            new Pourtour (idPourtour,
                                                                          limesOrdonnesTerreCourante,
                                                                          new PointSphere (longitudeCentrePourtour,
                                                                                           latitudeCentrePourtour),
                                                                          cosCercleLimitePourtour)) ;
                    terres.add (terreCourante) ;
                }
                
                // Collecter les limes ordonnés
                limesOrdonnesTerreCourante.add (new LimesOrdonne (indexLimes.get (idLimes), sensLimes)) ;
            }
            
            // Fermer les objets de résultat
            res.close() ;
            
            // Renvoyer le résultat
            return terres ;
        }
        catch (SQLException e)
        {
            throw new RuntimeException ("Erreur lors du chargement des données", e) ;
        }
    }


    /**
     * Charge les définitions des mers et les renvoie.
     * @param indexLimes Table d'association id -> limes.
     * @return La liste des mers.
     */
    public List<Mer> chargerMers (Map<Integer,Limes> indexLimes)
    {
        try
        {
            String txtRequete = "SELECT mers.id, mers.nom, mers.id_pourtour," +
                                "       pourtours.lat_centre, pourtours.long_centre, pourtours.cos_cercle_limite," +
                                "       pourtours_limes.id_limes, pourtours_limes.sens" +
                                "  FROM mers" +
                                "  JOIN pourtours       ON mers.id_pourtour = pourtours.id" +
                                "  JOIN pourtours_limes ON pourtours.id     = pourtours_limes.id_pourtour" +
                                " ORDER BY mers.id, pourtours_limes.rang" ;
            
            // Charger les données
            List<Mer>          mers                     = new ArrayList<>() ;
            ResultSet          res                      = connexionCourante().prepareStatement(txtRequete).executeQuery() ;
            Integer            idMerCourante            = null ;
            List<LimesOrdonne> limesOrdonnesMerCourante = null ;
            Mer                merCourante              = null ;
            while (res.next())
            {
                // Lire la ligne
                int     idMer                   = res.getInt     ("mers.id") ;
                String  nomMer                  = res.getString  ("mers.nom") ;
                int     idPourtour              = res.getInt     ("mers.id_pourtour");
                double  latitudeCentrePourtour  = res.getDouble  ("pourtours.lat_centre") ;
                double  longitudeCentrePourtour = res.getDouble  ("pourtours.long_centre") ;
                double  cosCercleLimitePourtour = res.getDouble  ("pourtours.cos_cercle_limite") ;
                int     idLimes                 = res.getInt     ("pourtours_limes.id_limes") ;
                boolean sensLimes               = res.getBoolean ("pourtours_limes.sens") ;
                
                // Créer un nouvel objet si nécessaire
                if (merCourante == null || idMer != idMerCourante)
                {
                    idMerCourante            = idMer ;
                    limesOrdonnesMerCourante = new ArrayList<>() ;
                    merCourante              = new Mer (idMer,
                                                        nomMer,
                                                        new Pourtour (idPourtour,
                                                                      limesOrdonnesMerCourante,
                                                                      new PointSphere (longitudeCentrePourtour,
                                                                                       latitudeCentrePourtour),
                                                                      cosCercleLimitePourtour)) ;
                    mers.add (merCourante) ;
                }
                
                // Collecter les limes ordonnés
                limesOrdonnesMerCourante.add (new LimesOrdonne (indexLimes.get (idLimes), sensLimes)) ;
            }
            
            // Fermer les objets de résultat
            res.close() ;
            
            // Renvoyer le résultat
            return mers ;
        }
        catch (SQLException e)
        {
            throw new RuntimeException ("Erreur lors du chargement des données", e) ;
        }
    }
    
    
    /**
     * Charge les définitions des fleuves et les renvoie.
     * 
     * Dans la méthode précédente, un fleuve était représenté par des limes non contigus contrairement aux
     * pourtours. C'était dinc une liste de limes. Donc on dessine le premier limes du fleuve, on relève le
     * crayon puis on dessine le suivant et ainsi de suite jusqu'à la fin des limes. Cela impliquait de 
     * créer des doublons et en particulier pour un nouveau fleuve on ne pouvait pas ré-utiliser des limes
     * existants pour les pourtours.
     *  
     * Dans la nouvelle méthode un fleuve est une liste de limes orientés. Les bras du delta
     * d'un fleuve deviennent des fleuves sans nom. C'est leur "géographie" qui les raccordent 
     * aux tracé du fleuve principal.
     * 
     * @param indexLimes Table d'association id -> limes.
     * @return La liste des fleuves.
     */
	public List<Fleuve> chargerFleuves (Map<Integer,Limes> indexLimes)
	{
		try
        {
			String txtRequete = "SELECT fleuves.id, fleuves.nom, fleuves.long_centre, "
			+ "	fleuves.lat_centre, fleuves.cos_cercle_limite,"
			+ " fleuves.lo_deb, fleuves.la_deb, fleuves.lo_fin, fleuves.la_fin,"
			+ "	fleuves_limes.id_limes, fleuves_limes.rang, fleuves_limes.sens"
			+ "	FROM fleuves"
			+ "	JOIN fleuves_limes ON fleuves.id = fleuves_limes.id_fleuve"
			+ "	ORDER by fleuves.id, fleuves_limes.rang;";
			List<Fleuve> fleuves = new ArrayList<>() ;
			ResultSet    res     = connexionCourante().prepareStatement(txtRequete).executeQuery() ;
    		// Charger les données
    		Integer     idFleuveCourant     = null ;
    		Limes  		lignesFleuveCourant = null ;
    		Fleuve      fleuveCourant       = null ;
    		while (res.next())
    		{
    			// Lire la ligne
    			int    idFleuve              = res.getInt    ("id") ;
    			String nomFleuve             = res.getString ("nom") ;
    			double latitudeCentreFleuve  = res.getDouble ("lat_centre") ;
    			double longitudeCentreFleuve = res.getDouble ("long_centre") ;
    			double cosCercleLimiteFleuve = res.getDouble ("cos_cercle_limite") ;
    			int    idLimes               = res.getInt    ("id_limes") ;
    			boolean sensLimes			 = res.getBoolean("sens");
    			double loDeb				 = res.getDouble ("lo_deb");
    			double laDeb				 = res.getDouble ("la_deb");
    			double loFin				 = res.getDouble ("lo_fin");
    			double laFin				 = res.getDouble ("la_fin");
    			
	              
    			// Créer un nouvel objet si nécessaire
    			if (fleuveCourant == null || idFleuve != idFleuveCourant)
    			{
    				idFleuveCourant     = idFleuve ;
    				lignesFleuveCourant = new Limes() ;
    				fleuveCourant       = new Fleuve (idFleuve,
	                                                  nomFleuve,
	                                                  lignesFleuveCourant,
	                                                  new PointSphere (longitudeCentreFleuve,
	                                                                   latitudeCentreFleuve),
	                                                  cosCercleLimiteFleuve,
	                                                  new PointSphere(loDeb,laDeb),
	                                                  new PointSphere(loFin, laFin)) ;
    				fleuves.add(fleuveCourant);
    			}
            	//intégrer les points du limes dans le fleuve
    			//une liste de points
    			List<PointSphere> points = new ArrayList<>();
    			//on remplit cette liste avec les points du limes
    			points.addAll(indexLimes.get (idLimes).points());
    			//si on parcourt dans le limes dans l'autre sens, on inverse la liste
    			if (!sensLimes) Collections.reverse(points);
    			//on ajoute les points à la liste courante
    			lignesFleuveCourant.points().addAll (points);
    		}
    		res.close() ;
    		
    		return fleuves;
        }
        catch (SQLException e)
        {
            e.printStackTrace();
            throw new RuntimeException ("Erreur lors du chargement des données", e) ;
        }
    }
    
    
}
