/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 * 
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou le modifier sous les
 * termes de la licence Affero GPL publiée par la Fondation pour le logiciel libre (Free
 * Software Foundation), en choisissant la version 3 de cette licence ou n'importe quelle
 * version ultérieure, à votre convenance.
 * 
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE D'AUCUNE SORTE
 * : y compris d'être vendable ou de pouvoir servir un but donné. Voir le texte de la
 * licence AGPL pour plus de détails.
 * 
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire. Si ce n'est pas le
 * cas, regardez à cette adresse : <http://www.gnu.org/licenses/>.
 * 
 */
package fr.histoiremondiale.histoire.utiles.math;

import static java.lang.Math.cos;
import static java.lang.Math.sin;
import static java.lang.Math.toRadians;


/**
 * Fonctions de calcul dans l'espace.<br>
 * Les calculs de projection sur un plan devraient se trouver ici également.
 */
public class CalculsEspace
{

	/**
	 * Renvoie le produit scalaire des vecteurs OA et OB.
	 * 
	 * @param o
	 *            Point dans l'espace.
	 * @param a
	 *            Point dans l'espace.
	 * @param b
	 *            Point dans l'espace.
	 * @return Le produit scalaire.
	 */
	public static double produitScalaire(PointEspace o, PointEspace a, PointEspace b)
	{
		return (a.x() - o.x()) * (b.x() - o.x()) + (a.y() - o.y()) * (b.y() - o.y()) + (a.z() - o.z())
				* (b.z() - o.z());
	}

	/**
	 * Renvoie le produit de la matrice par le point à trois dimensions.<br>
	 * Renvoie le résultat sous forme d'un nouveau point dans l'espace.
	 * 
	 * @param matrice
	 *            La matrice.
	 * @param ptEspace
	 *            Le point 3D.
	 * @return Le point résultat du calcul.
	 */
	public static PointEspace multiplier(Matrice matrice, PointEspace ptEspace)
	{
		// Calculer les coordonnées
		double x = matrice.val(1, 1) * ptEspace.x() + matrice.val(1, 2) * ptEspace.y() + matrice.val(1, 3)
				* ptEspace.z();
		double y = matrice.val(2, 1) * ptEspace.x() + matrice.val(2, 2) * ptEspace.y() + matrice.val(2, 3)
				* ptEspace.z();
		double z = matrice.val(3, 1) * ptEspace.x() + matrice.val(3, 2) * ptEspace.y() + matrice.val(3, 3)
				* ptEspace.z();

		// Ajuster les coordonnées
		// Commentaire de Patrice : normalement ce test n'est plus nécessaire : si
		// pNouvellesCoordonnees.X
		// est proche de 0, cela signifie que l'angle entre le point et le centre est
		// proche de 90°, or on
		// a supprimé les points au delà de 66° (cos(66°) = environ 0.55)
//		if (x < 0.00001)
//		{
//			x = 0.00001;
//		}

		// Renvoyer le nouveau point
		return new PointEspace(x, y, z);
	}

	/**
	 * Renvoie le point dans l'espace correspondant à un point en coordonnées sphériques.<br>
	 * On ne passe pas de rayon, les coordonnées sont entre -1 et +1.
	 * 
	 * @param ptSphere
	 *            Point en coordonnées sphériques.
	 * @return Le point en coordonnées cartésiennes.
	 */
	public static PointEspace ptEspaceDepuisPtSphere(PointSphere ptSphere)
	{
		double cosLon = cos(toRadians(ptSphere.longitude()));
		double cosLat = cos(toRadians(ptSphere.latitude()));
		double sinLon = sin(toRadians(ptSphere.longitude()));
		double sinLat = sin(toRadians(ptSphere.latitude()));
		return new PointEspace(cosLon * cosLat, sinLon * cosLat, sinLat);
	}

	/**
	 * Calcule la projection d'un point dans le plan carré de taille (côté) donné.
	 * 
	 * @param ptEspace
	 *            Le point à projeter.
	 * @param loupe
	 *            La valeur de la loupe.
	 * @param taillePlan
	 *            Côté du plan.
	 * @return Point projeté dans le plan.
	 */
	public static PointPlan projection(PointEspace ptEspace, double loupe, double taillePlan)
	{
		// Calculer les coefficients d'ajustement
		double angle = CoordSphere.ANGLE;
		double cosAngle = Math.cos(Math.toRadians(angle));
		double sinAngle = Math.sin(Math.toRadians(angle));
		double coeffX = (taillePlan * cosAngle) / (2 * sinAngle);
		double coeffY = coeffX;

		//si ptEspace.x() = 0 alors la droite [centre de la terre]-[ptEspace] est parallèle au plan
		//de projection et le projeté est rejeté à l'infini
		
		double x = ptEspace.x();
		if (x <= 0) x = 0.000001;
		// Calculer les coordonnées
		// (l'origine du plan est au centre de la carte)
		double coordX = coeffX * loupe * ptEspace.y() / x;
		double coordY = coeffY * loupe * ptEspace.z() / x;

		// Renvoyer le point
		return new PointPlan(coordX, coordY);
	}
}
