/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.utiles.math;



/**
 * Fonctions utilitaires de manipulation de plages de nombres.
 */
public class PlagesNombres
{

    /**
     * Applique un modulo "cyclique" (le résultat renvoyé se trouve dans [0 ; module[, y compris
     *   si nombre est négatif).
     * @param nombre Le nombre considéré.
     * @param module Le module.
     * @return Le nombre modulo le module.
     */
    public static int moduloCyclique (int nombre, int module)
    {
        // Vérifier les paramètres
        if (module <= 0)
            throw new IllegalArgumentException ("Le module devrait être strictement positif") ;
        
        // Ajuster le résultat à la plage donnée
        int res = nombre ;
        while (res < 0)
            res += module ;
        while (res >= module)
            res -= module ;
        
        // Renvoyer le résultat
        return res ;
    }
    
    /**
     * Applique un modulo "cyclique" (le résultat renvoyé se trouve dans [0 ; module[, y compris
     *   si nombre est négatif).
     * @param nombre Le nombre considéré.
     * @param module Le module.
     * @return Le nombre modulo le module.
     */
      public static double moduloCyclique (double nombre, double module)
    {
        // Vérifier les paramètres
        if (module <= 0)
            throw new IllegalArgumentException ("Le module devrait être strictement positif") ;
        
        // Ajuster le résultat à la plage donnée
        double res = nombre ;
        while (res < 0)
            res += module ;
        while (res >= module)
            res -= module ;
        
        // Renvoyer le résultat
        return res ;
    }
    
    
    /**
     * Centre un nombre dans une plage : on considère que la plage est cyclique (quand on ajoute 1
     *   à la valeur la plus haute de la plage, on se retrouve à la valeur la plus basse).
     * @param nombre Le nombre considéré.
     * @param valMin La valeur minimale de la plage.
     * @param valMax La valeur maximale de la plage.
     * @return Le nombre ramené dans la plage.
     */
    public static int dansPlageCyclique (int nombre, int valMin, int valMax)
    {
        // Vérifier les paramètres
        if (valMin >= valMax)
            throw new IllegalArgumentException ("La valeur minimal devrait être strictement inférieure à la valeur maximale") ;
        
        // Calculer le nombre de valeurs dans la plage
        int taillePlage = valMax - valMin + 1 ;
        
        // Ajuster le nombre
        return moduloCyclique (nombre - valMin, taillePlage) + valMin ;
    }
    
    /**
     * Centre un nombre dans une plage : on considère que la plage est cyclique (quand on ajoute 1
     *   à la valeur la plus haute de la plage, on se retrouve à la valeur la plus basse).
     * @param nombre Le nombre considéré.
     * @param valMin La valeur minimale de la plage.
     * @param valMax La valeur maximale de la plage.
     * @return Le nombre ramené dans la plage.
     */
     public static double dansPlageCyclique (double nombre, double valMin, double valMax)
    {
        // Vérifier les paramètres
        if (valMin >= valMax)
            throw new IllegalArgumentException ("La valeur minimal devrait être strictement inférieure à la valeur maximale") ;
        
        // Calculer le nombre de valeurs dans la plage
        double taillePlage = valMax - valMin + 1 ;
        
        // Ajuster le nombre
        return moduloCyclique (nombre - valMin, taillePlage) + valMin ;
    }
    
    
    /**
     * Fait entrer le nombre dans une plage donnée.<br>
     * Si le nombre est compris dans la plage (bornes incluses), il est renvoyé.<br>
     * Sinon la borne la plus proche du nombre est renvoyée.
     * @param nombre Le nombre considéré.
     * @param valMin La valeur minimale de la plage.
     * @param valMax La valeur maximale de la plage.
     * @return Le nombre ramené dans la plage.
     */
    public static int dansPlage (int nombre, int valMin, int valMax)
    {
        // Vérifier les paramètres
        if (valMin >= valMax)
            throw new IllegalArgumentException ("La valeur minimal devrait être strictement inférieure à la valeur maximale") ;
        
        // Ajuster le nombre
        return Math.max (valMin, Math.min (valMax, nombre)) ;
    }
    
    /**
     * Fait entrer le nombre dans une plage donnée.<br>
     * Si le nombre est compris dans la plage (bornes incluses), il est renvoyé.<br>
     * Sinon la borne la plus proche du nombre est renvoyée.
     * @param nombre Le nombre considéré.
     * @param valMin La valeur minimale de la plage.
     * @param valMax La valeur maximale de la plage.
     * @return Le nombre ramené dans la plage.
     */
     public static double dansPlage (double nombre, double valMin, double valMax)
    {
        // Vérifier les paramètres
        if (valMin >= valMax)
            throw new IllegalArgumentException ("La valeur minimal devrait être strictement inférieure à la valeur maximale") ;
        
        // Ajuster le nombre
        return Math.max (valMin, Math.min (valMax, nombre)) ;
    }
    
}
