/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.utiles.verificationdonnees;

import static fr.histoiremondiale.histoire.ConfigAppli.CHEM_REP_PAGES_HTML;
import static fr.histoiremondiale.histoire.stockage.AccesDonneesIGraphique.ENCODAGE_PAGES_HTML;
import static fr.histoiremondiale.histoire.stockage.AccesDonneesIGraphique.NOM_REP_DONNEES_PARAGRAPHES;
import static fr.histoiremondiale.histoire.utiles.exttypes.Chaines.sousChaine;

import java.io.File ;
import java.io.IOException ;
import java.util.ArrayList ;
import java.util.Collections ;
import java.util.List ;

import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.utiles.es.fichiers.Fichiers;
import fr.histoiremondiale.histoire.utiles.exttypes.Chaines;



/**
 * Vérifie les liens hypertextes dans les paragraphes html de l'application (ceux qui décrivent les
 *   civilisations et les périodes).<br>
 * Outil permettant aux développeurs de vérifier que les liens ne sont pas cassés.
 */
public class VerificateurLiensParagraphes
{

    
    /**
     * Vérifie les liens entre les paragraphes et vers les autres pages du projet
     * @throws IOException En cas de problème d'entrée/sortie.
     */
    public void verifierLiens () throws IOException
    {
        String debLien       = "<a href=\"" ;            // Début des balises représentant les liens à rechercher
        String marqueurLien  = "href" ;                  // Elément marquant la présence d'un lien
        
        List<String> erreurs = new ArrayList<>() ;         // Erreurs détectées

        File repHtml         = new File ("donnees" + File.separator + CHEM_REP_PAGES_HTML    + File.separator) ;
        File repParagraphes  = new File ("donnees" + File.separator + NOM_REP_DONNEES_PARAGRAPHES + File.separator) ;
        
        
        // Lister les fichiers des paragraphes
        System.out.println ("Vérification des fichiers dans le répertoire " + repParagraphes.getAbsolutePath()) ;
        List<String> chemFicsParagraphes = new ArrayList<>() ;
        for (File fic : repParagraphes.listFiles())
            chemFicsParagraphes.add (fic.getAbsolutePath()) ;
        Collections.sort (chemFicsParagraphes) ;
        
        
        // Charger les données de l'appli
        HistoireMondiale application = HistoireMondiale.instance() ;
        

        // Lire les paragraphes
        for (String chemFic : chemFicsParagraphes)
        {
            File fic = new File (chemFic) ;
            if (fic.isFile())
            {
                System.out.println ("Traitement du fichier " + fic.getAbsolutePath() + "...") ;
                String txtParagraphe = Fichiers.chargerContenuTexte (fic.getAbsolutePath(), ENCODAGE_PAGES_HTML) ;
                // Chercher les liens dans le code html
                int posLien = -1 ;
                int cptLiensTraites    = 0 ;
                while ((posLien = txtParagraphe.indexOf (debLien, posLien + 1)) != -1)
                {
                    // Récupérer l'adresse indiquée par le lien
                    int posDebAdresse = posLien + debLien.length() ;
                    int posFinAdresse = txtParagraphe.indexOf ("\"", posDebAdresse) ;
                    String adresse = sousChaine (txtParagraphe, posDebAdresse, posFinAdresse) ;
                    System.out.println ("\t" + adresse) ;
                    
                    // Lien vers un paragraphe
                    if (adresse.startsWith ("#"))
                    {
                        String ancreParagraphe = Chaines.sousChaine(adresse,1).replace ("&amp;", "&") ;
                        if (application.donneesIGraphique().paragrapheParAncre (ancreParagraphe) != null)
                        {
                            System.out.println ("\t\tLe paragraphe \"" + ancreParagraphe + "\" existe.") ;
                        }
                        else
                        {
                            System.out.println ("\t\tLe paragraphe \"" + ancreParagraphe + "\" n'existe pas.") ;
                            erreurs.add ("Le paragraphe \"" + ancreParagraphe + "\" n'existe pas.") ;
                        }
                    }
                    // Lien vers une autre page
                    else
                    {
                        // Vérifier que la page existe
                        String  nomPage     = Chaines.avantMarqueur (adresse, "#") ;
                        String  chemFicPage = repHtml.getAbsolutePath() + File.separator + nomPage ;
                        boolean pageExiste  = new File (chemFicPage).exists() ;
                        if (pageExiste)
                        {
                            System.out.println ("\t\tLa page \"" + nomPage + "\" existe.") ;
                        }
                        else
                        {
                            System.out.println ("\t\tLa page " + nomPage + " n'existe pas.") ;
                            erreurs.add (fic.getAbsolutePath() + " : la page \"" + nomPage + "\" n'existe pas") ;
                        }
                        
                        // Vérifier que la balise existe dans la page
                        if (pageExiste)
                        {
                            String ancre   = Chaines.apresMarqueur (adresse, "#") ;
                            String txtPage = Fichiers.chargerContenuTexte (chemFicPage, ENCODAGE_PAGES_HTML) ;
                            if (txtPage.contains ("name='"+ancre+"'>"))
                            {
                                System.out.println ("\t\tL'ancre \"" + ancre + "\" existe.") ;
                            }
                            else
                            {
                                System.out.println ("\t\tL'ancre \"" + ancre + "\" n'existe pas.") ;
                                erreurs.add (fic.getAbsolutePath() + " : l'ancre \"" + ancre + "\" n'existe pas dans la page " + nomPage) ;
                            }
                        }
                    }
                    
                    // Compter les liens traités
                    cptLiensTraites++ ;
                }
                
                // Compter les liens du fichier, avec une méthode un peu plus fiable
                // (compter les "href" dans le texte en minuscules)
                String txtParagrapheVerif = txtParagraphe.toLowerCase() ;
                int posComptage   = -1 ;
                int cptLiensComptes = 0 ;
                while ((posComptage = txtParagrapheVerif.indexOf (marqueurLien, posComptage + 1)) != -1)
                    cptLiensComptes++ ;
                
                // Vérifier que le nombre de liens traités est égal au nombre de liens comptés par ailleurs
                System.out.println ("\tNombre de liens traités : " + cptLiensTraites) ;
                System.out.println ("\tNombre de liens comptés : " + cptLiensComptes) ;
                if (cptLiensComptes != cptLiensTraites)
                    erreurs.add (fic.getAbsolutePath() + " : nombres de liens traités / comptés différents : " + cptLiensComptes + " / " + cptLiensComptes) ;
            }
        }
        
        // Afficher le rapport
        System.out.println() ;
        System.out.println ("Nombre d'erreurs : " + erreurs.size()) ;
        for (String erreur : erreurs)
            System.out.println ("\t" + erreur) ;
    }
    
    
    
    // Méthode principale
    public static void main (String[] args) throws IOException
    {
        new VerificateurLiensParagraphes().verifierLiens() ;
    }
    
}
