/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire;

import java.awt.Font;
import java.awt.FontFormatException;
import java.awt.GraphicsEnvironment;
import java.io.File;
import java.io.IOException;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import fr.histoiremondiale.histoire.igraphique.DialAnnee ;
import fr.histoiremondiale.histoire.igraphique.DialCentre;
import fr.histoiremondiale.histoire.igraphique.DialCivilisations;
import fr.histoiremondiale.histoire.igraphique.DialLoupe;
import fr.histoiremondiale.histoire.igraphique.FenPrincipale;
import fr.histoiremondiale.histoire.igraphique.evenements.GestEvenementsHistoireMondiale;
import fr.histoiremondiale.histoire.stockage.AccesDonneesConfiguration;
import fr.histoiremondiale.histoire.stockage.AccesDonneesGeographiques;
import fr.histoiremondiale.histoire.stockage.AccesDonneesHistoriques;
import fr.histoiremondiale.histoire.stockage.AccesDonneesIGraphique;
import fr.histoiremondiale.histoire.utiles.exceptions.ExceptionConnexionBaseDonnees;
import fr.histoiremondiale.histoire.utiles.exceptions.Exceptions;



/**
 * Classe principale de l'application.
 */
public class HistoireMondiale
{

    // Constantes
    public static final String NUMERO_VERSION                 = "2.1" ;
    public static final String LICENCE_CODE_SOURCE            = "AGPLv3" ;
    public static final String LIEN_PAGE_LICENCE_CODE_SOURCE  = "http://www.gnu.org/licenses/translations.html" ;
    public static final String DATES_DROITS_COPIE_CODE_SOURCE = "2010-2020" ;
    public static final String LICENCE_DONNEES                = "Creative Commons Attribution Partage dans les mêmes conditions (CC-by-sa)" ;
    public static final String LIEN_PAGE_LICENCE_DONNEES      = "http://creativecommons.org/licenses/by-sa/3.0/deed.fr" ;
    public static final String DATES_DROITS_COPIE_DONNEES     = "2010-2020" ;
    public static final String LICENCE_ICONES                 = "Licence Art Libre" ;
    public static final String LIEN_PAGE_LICENCE_ICONES       = "http://artlibre.org/licence/lal" ;
    public static final String DATES_DROITS_COPIE_ICONES      = "2011-2020" ;
    public static final String URL_SITE_WEB                   = "http://histoiremondiale.net/" ;
    public static final String URL_PROJET                     = "http://sourceforge.net/projects/histmondiale/" ;
//    public static final String URL_SITE_GRAPHISTE             = "http://supremempm.fr/" ;     // (le site n'existe plus)
    public static final String ADRESSE_CONTACT_HISTOIRE       = "histoire.mondiale@gmail.com" ;

    
    // Instance de l'application
    private static HistoireMondiale instance ;
    
    
    // Etat modifiable de l'application
    private ConfigAppli config ;                        // Configuration générale de l'appli
    private EtatAppli   etat ;                          // Données modifiables par l'utilisateur

    // Données générales de l'application
    private DonneesGeographiques carte ;                    // Données de la carte du monde
    private DonneesHistoriques   donneesHisto ;             // Données historiques
    private DonneesIGraphique    donneesIGraphique ;        // Données pour l'interface graphique
    

    
    // Fenêtres et boîtes de dialogues
    private GestEvenementsHistoireMondiale gestEvenements ;     // Gestionnaire d'événements de l'interface graphique
    private FenPrincipale                  fenPrincipale ;
    private DialCivilisations              dialCivilisations ;
    private DialLoupe                      dialLoupe ;
    private DialAnnee                      dialAnnee ;
	private DialCentre 					   dialCentre;

    
    
    /**
     * Renvoie l'instance unique de l'application.
     * @return L'instance de l'application.
     */
    public synchronized static HistoireMondiale instance ()
    {
        if (instance == null)
        {
            try
            {
                instance = new HistoireMondiale() ;
            }
            catch (RuntimeException e)
            {
                // (préserver le type de cette exception (ne pas l'emballer), ça peut ête utile)
                throw e ;
            }
            catch (Throwable e)
            {
                throw new RuntimeException ("Erreur lors de la création de l'application", e) ;
            }
        }
        return instance ;
    }
    
    
    
    /**
     * Constructeur privé (pas de constructeur public).
     */
    private HistoireMondiale ()
    {
    	System.out.println("Initialisation du programme");
    	long debut = System.currentTimeMillis();

    	// Charger la configuration générale de l'application
        // (une première fois uniquement pour récupérer le chemin d'accès aux données)
    	// la configuration est un fichier texte en UTF-8 que l'on trouve dans le répertoire 
        // System.getProperty ("user.home") + File.separator + ".config/histoiremondiale/histoire.conf".replace ('/', File.separatorChar),
    	// c'est à dire qu'il dépend du système d'exploitation
    	// Sous Windows 10 avec l'utilisateur Patrice il s'agit de C:\Utilisateurs\Patrice\.config\histoiremondiale 
    	// ce répertoire contient 3 fichiers 
    	// histoire.conf donne les paramètres de dessin, l'année et le point central de la carte lors de la fermeture du programme
    	// loupes.conf spécifie pour chaque région historique la valeur de la loupe pour l'affichage du texte (si modifiée)
    	// positions.conf spécifie pour chaque région l'endroit où s'affiche le texte correspondant (si modifié)
    	// Par contre aucun de ces trois fichiers ne donne la localisation des données (la base de données histoire.mv.db), celle-ci
    	// se trouve dans le répertoire /donnees/ du répertoire contenant l'exécutable
//    	System.out.println("\tDébut du chargement de l'accès à la configuration");
    	long dateDeb = System.currentTimeMillis();
        AccesDonneesConfiguration accesConfig ;
        try
        {
            accesConfig = new AccesDonneesConfiguration() ;
            this.config = accesConfig.chargerConfigGenerale() ;
        }
        catch (Throwable e)
        {
            throw new RuntimeException ("Erreur lors du chargement de la configuration générale de l'application.\n"
            		+ "Vous pouvez essayer de détruire le fichier de configuration générale (vos modifications seront perdues).", e) ;
        }
    	long dateFin = System.currentTimeMillis();
    	System.out.println("\tfin du chargement de l'accès à la configuration en : " + (dateFin-dateDeb) + " ms");

    	//ajouter la police Ecolier
//    	System.out.println("\tDébut du chargement de la fonte écolier");
    	dateDeb = System.currentTimeMillis();
        File fonte = null;
	    GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
		try 
		{
		     String police = this.config.chemRepDonnees() + File.separatorChar + "histoire" + File.separatorChar + "ec.TTF";
		     fonte = new File(police);
		     ge.registerFont(Font.createFont(Font.TRUETYPE_FONT, fonte));
		} 
		catch (IOException|FontFormatException e) 
		{
			throw new RuntimeException ("Erreur lors du chargement de la fonte écolier (ec.TTF).\n"
					+ "Elle devrait se trouver dans le répertoire " + config.chemRepDonnees() +".", e) ;
		}
    	dateFin = System.currentTimeMillis();
    	System.out.println("\tfin du chargement de la fonte écolier : " + (dateFin-dateDeb) + " ms");

        // Initialiser les données géographiques et historiques
    	try
        {
//        	System.out.println("\tDébut de l'initialisation des données géographiques");
        	dateDeb = System.currentTimeMillis();
            this.carte        = new DonneesGeographiques (new AccesDonneesGeographiques (this.config.chemRepDonnees())) ;
        	dateFin = System.currentTimeMillis();
        	System.out.println("\tfin de l'initialisation des données géographiques en : " + (dateFin-dateDeb) + " ms");
             
//        	System.out.println("\tDébut de l'initialisation des données historiques");
        	dateDeb = System.currentTimeMillis();
            this.donneesHisto = new DonneesHistoriques   (new AccesDonneesHistoriques   (this.config.chemRepDonnees()),
                    																	 this.carte.indexLimes()) ;            		
            dateFin = System.currentTimeMillis();
        	System.out.println("\tfin de l'initialisation des données historiques en : " + (dateFin-dateDeb) + " ms");
        }
        catch (Throwable e)
        {
            throw new RuntimeException ("Erreur lors du chargement des données géographiques et historiques.", e) ;
        }
        
        // Initialiser les données pour l'interface graphique
//    	System.out.println("\tDébut de l'initialisation des données de l'interface graphique");
    	dateDeb = System.currentTimeMillis();
        try
        {
            this.donneesIGraphique = new DonneesIGraphique (new AccesDonneesIGraphique (this.config.chemRepDonnees(),
                                                                                        this.config.chemRepIcones()),
                                                            this.donneesHisto()) ;
        }
        catch (Throwable e)
        {
            throw new RuntimeException ("Erreur lors du chargement des données pour l'interface graphique.", e) ;
        }
    	dateFin = System.currentTimeMillis();
    	System.out.println("\tfin de l'initialisation des données de l'interface graphique en : " + (dateFin-dateDeb) + " ms");
        
        // Charger l'état de l'application
//    	System.out.println("\tDébut du chargement de l'état de l'application");
    	dateDeb = System.currentTimeMillis();
    	
        try
        {
            this.etat = accesConfig.chargerEtatAppli (this.donneesHisto) ;
        }
        catch (Throwable e)
        {
            throw new RuntimeException ("Erreur lors du chargement de l'état de l'application.\n"
            		+ "Vous pouvez essayer de détruire le fichier de configuration générale (vos modifications seront perdues).", e) ;
        }
    	dateFin = System.currentTimeMillis();
//    	System.out.println("\tfin du chargement de l'état de l'application en : " + (dateFin-dateDeb) + " ms.");
    	
    	System.out.println("Fin de l'initialisation du programme en : " + (dateFin-debut) + " ms.");
    }
    
    
    /**
     * Renvoie le gestionnaire d'événements de l'application.<br>
     * Note : Comme pour la fenêtre principale, il a besoin d'une instance de l'application pour
     *        être créé et ne peut donc pas être créé dans le constructeur.
     * @return Le gestionnaire d'événements.
     */
    public GestEvenementsHistoireMondiale gestEvenements ()
    {
        if (this.gestEvenements == null)
            throw new IllegalStateException ("Le gestionnaire d'événements n'est pas initialisé") ;
        return this.gestEvenements ;
    }
    
    
    
    /**
     * Affiche l'interface graphique de l'application.
     * @throws Exception en cas d'erreur
     */
    public void afficherInterface () throws Exception
    {
    	System.out.println("Affichage de l'interface");
    	long debut = System.currentTimeMillis();
    	long dateDeb, dateFin;
    	
        // Créer la fenêtre si besoin
        // Note : Il ne faut pas la créer dans le constructeur de HistoireMondiale : l'interface graphique
        //        a besoin que l'application soit complètement initialisée pour être créée (récupération
        //        de l'instance de l'appli pour accéder aux données chargées)
        //          => Une fois le chargement différé complètement géré, on n'aura peut-être plus ce problème.
        if (this.fenPrincipale == null)
        {
            dateDeb = System.currentTimeMillis();
            SwingUtilities.invokeAndWait (new Runnable () {public void run ()
            {
                fenPrincipale  = new FenPrincipale() ;
                fenPrincipale.setVisible (true) ;
            }}) ;
            dateFin = System.currentTimeMillis();
            System.out.println("\tl'interface graphique est construite et rendue visible en : " + (dateFin - dateDeb) + " ms.");

            // (un petit délai permet aux composants de s'ajuster correctement avant d'afficher la carte, qui
            //  ne devrait plus s'afficher en petit puis en grand, et le séparateur carte/navigateur, qui
            //  devrait s'afficher à la bonne place)
            
            dateDeb = System.currentTimeMillis();
            try { Thread.sleep (100) ; } catch (Exception e) {}
            this.gestEvenements = new GestEvenementsHistoireMondiale() ;
            dateFin = System.currentTimeMillis();
            System.out.println("\tle gestionnaire des évènements est construit en : " + (dateFin - dateDeb) + " ms."); 
            
//            dateDeb = System.currentTimeMillis();
            // (une partie de l'initialisation doit se faire dans le fil d'exécution de Swing, pour éviter
            //  les erreurs ; tout finalement, le début aussi)
            SwingUtilities.invokeLater (new Runnable () {public void run ()
            {
                fenPrincipale.finirInitialisation() ;
                gestEvenements().declarerCapturesEvenementsFenPrincipale (fenPrincipale) ;
            }}) ;
            dateFin = System.currentTimeMillis();
//            System.out.println("\tfin de l'initialisation en : " + (dateFin - dateDeb) + " ms."); 
        }
        else
        {
            // (si on l'appelle alors qu'elle est déjà visible, ça l'amène devant les autres fenêtres)
            this.fenPrincipale.setVisible (true) ;
        }
    	dateFin = System.currentTimeMillis();
    	System.out.println("fin de l'affichage de l'interface en : " + (dateFin - debut) + " ms.");
        
    }
    
    // Accesseurs
    public ConfigAppli          config            () { return this.config ; }
    public EtatAppli            etat              () { return this.etat ; }
    public DonneesGeographiques carte             () { return this.carte ; }
    public DonneesHistoriques   donneesHisto      () { return this.donneesHisto ; }
    public DonneesIGraphique    donneesIGraphique () { return this.donneesIGraphique ; }
    public FenPrincipale        fenPrincipale     () { return this.fenPrincipale ; }
    
    public DialCivilisations dialCivilisations () 
    { 
        if (this.dialCivilisations == null)
            this.dialCivilisations = new DialCivilisations (this.fenPrincipale) ;
    	return this.dialCivilisations ; 
    }
    
    public DialLoupe dialLoupe () 
    { 
        if (this.dialLoupe == null)
        {
        	this.dialLoupe = new DialLoupe (this.fenPrincipale, "Loupe", EtatAppli.MINI_LOUPE,  EtatAppli.MAXI_LOUPE) ;
            gestEvenements().declarerCapturesEvenementsDialLoupe(this.dialLoupe) ;
        }
    	return this.dialLoupe ; 
    }
    
    public DialAnnee dialAnnee () 
    { 
        if (this.dialAnnee == null)
        {
            this.dialAnnee = new DialAnnee (this.fenPrincipale) ;
            gestEvenements().declarerCapturesEvenementsDialAnnee(this.dialAnnee) ;
        }
        return this.dialAnnee ; 
    }
    
    public DialCentre dialCentre() 
    {
        if (this.dialCentre == null)
        {
            this.dialCentre = new DialCentre (this.fenPrincipale, "Centre") ;
            gestEvenements().declarerCapturesEvenementsDialCentre(this.dialCentre) ;
        }
        return this.dialCentre ; 
    }
    


    /**
     * Méthode principale.
     * @param args Les arguments passés dans la ligne de commande.
     */
    public static void main (String[] args)
    {
    	//l'application
    	try
        {
            // Créer l'application
            HistoireMondiale application = HistoireMondiale.instance() ;
            
            // Lancer l'interface graphique
            application.afficherInterface() ;
        }
        catch (ExceptionConnexionBaseDonnees e)
        {
            e.printStackTrace() ;
            JOptionPane.showMessageDialog (null, "Erreur de connexion à la base de données. Vérifiez que la base n'est pas en cours d'utilisation (par exemple que le logiciel n'est pas déjà lancé) : \n" + Exceptions.extraireMessagesErreur (e), "Erreur lors du lancement de l'application", JOptionPane.ERROR_MESSAGE) ;
        }
        catch (Throwable e)
        {
            e.printStackTrace() ;
            JOptionPane.showMessageDialog (null,
                                           "Erreur lors du lancement de l'application.\nN'hésitez pas à contacter les auteurs : " + ADRESSE_CONTACT_HISTOIRE + "\n\n" +
                                             "Erreur : \n" + Exceptions.extraireMessagesErreur (e),
                                           "Erreur lors du lancement de l'application.", JOptionPane.ERROR_MESSAGE) ;
        }
    }

}
