/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique;

import static fr.histoiremondiale.histoire.utiles.igraphique.swing.gestplacement.PlaceurListe.ALIGN_CENTRE;
import static fr.histoiremondiale.histoire.utiles.igraphique.swing.gestplacement.PlaceurListe.ALIGN_GAUCHE;
import static fr.histoiremondiale.histoire.utiles.igraphique.swing.gestplacement.PlaceurListe.LIGNE;

import java.awt.BorderLayout ;
import java.awt.Color ;
import java.awt.Component ;
import java.awt.Dimension ;
import java.awt.Font ;
import java.awt.GridLayout ;
import java.awt.Point ;
import java.awt.event.ComponentEvent ;
import java.awt.event.ComponentListener ;
import java.util.ArrayList ;
import java.util.List ;
import java.util.Observable ;
import java.util.Observer ;

import javax.swing.Box ;
import javax.swing.BoxLayout ;
import javax.swing.JCheckBox ;
import javax.swing.JLabel ;
import javax.swing.JPanel ;
import javax.swing.JScrollPane ;
import javax.swing.JViewport ;
import javax.swing.SwingUtilities ;
import javax.swing.event.ChangeEvent ;
import javax.swing.event.ChangeListener ;
import javax.swing.text.BadLocationException ;

import fr.histoiremondiale.histoire.DonneesIGraphique;
import fr.histoiremondiale.histoire.EtatAppli;
import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.donnees.Civilisation;
import fr.histoiremondiale.histoire.igraphique.composants.CaseACocher;
import fr.histoiremondiale.histoire.igraphique.composants.Texte;
import fr.histoiremondiale.histoire.igraphique.donnees.InfosParagrapheNavig;
import fr.histoiremondiale.histoire.utiles.igraphique.swing.gestplacement.PlaceurListe;



/**
 * Panneau pour l'affichage du navigateur HTML.
 */
public class PanNavigateur extends JPanel implements ChangeListener, ComponentListener, Observer
{
    
    // Constantes
    private static final int HAUTEUR_LIGNE = 13 ;
    
    
    public  JScrollPane           vuedefil_paragraphes ;        // Vue défilante pour l'affichage des paragraphes
    public  JCheckBox             cc_suiviCivilisation ;        // Coche permettant d'activer/désactiver le suivi de civilisation
    private Box                   p_paragraphes ;               // Panneau permettant d'afficher les paragraphes html

    private List<ParagrapheNavig> paragraphesEnCreation ;       // Paragraphes affichables dans le navigateur (en cours de création)
    private List<ParagrapheNavig> paragraphes ;                 // Paragraphes affichables dans le navigateur
    
    private InfosParagrapheNavig demandeAffParagrapheEnAttente ;    // Demande d'affichage d'un paragraphe en attente (que les paragraphes soient affichés)
    private boolean marqueurPremApplicationAffParagraphe = true ;   // Indique qu'on n'a pas encore demandé l'application de l'affichage du paragraphe en attente et que la prochaine sera la première (le problème est ici que les paragraphes s'affichent en deux fois et qu'il faut refaire la demande quand elle est traitée la première fois)
    private boolean marqueurDeuxApplicationAffParagraphe = true ;   //   (le paragraphe n'est pas bien ajusté si on ne le demande pas une deuxième fois)
    private boolean reactionChangementAnneeDesactivee ;             // Permet de désactiver le centrage sur la civilisation quand l'année est modifiée
    
    // (réaffichage du même paragraphe après un redimensionnement du navigateur)
    private ParagrapheNavig parAfficheAvantRedim ;          // Paragraphe affiché avant le redimensionnement
    private int             posDansParAffiche = -1 ;        // Position dans le paragraphe affiché avant redimensionnement (indice du curseur dans le texte)
    private int             cptRedim ;                      // Nombre de redimensionnement restants avant réaffichage du paragraphe

    // Suivi du temps d'affichage
    private Long dateDebCreationParagraphes  = null ;
    private long tempsRecupInfosCreation     = 0 ;
    private long tempsConsommeCreation       = 0 ;


    
    /**
     * Constructeur.
     */
    public PanNavigateur ()
    {
//    	long debut = System.currentTimeMillis();
        EtatAppli etatAppli = HistoireMondiale.instance().etat() ;
        
        
        // Panneau d'activation/désactivation du suivi de la civilisation
        JPanel p_civilisation = new JPanel (new PlaceurListe (LIGNE, ALIGN_GAUCHE + ALIGN_CENTRE)) ;
        this.cc_suiviCivilisation = new CaseACocher (etatAppli, "suiviCivilisationSelectionnee") ;
        p_civilisation.add (this.cc_suiviCivilisation) ;
        String ttt = "<html><body>"
                    + "<div style=\"text-align : center\">"
                    + "Si cette case est cochée, le navigateur suit la civilisation sélectionnée quand l'année change."
                    + "</div></body></html>";
        this.cc_suiviCivilisation.setToolTipText(ttt);
        p_civilisation.add (new TexteCivilisationSelectionnee ("")) ;

        // Panneau des paragraphes html
        this.p_paragraphes = new PanParagraphes (BoxLayout.Y_AXIS) ;
        JPanel p_paragraphesExterne = new JPanel (new GridLayout (1, 1)) ;
        this.vuedefil_paragraphes = new JScrollPane (this.p_paragraphes,
                                                     JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
                                                     JScrollPane.HORIZONTAL_SCROLLBAR_NEVER) ;
        p_paragraphesExterne.add (this.vuedefil_paragraphes) ;
        this.vuedefil_paragraphes.getViewport().addChangeListener (this) ;
        this.p_paragraphes.addComponentListener                   (this) ;
        this.vuedefil_paragraphes.getVerticalScrollBar().setUnitIncrement (HAUTEUR_LIGNE) ;
        // (message pour patienter en attendant le chargement des paragraphes)
        JLabel txt_Patienter = new JLabel ("Chargement des paragraphes en cours...") ;
        txt_Patienter.setFont (this.getFont().deriveFont (Font.ITALIC)) ;
        this.p_paragraphes.add (txt_Patienter) ;

        // Assemblage des composants
        this.setLayout (new BorderLayout()) ;
        this.add (p_paragraphesExterne, BorderLayout.CENTER) ;
        this.add (p_civilisation,       BorderLayout.SOUTH) ;

        // Créer les composants graphiques permettant d'afficher les paragraphes html
        this.paragraphesEnCreation = new ArrayList<>() ;
        this.paragraphes           = new ArrayList<>() ;
        // (demander à centrer sur la civilisation suivie ; à faire rapidement pour que ça ne s'exécute pas
        //  après une éventuelle demande de l'utilisateur de centrage sur un autre paragraphe)
        centrerSurCivilisationSuivie() ;
//        long dateDeb = System.currentTimeMillis();
        // (appel asynchrone même pour le premier appel de la méthode : le premier navigateur est lent à créer
        //  et le fait de faire un appel différé pour cette méthode permet de gagner 400-500 ms sur la durée
        //  avant affichage de la fenêtre [sur un Athlon 1800+])
        SwingUtilities.invokeLater (new Runnable () { public void run () {creerComposantsParagraphes() ;} }) ;
//        long dateFin = System.currentTimeMillis();
//        System.out.println("\t\tFin de la création des composants paragraphes en : " + (dateFin - dateDeb) + " ms.");
        // S'inscrire comme observateur de l'état de l'application
        HistoireMondiale.instance().etat().addObserver (this) ;
//        long dateFin = System.currentTimeMillis();
//        System.out.println("\t\tFin de la constrution du panneau navigateur en : " + (dateFin - debut) + " ms.");
    }
    
    
    // Accesseurs
    public List<ParagrapheNavig> paragraphes() { return this.paragraphes ; }
    

    
    // Crée les composants permettant d'afficher les paragraphes html
    private void creerComposantsParagraphes ()
    {
        if (this.dateDebCreationParagraphes == null)
            this.dateDebCreationParagraphes = System.currentTimeMillis() ;
        long dateDebRecupinfos = System.currentTimeMillis() ;
        
        // Récupérer les paragraphes
        List<InfosParagrapheNavig> paragraphes = HistoireMondiale.instance().donneesIGraphique().paragraphes() ;
        int nbParagraphes      = paragraphes.size() ;
//        for (int n = 0; n < nbParagraphes; n++)
//        {
//        	InfosParagrapheNavig paragraphe = paragraphes.get (n) ;
//        	ParagrapheNavig navig = new ParagrapheNavig (paragraphe) ;
//        	navig.setEditable      (false) ;    // (avant l'URL, pour éviter des blocages et plantages lors du chargement des pages (!).
//        	navig.setContentType   ("text/html") ;
//        	navig.setText          (paragraphe.codeHtml()) ;
//        	this.paragraphesEnCreation.add (navig) ;
//        }

        int nbParagraphesPrets = this.paragraphesEnCreation.size() ;
        
        long dateFinRecupinfos = System.currentTimeMillis() ;
        this.tempsRecupInfosCreation += (dateFinRecupinfos - dateDebRecupinfos) ;

//        while(nbParagraphesPrets < nbParagraphes)
        // Charger les paragraphes html
        if (nbParagraphesPrets < nbParagraphes)
        {
            long dateDebCreation = System.currentTimeMillis() ;
            
            InfosParagrapheNavig paragraphe = paragraphes.get (nbParagraphesPrets) ;
            ParagrapheNavig navig = new ParagrapheNavig (paragraphe) ;
            navig.setEditable      (false) ;    // (avant l'URL, pour éviter des blocages et plantages lors du chargement des pages (!).
            navig.setContentType   ("text/html") ;
            navig.setText          (paragraphe.codeHtml()) ;
            this.paragraphesEnCreation.add (navig) ;
            
            // Rappeler la méthode plus tard
            SwingUtilities.invokeLater (new Runnable () { public void run () {creerComposantsParagraphes() ;} }) ;
            
            long dateFinCreation = System.currentTimeMillis() ;
            this.tempsConsommeCreation += (dateFinCreation - dateDebCreation) ;
            nbParagraphesPrets++;
        }
        else
        {
            // Afficher le résumé des temps de chargement
            long dateFinCreationParagraphes = System.currentTimeMillis() ;
            System.out.println ("Temps de création des paragraphes                : " + (dateFinCreationParagraphes - this.dateDebCreationParagraphes) + " ms.") ;
            System.out.println ("Temps réellement consommé par la création        : " + this.tempsConsommeCreation + " ms.") ;
            System.out.println ("Temps de récupération des infos pour la création : " + this.tempsRecupInfosCreation + " ms.") ;
            
            // Ajouter les écouteurs dès que la fenêtre est prête
            for (ParagrapheNavig navig : this.paragraphesEnCreation)
                HistoireMondiale.instance().gestEvenements().declarerCapturesEvenementsParagrapheHtml (navig) ;
            
	        // Stocker les paragraphes dans la liste et lancer l'affichage
	        this.paragraphes = this.paragraphesEnCreation ;
	        this.p_paragraphes.removeAll() ;    // (retirer le message demandant de patienter)
	        ajusterTailleParagraphes() ;
	        ajouterParagraphes() ;
        }
    }
    
    
    // Ajoute les paragraphes au panneau
    private void ajouterParagraphes ()
    {
        // Ajouter les paragraphes au panneau
        for (ParagrapheNavig p : this.paragraphes())
            this.p_paragraphes.add (p) ;

        // Ajouter un composant invisible pour que le dernier paragraphe puisse être affiché en haut du
        //   panneau de navigation quand c'est nécessaire (plutôt qu'ajusté en bas du panneau parce que c'est
        //   le dernier composant).
        JPanel p_remplissage = new BouchonParagraphesNavig() ;
        p_remplissage.setPreferredSize (new Dimension ((int)this.p_paragraphes.getPreferredSize().getWidth(), 2000)) ;
        // (peindre le composant)
        int nbComposants = this.p_paragraphes.getComponentCount() ;
        p_remplissage.setBackground (nbComposants > 0 ?
                                        this.p_paragraphes.getComponent(nbComposants-1).getBackground() :
                                        Color.WHITE) ;
        this.p_paragraphes.add (p_remplissage) ;
        
        // Finaliser l'opération
        this.updateUI() ;
        
        // Appliquer l'éventuelle demande d'affichage en attente
        // (après l'affichage des paragraphes, sinon les affichages ne sont pas affichés complètement et
        //  la vue défilante ne s'ajuste pas)
        SwingUtilities.invokeLater (new Runnable () { public void run () {appliquerDemandeAffParagrapheEnAttente() ;} }) ;
    }
    

    // Gestion des changements d'état des composants
    // (on utilise stateChanged et pas ComponentListener.componentResized() parce que cette deuxième méthode
    //  ne fonctionne pas si on ne vire pas tous les paragraphes pour les réinsérer : la largeur des
    //  paragraphes reste identique quand on diminue la largeur du panneau et donc ils ne sont pas complètement
    //  affichés)
    public void stateChanged (ChangeEvent evt)
    {
        JViewport source = (JViewport) evt.getSource() ;
        if (source.getWidth() != this.p_paragraphes.getPreferredSize().getWidth())
        {
            // Noter la position du premier paragraphe affiché dans le navigateur
            // (note sur une tentative d'autre solution : non, on ne peut pas simplement dire : "on est au
            //  tiers du défilement, donc on va se repositionner au tiers après redimensionnement" : c'est
            //  trop imprécis et ça pose des problèmes quand la hauteur du navigateur diminue et que la position
            //  (valeur) du curseur est diminuée parce que le curseur est arrivé en bas de l'ascenseur)
            // (si le panneau contient des paragraphes)
            Component c = this.p_paragraphes.getComponentAt (this.vuedefil_paragraphes.getViewport().getViewPosition()) ;
            if (c instanceof ParagrapheNavig)
            {
                // Récupérer les coordonnées du point affiché en haut à gauche dans le navigateur
                Point posHG = this.vuedefil_paragraphes.getViewport().getViewPosition() ;
                
                // Récupérer le paragraphe affiché en haut dans le navigateur
                this.parAfficheAvantRedim = (ParagrapheNavig) this.p_paragraphes.getComponentAt (posHG) ;

                // Récupérer l'indice dans le texte du paragraphe du caractère affiché en haut à gauche du navigateur
                int xCar = (int) posHG.getX() - this.parAfficheAvantRedim.getX() ;
                int yCar = (int) posHG.getY() - this.parAfficheAvantRedim.getY() ;
                this.posDansParAffiche = this.parAfficheAvantRedim.viewToModel (new Point (xCar, yCar)) ;

                // Préparer le réaffichage
                // (le redimensionnement du navigateur donne lieu à 2 événements de redimensionnement et il
                //  faut traiter le dernier [sinon le deuxième modifie parfois le positionnement du navigateur
                //  qu'on vient de repositionner])
                this.cptRedim = 2 ;
            }
            // Ajuster la taille des paragraphes à la nouvelle largeur du navigateur
            ajusterTailleParagraphes() ;

        }
    }

    // Redimensionnement du panneau des paragraphes
    public void componentResized (ComponentEvent evt)
    {
        // Décompter les redimensionnements
        if (this.cptRedim > 0)
            this.cptRedim-- ;

        // S'il est temps de repositionner le navigateur
        if (this.parAfficheAvantRedim != null && this.cptRedim == 0)
        {
            // Récupérer les nouvelles coordonnées dans le paragraphe du point mémorisé
            Point ptDansPar ;
            try
            {
                ptDansPar = this.parAfficheAvantRedim.modelToView(this.posDansParAffiche).getLocation() ;
            }
            catch (BadLocationException e)
            {
                // (en cas d'erreur, laisser tomber)
                return ;
            }

            // Calculer les nouvelles coordonnées dans le panneau du point mémorisé
            Point ptDansPanneau = new Point ((int) ptDansPar.getX() + this.parAfficheAvantRedim.getX(),
                                             (int) ptDansPar.getY() + this.parAfficheAvantRedim.getY()) ;
            
            // Ajuster la vue sur la position mémorisée
            this.vuedefil_paragraphes.getViewport().setViewPosition (ptDansPanneau) ;
            
            // Terminer l'opération de réaffichage
            this.parAfficheAvantRedim = null ;
            this.posDansParAffiche    = -1 ;
        }
    }


    
    // Ajuste la taille de paragraphes (quand leur panneau est redimensionné).
    private void ajusterTailleParagraphes ()
    {
        if (! this.paragraphes.isEmpty())
        {
            // Si la taille a changé, il faut redimensionner les navigateurs
            // (par défaut ils ne sont pas contenus dans la zone de défilement)
            int largeur = this.vuedefil_paragraphes.getViewport().getWidth() ;
            // (hauteur des paragraphes pour spécifier la hauteur du panneau ;
            //  ça ne fonctionne pas en reprenant la hauteur de getPreferredSize())
            // Si la taille a changé, ajuster les composants
            for (Component c : this.p_paragraphes.getComponents())
            {
                if (c instanceof ParagrapheNavig)
                {
                    ParagrapheNavig p = (ParagrapheNavig) c ;
                    if (p.largeur() == null || ! p.largeur().equals (largeur))
                        p.modifLargeur (largeur) ;
                }
                else if (c instanceof BouchonParagraphesNavig)
                {
                    BouchonParagraphesNavig b = (BouchonParagraphesNavig) c ;
                    if (b.largeur() == null || ! b.largeur().equals (largeur))
                        b.modifLargeur (largeur) ;
                }
                else
                {
                    // (pour ne pas oublier d'ajouter d'éventuels autres composants à la liste, sinon
                    //  ça ne fonctionne plus parce que certains ne sont pas redimensionnés)
                    throw new IllegalStateException ("Redimensionnement - composant inconnu : " + c) ;
                }
            }
    
        }
        
    }
    
    
    // Affiche le paragraphe demandé dans le navigateur.
    // Si le paragraphe n'est pas trouvé, rien ne se passe.
    // Si les paragraphes ne sont pas encore chargés, la demande est mise en attente.
    public void afficherParagraphe (InfosParagrapheNavig paragraphe)
    {
        if (paragraphe == null)
            return ;

        // Inscrire la demande en attente
        this.demandeAffParagrapheEnAttente = paragraphe ;
        
        // Afficher le paragraphe si possible
        if (! this.paragraphes.isEmpty())
            appliquerDemandeAffParagrapheEnAttente() ;
    }
    
    
    // Affiche le paragraphe se trouvant dans la liste d'attente (paragraphe en attente d'être affiché).
    private void appliquerDemandeAffParagrapheEnAttente ()
    {
        // Une bidouille pour afficher correctement la demande d'affichage initiale (demandée avant l'affichage
        // des paragraphes) ; voir le commentaire de l'attribut marqueur.
        // (et il faut refaire une demande supplémentaire ; quelque chose ne doit pas être bien géré quelque
        //  part... un problème dans l'ordre de traitement de certains événements ?)
        if (this.marqueurPremApplicationAffParagraphe)
        {
            this.marqueurPremApplicationAffParagraphe = false ;
            this.marqueurDeuxApplicationAffParagraphe = true ;
            SwingUtilities.invokeLater (new Runnable () { public void run () {appliquerDemandeAffParagrapheEnAttente() ;} }) ;
            return ;
        }
        if (marqueurDeuxApplicationAffParagraphe)
        {
            SwingUtilities.invokeLater (new Runnable () { public void run () {appliquerDemandeAffParagrapheEnAttente() ;} }) ;
            marqueurDeuxApplicationAffParagraphe = false ;
            return ;
        }

        
        // Si une demande est en attente
        if (this.demandeAffParagrapheEnAttente != null)
        {
            InfosParagrapheNavig paragraphe = this.demandeAffParagrapheEnAttente ;
            
            // Chercher le paragraphe
            for (int i = 0 ; i < this.paragraphes.size() ; i++)
            {
                // Si on le trouve, mettre à jour l'affichage
                if (this.paragraphes.get(i).paragraphe().equals (paragraphe))
                {
                    Point position = ((ParagrapheNavig)this.p_paragraphes.getComponent(i)).getLocation() ;
                    this.vuedefil_paragraphes.getViewport().setViewPosition (position) ;
                    break ;
                }
            }
        }
    }
    
    
    // Centre le navigateur sur la civilisation suivie (si une civilisation est suivie)
    private void centrerSurCivilisationSuivie ()
    {
        HistoireMondiale application = HistoireMondiale.instance ();
        EtatAppli        etatAppli   = application.etat() ;
        
        // Si une civilisation est suivie
        Civilisation civilisationSuivie = etatAppli.civilisationSuivie() ;
        if (civilisationSuivie != null)
        {
            // Afficher le paragraphe correspondant
            DonneesIGraphique donneesIGraphique = application.donneesIGraphique() ;
            InfosParagrapheNavig paragraphe = donneesIGraphique.paragrapheParCriteres (civilisationSuivie,
                                                                                       etatAppli.annee()) ;
            if (paragraphe != null)
                this.afficherParagraphe (paragraphe) ;
        }
    }


    // Désactivation et réactivation de la réaction au changement d'année courante.
    // Il est parfois utile de désactiver temporairement le centrage automatique sur la civilisation
    //   suivie (par exemple quand on clique sur un lien : on change l'année courante pour
    //   centrer la carte, mais on ne veut pas que le navigateur s'ajuste en conséquence puisqu'on
    //   va le positionner ailleurs)
    public void desactiverReactionChangementAnnee () { this.reactionChangementAnneeDesactivee = true ; }
    public void reactiverReactionChangementAnnee  () { this.reactionChangementAnneeDesactivee = false ; }

    
    // Méthode de traitement des modifications de l'état de l'appli
    public void update (Observable o, Object params)
    {
        String nomAttribut = (String) params ;

        // Vérifier que la réaction n'a pas été désactivée
        if (! this.reactionChangementAnneeDesactivee)
        {
            // Si une information concernant la sélection / le suivi de civilisation change, modifier le paragraphe affiché
            if ("annee".equals                         (nomAttribut)
             || "civilisationSelectionnee".equals      (nomAttribut)
             || "suiviCivilisationSelectionnee".equals (nomAttribut))
            {
                centrerSurCivilisationSuivie() ;
            }
        }
    }



    // Méthodes non implémentées de ComponentListener
    public void componentMoved  (ComponentEvent evt) {}
    public void componentShown  (ComponentEvent evt) {}
    public void componentHidden (ComponentEvent evt) {}

}



// Panneau pour contenir les paragraphes
// (juste pour implémenter getPreferredSize simplement...)
class PanParagraphes extends Box
{
    public PanParagraphes (int axe)
    {
        super (axe) ;
    }
    
    public Dimension getPreferredSize ()
    {
        int largeur = 1 ;
        int hauteur = 1 ;
        for (Component c : this.getComponents())
        {
            Dimension taillePrefereeComposant = c.getPreferredSize() ;
            largeur =  Math.max (largeur, (int)taillePrefereeComposant.getWidth()) ;
            hauteur += (int) taillePrefereeComposant.getHeight() ;
        }
        return new Dimension (largeur, hauteur) ;
    }
}


// Bouchon permettant de laisser de la place après les paragraphes
class BouchonParagraphesNavig extends JPanel
{
    
    private Integer largeur ;           // Largeur imposée (null si aucune)
    
    
    
    // Méthode permettant de fixer la largeur du panneau.
    // Si une largeur a été spécifiée pour le panneau, elle est considérée comme la largeur à indiquer
    //   quand on demande la taille préférée du composant.
    public void modifLargeur (Integer largeur)
    {
        this.largeur = largeur ;
    }
    // (attention, peut renvoyer null, bien regarder ce que veut dire cette valeur)
    public Integer largeur ()
    {
        return this.largeur ;
    }
    public Dimension getPreferredSize ()
    {
        Dimension taillePreferee = super.getPreferredSize() ;
        return (this.largeur == null ?
                   taillePreferee :
                   new Dimension (this.largeur, (int)taillePreferee.getHeight())) ;
    }
    
}


// Texte désignant la civilisation sélectionnée
class TexteCivilisationSelectionnee extends Texte
{
    
    private String texte ;          // Texte à afficher (en plus du nom de la civilisation)
    
    
    
    // Constructeur
    public TexteCivilisationSelectionnee (String texte)
    {
        super ("", HistoireMondiale.instance().etat(), "civilisationSelectionnee") ;
        this.texte = texte ;
        
        // Représenter l'élément
        ajusterRepresentation() ;

        // Enregistrer le composant comme observateur de l'état de l'appli
        HistoireMondiale.instance().etat().addObserver (this) ;
    }
    

    
    // Affiche le texte en fonction de la civilisation sélectionnée
    public void ajusterRepresentation ()
    {
        String nomCivilisation = HistoireMondiale.instance().etat().nomCivilisationSelectionnee() ;
        String nomAAfficher    = (nomCivilisation == null ? "" : nomCivilisation) ;
        this.setText        (this.texte + nomAAfficher) ;
        this.setToolTipText (nomAAfficher) ;
    }
    
}
