/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique.actions;

import static fr.histoiremondiale.histoire.utiles.es.fichiers.Fichiers.urlLocaleDeNomFichier;

import java.awt.Point ;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.io.File ;
import java.io.IOException ;
import java.sql.ResultSet;
import java.sql.SQLException;

import javax.swing.JOptionPane ;
import javax.swing.text.AbstractDocument.LeafElement;
import javax.swing.text.AttributeSet ;
import javax.swing.text.Element ;
import javax.swing.text.html.HTML ;
import javax.swing.text.html.HTMLDocument ;

import fr.histoiremondiale.histoire.EtatAppli;
import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.igraphique.FenPageHTML;
import fr.histoiremondiale.histoire.igraphique.FenPrincipale;
import fr.histoiremondiale.histoire.igraphique.PanNavigateur;
import fr.histoiremondiale.histoire.igraphique.ParagrapheNavig;
import fr.histoiremondiale.histoire.igraphique.composants.DialogBiblio;
import fr.histoiremondiale.histoire.igraphique.donnees.InfosParagrapheNavig;
import fr.histoiremondiale.histoire.stockage.AccesDonneesBase;
import fr.histoiremondiale.histoire.utiles.exceptions.Exceptions;
import fr.histoiremondiale.histoire.utiles.math.PointSphere;



/**
 * Actions sur le navigateur web affichant les paragraphes.
 */
public class ActionsNavigateur
{

    /**
     * Affiche le paragraphe indiqué dans le navigateur.<br>
     * Rien ne se passe si le paragraphe passé en paramètre vaut null.
     * @param paragraphe Le paragraphe jusqu'auquel faire défiler le navigateur.
     */
    public static void afficherParagraphe (InfosParagrapheNavig paragraphe)
    {
        if (paragraphe != null)
            HistoireMondiale.instance().fenPrincipale().p_navigateur.afficherParagraphe (paragraphe) ;
    }
    
    
    /**
     * Modifie l'option "suivi de civilisation".
     * @param suiviCivilisationSelectionnee Nouvelle valeur pour le suivi.
     */
    public static void modifierSuiviCivilisation (boolean suiviCivilisationSelectionnee)
    {
        HistoireMondiale.instance().etat().modifSuiviCivilisationSelectionnee (suiviCivilisationSelectionnee) ;
    }
    /**
     * Inverse l'option "suivi de civilisation".
     */
    public static void inverserSuiviCivilisationSelectionnee ()
    {
        HistoireMondiale.instance().etat().inverserSuiviCivilisationSelectionnee() ;
    }
    
    
    /**
     * Traite un clic sur un paragraphe.
     * @param paragNavig Le paragraphe sur lequel l'utilisateur a cliqué.
     * @param posClic    La position à laquelle il a cliqué.
     */
    public static void naviguer (ParagrapheNavig paragNavig, Point posClic)
    {
        final String PREFIXE_PAGE_COURANTE  = "#" ;      // Début des liens vers un autre Paragraphe

        // Récupérer le navigateur
        HistoireMondiale application = HistoireMondiale.instance() ;
        FenPrincipale    fenetre     = application.fenPrincipale() ;
        PanNavigateur    p_navig     = fenetre.p_navigateur ;
        
        // Regarder si on a cliqué sur un lien
        HTMLDocument document     = (HTMLDocument) paragNavig.getDocument() ;
        int          posDoc       = paragNavig.viewToModel (posClic) ;
        Element      eltCaractere = document.getCharacterElement (posDoc) ;
        	
        AttributeSet attrs        = eltCaractere.getAttributes() ;
        AttributeSet lien         = (AttributeSet) attrs.getAttribute (HTML.Tag.A) ;
        
        // Si on a cliqué sur un lien
        if (lien != null)
        {
            String cibleLien = (String) lien.getAttribute (HTML.Attribute.HREF) ;
            // Lien vers un autre Paragraphe
            if (cibleLien.startsWith (PREFIXE_PAGE_COURANTE))
            {
                // Afficher le paragraphe
                String               ancreParagraphe = cibleLien.substring (PREFIXE_PAGE_COURANTE.length()) ;
                InfosParagrapheNavig paragraphe      = HistoireMondiale.instance().donneesIGraphique().paragrapheParAncre (ancreParagraphe) ;
                p_navig.afficherParagraphe (paragraphe) ;
                
                // Modifier l'année affichée par l'appli pour correspondre au nouveau paragraphe
                int ancienneAnnee = application.etat().annee() ;
                int nouvelleAnnee = paragraphe.ajusterAnneeDansParagraphe (ancienneAnnee) ;
                if (ancienneAnnee != nouvelleAnnee)
                {
                    // (modifier l'année sans que le navigateur ne s'ajuste sur la civilisation suivie)
                    p_navig.desactiverReactionChangementAnnee() ;
                    application.etat().modifAnnee(nouvelleAnnee) ;
//                    application.etat().modifPtCentralCarte(paragraphe.civilisation().ptCentral(nouvelleAnnee));
                    p_navig.reactiverReactionChangementAnnee() ;
                    
                }
            }
            // Lien vers une page externe
            else
            {
            	if (cibleLien.equals("bibliographie.html")) afficherBibliographie(paragNavig);
             	else
            	{
	                String chemPagesHtml = HistoireMondiale.instance().config().chemRepPagesHtml() ;
	                String urlPage       = urlLocaleDeNomFichier (chemPagesHtml + File.separator) + "/" + cibleLien ;
	                try
	                {
	                    new FenPageHTML ("", urlPage, 600, 400).setVisible (true) ;
	                }
	                catch (IOException e)
	                {
	                    e.printStackTrace() ;
	                    System.err.println (e.getMessage()) ;
	                    JOptionPane.showMessageDialog (null,
	                                                   "Impossible d'accéder au fichier désigné par ce lien\n" +
	                                                   "Fichier : " + urlPage + "\n" +
	                                                   "Message d'erreur : " + Exceptions.extraireMessagesErreur (e),
	                                                   "Erreur de suivi du lien",
	                                                   JOptionPane.ERROR_MESSAGE) ;
	                }
            	}
            }
        }
    }
    
    
    private static void afficherBibliographie(ParagrapheNavig paragNavig) 
    {
       	{
    		InfosParagrapheNavig paragraphe = paragNavig.paragraphe();
    		int id_par = paragraphe.id();
    		try 
    		{
				String biblio = AccesDonneesBase.obtenirBiblio(id_par);
				if (biblio != null)
				{
//					FenPageHTML fenetre = new FenPageHTML(biblio);
					DialogBiblio fenetre = new DialogBiblio(biblio, HistoireMondiale.instance().fenPrincipale());
					fenetre.setVisible (true) ;
				}
			} 
    		catch (SQLException e) 
    		{
				e.printStackTrace();
			}
    	}
	}

	/**
     * Afficher la vue correspondant au paragraphe donné.
     * @param paragraphe Le paragraphe concerné.
     */
    public static void affVueCorrespondante (InfosParagrapheNavig paragraphe)
    {
        HistoireMondiale application   = HistoireMondiale.instance() ;
        EtatAppli        etatAppli     = application.etat() ;
        FenPrincipale    fenPrincipale = application.fenPrincipale() ;
        PanNavigateur    panNavig      = fenPrincipale.p_navigateur ;
       
        
        // Traiter l'événement s'il y a une civilisation associée (ce qui n'est pas le cas des paragraphes spéciaux)
        if (paragraphe.civilisation() != null)
        {
            // Centrer la carte sur la civilisation concernée par le paragraphe
            PointSphere nouvPtCentralCarte = paragraphe.civilisation().ptCentral (etatAppli.annee()) ;
            int         nouvAnnee          = paragraphe.ajusterAnneeDansParagraphe      (etatAppli.annee()) ;
            etatAppli.modifCoordonneesEspaceTemps (nouvPtCentralCarte, nouvAnnee, true) ;
            
            // Afficher le paragraphe dans le navigateur
            panNavig.afficherParagraphe (paragraphe);
        
            // Modifier la civilisation sélectionnée
            etatAppli.modifCivilisationSelectionnee (paragraphe.civilisation()) ;
        }
        
    }
    
}
