/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique.tracercartes;

import java.awt.Color ;
import java.awt.Graphics ;
import java.awt.image.BufferedImage ;
import java.util.List ;
import java.util.concurrent.BlockingQueue ;
import java.util.concurrent.LinkedBlockingQueue ;

import fr.histoiremondiale.histoire.utiles.math.PointSphere;
import fr.histoiremondiale.histoire.DonneesIGraphique;
import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.donnees.Fleuve;
import fr.histoiremondiale.histoire.donnees.Mer;
import fr.histoiremondiale.histoire.donnees.Terre;
import fr.histoiremondiale.histoire.donnees.Territoire;
import fr.histoiremondiale.histoire.igraphique.PanCarte;



/**
 * Gère les demandes de tracer de cartes.<br>
 * Les demandes étant réalisées de manière asynchrone, elles sont stockées dans une file d'attente.
 */
public class GestTracerCartes implements Runnable
{

    private TraceurCarte       traceurCarte ;           // Outil permettant de dessiner des cartes
    private PanCarte           p_carte ;                // Panneau sur lequel les cartes doivent être tracées
    private InfosTracerImage   infosTraceImgActuelle ;  // Informations sur le tracé de l'image actuelle
    
    
    
    // Gestion des tracés asynchrones de carte
    private BlockingQueue<DemandeTracerCarte> demandesTrace ;           // Demandes de tracé de carte
    
    // Images en réserve (pour éviter de réallouer une image à chaque tracer de carte)
    private BufferedImage imgReserve         = null ;
    private int           largeurImgReserve  = -1 ;
    private int           hauteurImgReserve  = -1 ;
    private BufferedImage imgReserve2        = null ;
    private int           largeurImgReserve2 = -1 ;
    private int           hauteurImgReserve2 = -1 ;

    
    
    /**
     * Création d'un gestionnaire de tracers et lancement du fil de tracer des cartes demandées.
     * @param p_carte Panneau sur lequel tracer la carte.
     * @return Une instance de la classe.
     */
    public static GestTracerCartes creerInstance (PanCarte p_carte)
    {
        // Créer le traceur et lancer le fil d'exécution
        GestTracerCartes gestTracers = new GestTracerCartes (p_carte) ;
        Thread filExec = new Thread (gestTracers) ;
        filExec.setDaemon (true) ;
        filExec.start() ;
        
        // Renvoyer le gestionnaire de tracers
        return gestTracers ;
    }
    
    
    /**
     * Constructeur.
     * @param p_carte Panneau sur lequel tracer la carte.
     */
    private GestTracerCartes (PanCarte p_carte)
    {
        this.traceurCarte          = new TraceurCarte() ;
        this.p_carte               = p_carte ;
        this.demandesTrace         = new LinkedBlockingQueue<DemandeTracerCarte>() ;
        this.infosTraceImgActuelle = new InfosTracerImage() ;
    }
    
    
    
    /**
     * Exécution des traitements du fil d'exécution.
     */
    public void run ()
    {
        tracerCartesDemandees() ;
        System.err.println ("! Fin du fil d'exécution de tracé des cartes") ;
    }
    
    
    
    /**
     * Demande de création d'une nouvelle carte.<br>
     * La demande est mise dans une file d'attente pour être tracée à son tour sur le panneau.
     * @param annee                       Année choisie pour la carte.
     * @param fleuvesAffiches             Indique si les fleuves doivent être affichés.
     * @param meridiensParallelesAffiches Indique si les méridiens et parallèles doivent être affichés.
     * @param ptCentrePlan                Point à projeter au centre de la carte.
     * @param largeurCarte                Largeur de la carte.
     * @param hauteurCarte                Hauteur de la carte.
     * @param loupe                       Valeur de grossissement de la loupe.
     * @param coulMers                    Couleur des mers.
     * @param coulTerres                  Couleur des terres.
     * @param coulFleuves                 Couleur des fleuves.
     */
    public synchronized void demanderNouvelleCarte (int annee,
                                                    boolean fleuvesAffiches, boolean meridiensParallelesAffiches, 
                                                    PointSphere ptCentrePlan, int largeurCarte, int hauteurCarte, double loupe,
                                                    Color coulMers, Color coulTerres, Color coulFleuves)
    {
        // Créer la demande
        DemandeTracerCarte demande = new DemandeTracerCarte (annee,
                                                             fleuvesAffiches, meridiensParallelesAffiches, 
                                                             ptCentrePlan,
                                                             largeurCarte, hauteurCarte,
                                                             //max (largeurCarte, 1), max (hauteurCarte, 1),  // (erreur si on trace des cartes de dimensions négatives ou nulles ; traitée en amont en fait)
                                                             loupe,
                                                             coulMers, coulTerres, coulFleuves) ;
        
        // Actuellement une seule demande dans la file, les anciennes étant ignorées
        // (opérations atomiques, mais la méthode doit être synchronisée pour éviter l'ajout dans le
        //  désordre de cartes à tracer [mouais... enfin elles doivent toutes venir du fil d'exécution
        //  de Swing ces demandes, non ?])
        this.demandesTrace.clear() ;
        this.demandesTrace.add (demande) ;
    }
    
    
    /**
     * @return la prochaine carte à tracer (méthode bloquante).
     * @throws InterruptedException si la méthode est interrompu
     */
    private DemandeTracerCarte prochaineCarteATracer () throws InterruptedException
    {
        // (opération atomique et bloquante, donc pas de synchronisation : ça risquerait de bloquer tout le reste)
        return this.demandesTrace.take() ;
    }
    
    
    /**
     * Renvoie les informations sur le tracer de l'image.<br>
     * Renvoie toujours un objet, même s'il n'y a pas d'image actuellement tracée.
     * @return L'objet contenant les informations sur l'image tracée.
     */
    public synchronized InfosTracerImage infosImageActuelle ()
    {
        return this.infosTraceImgActuelle ;
    }
    
    /**
     * Modifie l'image affichée par le panneau : les informations sur l'image remplace celles sur l'ancienne.
     * @param infosTrace Informations sur la nouvelle image.
     */
    private synchronized void modifImageActuelle (InfosTracerImage infosTrace)
    {
        this.infosTraceImgActuelle = infosTrace ;
        this.infosTraceImgActuelle.terminerTracer() ;
    }
    
    /**
     * Indique que l'image actuelle est en cours de recréation.
     * @param demandeNouvCarte Demande à l'origine de la recréation.
     */
    private synchronized void imageActuelleEnModification (DemandeTracerCarte demandeNouvCarte)
    {
        InfosTracerImage infosImg = infosImageActuelle() ;
        if (infosImg.imageDisponible())
        {
            // g.fillOval (0, 0, 20, 20) ;  // (le coin en haut à gauche n'est pas toujours visible, par exemple lors de redimensionnements)
            // (l'image est "périmée" si la demande qui y a conduit n'est pas la même que celle en cours de tracer)
            //    => Sinon l'image est toujours utilisable telle quelle [pas dessiner de symbole "carte en cours de tracer", pour le déplacement des textes notamment]
            if (! demandeNouvCarte.equals (infosImg.ancImg.demande))
                infosImg.ancImg.perimee = true ;    // (difficile de compter le nombre d'image en cours de tracé, notamment à cause de la méthode de récupération de la prochaine demande dans la file d'attente, laquelle méthode n'est pas synchronisée avec le reste ; le plus smple (et le plus juste sans doute) reste de marquer l'image comme périmée quand une nouvelle est en cours de tracé)
            this.p_carte.repaint() ;
        }
        
    }
    
    
    /**
     * Trace les cartes qui se trouvent dans la file des demandes.
     */
    private void tracerCartesDemandees ()
    {

        // Continuer jusqu'à une interruption
        while (true)
        {
            
            // Gestion des erreurs lors du tracé de la carte
            try
            {
                // Récupérer une demande de tracé de la carte
                DemandeTracerCarte demandeNouvelleImage ;
                try
                {
                    demandeNouvelleImage = prochaineCarteATracer() ;
                }
                catch (InterruptedException e)
                {
                    System.out.println ("! Tracer des cartes : demande d'interruption du fil d'exécution") ;
                    break ;
                }
                
                
                // Récupérer les informations sur l'image actuelle
                // (note : pas de problème de synchronisation ici parce qu'on ne peut traiter qu'une image à la fois)
                InfosTracerImage   infosTraceImgActuelle = this.infosTraceImgActuelle ;
                DemandeTracerCarte demandeImgActuelle    = infosTraceImgActuelle.ancImg.demande ;
                DonneesIGraphique  donneesIGraphique     = HistoireMondiale.instance().donneesIGraphique() ;
            
                // Si la nouvelle demande correspond à la même chose que la précédente et que les
                //   données externes n'ont pas été modifiées, ne rien faire
                // (on peut utiliser demandeImgActuelle parce qu'on ne la modifie qu'ici)
                boolean donneesIGraphiqueModifiees = donneesIGraphique.donneesTraceCarteModifiees() ;   // (l'appel modifie une valeur, il faut tout le temps le faire)
                if (demandeNouvelleImage.largeurCarte > 0 &&
                    demandeNouvelleImage.hauteurCarte > 0 &&
                    (! demandeNouvelleImage.equals (demandeImgActuelle) || donneesIGraphiqueModifiees))
                {
                    Graphics g ;                    // Outil de tracé de la carte

                    
                    // Marquer l'image actuelle comme étant en cours de modification
                    imageActuelleEnModification (demandeNouvelleImage) ;
                    
                    // Créer une nouvelle image
                    BufferedImage nouvImage = (demandeNouvelleImage.largeurCarte == this.largeurImgReserve &&
                                               demandeNouvelleImage.hauteurCarte == this.hauteurImgReserve ?
                                                  this.imgReserve :
                                                  new BufferedImage (demandeNouvelleImage.largeurCarte,
                                                                     demandeNouvelleImage.hauteurCarte,
                                                                     BufferedImage.TYPE_INT_ARGB)) ;
                    g = nouvImage.getGraphics() ;
            
                    // Si la position ou la taille de la carte ont changé, réinitialiser les informations
                    if (demandeImgActuelle == null ||
                        demandeImgActuelle.largeurCarte != demandeNouvelleImage.largeurCarte ||
                        demandeImgActuelle.hauteurCarte != demandeNouvelleImage.hauteurCarte ||
                        demandeImgActuelle.loupe        != demandeNouvelleImage.loupe        ||
                        ! demandeImgActuelle.ptCentrePlan.equals (demandeNouvelleImage.ptCentrePlan))
                    {
                        infosTraceImgActuelle = new InfosTracerImage() ;
                    }
                    // Sinon commencer à remplir les infos sur la nouvelle image
                    else
                    {
                        infosTraceImgActuelle.nouvImg.cachePoints = infosTraceImgActuelle.ancImg.cachePoints ;
                        
                        // Décider quelles informations de l'ancienne image peuvent être récupérées
                        if (demandeImgActuelle.annee == demandeNouvelleImage.annee)
                        {
                            infosTraceImgActuelle.nouvImg.multiPolygonesPourtoursTraces = infosTraceImgActuelle.ancImg.multiPolygonesPourtoursTraces ;
                            infosTraceImgActuelle.nouvImg.territoiresTraces             = infosTraceImgActuelle.ancImg.territoiresTraces ;
                        }
                        // (si les deux demandes sont identiques, on peut récupérer l'image presque complète)
                        if (demandeNouvelleImage.equals (demandeImgActuelle))
                            infosTraceImgActuelle.nouvImg.imgCompleteSansTextes = infosTraceImgActuelle.ancImg.imgCompleteSansTextes ;
                        // (les textes sont toujours réaffichés)
                    }
                    
                    // Noter la demande associée à l'image
                    infosTraceImgActuelle.nouvImg.demande = demandeNouvelleImage ;
            
                    // Récupérer les données nécessaires au tracé
                    List<Terre>      terres             = HistoireMondiale.instance().carte().terres() ;
                    List<Mer>        mersInterieures    = HistoireMondiale.instance().carte().mers() ;
                    List<Fleuve>     fleuves            = HistoireMondiale.instance().carte().fleuves() ;
                    List<Territoire> territoiresATracer = HistoireMondiale.instance().donneesHisto().territoiresAnnee (demandeNouvelleImage.annee) ;
                    
                    // Tracer la carte
                    this.traceurCarte.tracerCarte (terres, mersInterieures, fleuves, territoiresATracer, demandeNouvelleImage.annee,
                                                   demandeNouvelleImage.fleuvesAffiches, demandeNouvelleImage.meridiensParallelesAffiches, demandeNouvelleImage.ptCentrePlan, demandeNouvelleImage.largeurCarte, demandeNouvelleImage.hauteurCarte,
                                                   demandeNouvelleImage.loupe, demandeNouvelleImage.coulMers, demandeNouvelleImage.coulTerres, demandeNouvelleImage.coulFleuves, g, infosTraceImgActuelle) ;
                    infosTraceImgActuelle.nouvImg.demande = demandeNouvelleImage ;
                    infosTraceImgActuelle.nouvImg.image   = nouvImage ;
                    modifImageActuelle (infosTraceImgActuelle) ;
                    
                    // Libérer le pinceau
                    g.dispose() ;

                    // Indiquer au panneau de se réafficher
                    this.p_carte.repaint() ;

                    // Décaler l'ancienne image et stocker l'image actuelle comme future ancienne image
                    this.imgReserve         = this.imgReserve2 ;
                    this.largeurImgReserve  = this.largeurImgReserve2 ;
                    this.hauteurImgReserve  = this.hauteurImgReserve2 ;
                    this.imgReserve2        = nouvImage ;
                    this.largeurImgReserve2 = demandeNouvelleImage.largeurCarte ;
                    this.hauteurImgReserve2 = demandeNouvelleImage.hauteurCarte ;
                }
            }
            catch (Throwable e)
            {
                // Afficher l'erreur dans la console
                System.err.println ("Erreur lors du tracer d'une carte") ;
                e.printStackTrace() ;
                
                // Afficher l'erreur dans l'interface
                try
                {
                    this.infosTraceImgActuelle.ancImg.erreur = e ;
                    this.p_carte.repaint() ;
                }
                catch (Throwable e2)
                {}
            }
            
        }
    }
    
}
