/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique.tracercartes;

import java.awt.Polygon ;
import java.awt.Rectangle ;
import java.util.List ;
import java.util.Map ;

import fr.histoiremondiale.histoire.utiles.math.PointPlan;
import fr.histoiremondiale.histoire.utiles.math.PointSphere;
import fr.histoiremondiale.histoire.donnees.Pourtour;
import fr.histoiremondiale.histoire.donnees.Territoire;
//import fr.histoiremondiale.histoire.utiles.math.PointPlan;
//import fr.histoiremondiale.histoire.utiles.math.PointSphere;



/**
 * Rassemble des informations sur le tracer d'une image.
 * Note : L'image actuelle (une fois le tracé terminé) est l'"ancienne" image.
 */
public class InfosTracerImage
{

    public InfosImage ancImg ;      // Informations éventuelles sur l'ancien tracé (informations réutilisables, pour gagner du temps)
    public InfosImage nouvImg ;     // Informations sur l'image en cours de tracé

    
    
    /**
     * Constructeur simple.
     */
    public InfosTracerImage ()
    {
        this.ancImg  = new InfosImage() ;
        this.nouvImg = new InfosImage() ;
    }
    
    
    /**
     * Indique si une image est disponible.
     * @return Vrai si une image est disponible, faux sinon.
     */
    public boolean imageDisponible ()
    {
        return this.ancImg.image != null ;
    }
    
    
    /**
     * Renvoie la liste de polygones représentant les contours du pourtour de numéro donné (si les polygones
     *   sont connus).<br>
     * Si aucune liste de polygones n'est trouvée, renvoie null.
     * @param idPourtour Identifiant du pourtour cherché.
     * @return La liste des polygones constituant le pourtour.
     */
    public List<Polygon> polygonesTraces (int idPourtour)
    {
        List<Polygon> multiPolygones ;
        // TODO Depuis la réécriture du code de tracer par Patrice, les polygones ne sont écrits dans la
        //      nouvelle liste (celle de nouvImg) que s'ils viennent d'être construits et pas s'il ont été
        //      trouvé dans le cache (même ancien puisque c'est transparent). Et donc certains territoires
        //      ne sont pas trouvés quand on fait un clic droit dessus.
        //        ! On pourrait modifier ce code (celui de tracer des terres et mers aussi , ça ne ferait pas
        //          de mal pour le fonctionnement du cache, même si ça n'entraîne pas un bug aussi gênant).
        //          Une bonne solution sans doute, avec un commentaire dans les méthodes appelantes indiquant
        //          qu'il faut systématiquement réécrire les données dans le cache. Mais puisque ça parle de
        //          problèmes de performances, refaire toutes ces écritures coûte peut-être quelque chose ?
        //        ! Ou alors décider que ce fonctionnement est trop complexe et garder ça comme ça ou rendre
        //          les choses plus explicites (par exemple taper dans l'ancien et le nouveau cache
        //          explicitement dans les méthodes concernées).
        // if ((multiPolygones = this.ancImg.multiPolygonesPourtoursTraces.get (idPourtour)) != null)
        //     return multiPolygones ;
        if ((multiPolygones = this.nouvImg.multiPolygonesPourtoursTraces.get (idPourtour)) != null)
            return multiPolygones ;
        return null ;
    }
    
    /**
     * Indique si le territoire se trouve dans les territoires tracés.
     * @param idTerritoire L'identifiant du territoire recherché.
     * @return Vrai si le territoire est sauvegardé et faux sinon.
     */
    public boolean territoireConnu(int idTerritoire)
    {
    	return this.nouvImg.territoiresTraces.containsKey (idTerritoire) ;
    }
    
    /**
     * Indique si le point du plan correspondant au point de la sphère en paramètre est connu.
     * @param ptSphere Point de la sphère considéré.
     * @return Vrai si le point projeté est connu, faux sinon.
     */
    public boolean pointPlanConnu (PointSphere ptSphere)
    {
        return this.nouvImg.cachePoints.containsKey (ptSphere) ;
    }
    
    /**
     * Renvoie le point du plan correspondant au point de la sphère en paramètre.<br>
     * Renvoie null si le point n'est pas trouvé.
     * @param ptSphere Point de la sphère considéré.
     * @return Le point projeté sur le plan correspondant.
     */
    public PointPlan pointPlan (PointSphere ptSphere)
    {
        return this.nouvImg.cachePoints.get (ptSphere) ;
    }
    
    
    /**
     * Indique le territoire qui se trouve aux coordonnées (x, y) sur la carte.<br>
     * Renvoie null s'il n'y a pas de territoire à cette position.
     * @param x L'abscisse du point de la carte.
     * @param y L'ordonnée du point de la carte.
     * @return Le territoire tracé aux coordonnées indiquées, null si aucun.
     */
    public Territoire territoireAuxCoordonnees (double x, double y)
    {
    	Territoire territoireDePlusHautRang = null ;       // Territoire de plus haut rang trouvé
    	
    	
        // Parcourir la collection des territoires tracés
    	for (Territoire territoire : this.ancImg.territoiresTraces.values())
    	{
    	    Pourtour pourtour = territoire.pourtour() ;
    	    
    		// Récupérer la liste des polygones du pourtour correspondant
    	    if (this.ancImg.multiPolygonesPourtoursTraces.get (pourtour.id()) != null)
    	    {
        	    for (Polygon polygone : this.ancImg.multiPolygonesPourtoursTraces.get (pourtour.id()))
        		{
                    // Si ce polygone contient le point (x,y)
        			if (polygone.contains(x, y))
        			{
        			    if (territoireDePlusHautRang == null || territoire.rang() > territoireDePlusHautRang.rang())
        			        territoireDePlusHautRang = territoire ;
        			}
        		}
    		}
    	}

    	// Renvoyer le territoire trouvé
    	return territoireDePlusHautRang ;
    }

    
    /**
     * Indique territoire dont le texte se trouve aux coordonnées (x, y) sur la carte.<br>
     * Renvoie null s'il n'y a pas de texte de territoire à cette position.
     * @param x L'abscisse du point de la carte.
     * @param y L'ordonnée du point de la carte.
     * @return Le texte qui se trouve aux coordonnées indiquées, null si aucun.
     */
    public Territoire territoireTexteAuxCoordonnees (double x, double y)
    {
        Territoire territoire = null ;
        for (Map.Entry<Rectangle,Territoire> texteTerritoire : this.ancImg.textesTerritoiresTraces.entrySet())
        {
            if (texteTerritoire.getKey().contains (x, y))
            {
                
                if (territoire == null || territoire.rang() < texteTerritoire.getValue().rang())
                {
                    territoire = texteTerritoire.getValue() ;
                }
            }
        }

        // Renvoyer le territoire trouvé
        return territoire ;
    }
    
    
    /**
     * Indique si l'image est périmée (une autre en cours de tracé).
     * @return Vrai si l'image est périmée, faux sinon.
     */
    public boolean perimee ()
    {
        return this.ancImg.perimee ;
    }
    
    
    /**
     * Libère les ressources associées à l'ancienne image et mémorise à la place les ressources associées
     *   à la nouvelle image.
     */
    public void terminerTracer ()
    {
        this.ancImg  = this.nouvImg ;
        this.nouvImg = new InfosImage() ;
    }
    
}
