/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.stockage;

import static fr.histoiremondiale.histoire.utiles.es.fichiers.Fichiers.creerRepertoires;
import static fr.histoiremondiale.histoire.utiles.exttypes.Chaines.saufEntreMarqueurs;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

import javax.swing.JOptionPane;

import fr.histoiremondiale.histoire.igraphique.donnees.InfosParagrapheNavig;
import fr.histoiremondiale.histoire.utiles.es.fichiers.Fichiers;
import fr.histoiremondiale.histoire.utiles.math.PointSphere;



/**
 * Exporte une carte et son texte sous forme de document html.
 */
public class ExporteurHTML
{

//    public static final String NOM_REP_HTML   = "html" ;        // Répertoire dans lequel stocker le paragraphe html
    public static  final String NOM_REP_IMAGES = "images" ;      // Répertoire dans lequel stocker l'image
    public static  final String ENCODAGE_HTML  = "UTF-8" ;
    private static final String MARQUEUR_TITRE = "###TITRE_PAGE###" ;
    private static final String DEBUT_HTML     = "<!DOCTYPE HTML PUBLIC '-//W3C//DTD HTML 4.01 Transitional//EN'>" + "\n" + // HTML 4 parce qu'en HTML 3.2 certains éléments ne passaient pas.
                                                "<html>" + "\n" +
                                                "  <head>" + "\n" +
                                                "    <meta http-equiv=\"Content-Style-Type\" content=\"text/css\">" + "\n" +
                                                "    <meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\">" + "\n" +
                                                "    <title>" + MARQUEUR_TITRE + "</title>" + "\n" +
                                                "  </head>" + "\n" +
                                                "<body>" + "\n" ;
    private static final String FIN_HTML       = "</body>\n</html>" + "\n" ;
    

    
    /**
     * Stocke le paragraphe et l'image fournis sous forme de document dans le répertoire en paramètre.<br>
     * Le paragraphe peut être null.<br>
     * Attention : la méthode écrase les fichiers s'ils existent déjà, c'est à l'appelant d'effectuer les
     *   vérifications.
     * @param paragraphe Paragraphe à exporter.
     * @param annee      Année de la carte affichée.
     * @param ptCentre   Point central de la carte exportée.
     * @param imageCarte Image de la carte à exporter.
     * @param repDest    Répertoire dans lequel exporter les fichiers.
     * @return Le chemin complet du fichier html généré. null en cas de problème.
     * @throws IOException En cas de problème d'accès aux fichiers.
     */
    public String exporter (InfosParagrapheNavig paragraphe, int annee, PointSphere ptCentre, BufferedImage imageCarte, File repDest) throws IOException
    {
        // Déterminer le nom des fichiers
        String nomCivilisation = ((paragraphe == null || paragraphe.civilisation() == null) ? "" : paragraphe.civilisation().nom()) ;
        String nomFicHtml      = nomCivilisation + "$" + annee + ".html" ;
        String nomFicImage     = nomCivilisation + "$" + annee + ".png" ;
        
        // Répertoires de génération
        String chemRepHtml   = repDest.getAbsolutePath();
        File   repHtml       = new File (chemRepHtml) ;
        String chemRepImages = repDest.getAbsolutePath() + File.separator + NOM_REP_IMAGES ;
        File   repImages     = new File (chemRepImages) ;
        
        
        // S'assurer que le répertoire destination du fichier html existe bien
        if (! creerRepertoires (repHtml))
        {
            JOptionPane.showMessageDialog (null,
                                           "Impossible de créer le répertoire pour la page html : \"" + repHtml + "\"",
                                           "Erreur lors de l'exportation de la carte",
                                           JOptionPane.ERROR_MESSAGE) ;
            return null ;
        }
        
        // S'assurer que le répertoire destination de l'image existe bien
        if (! creerRepertoires (repImages))
        {
            JOptionPane.showMessageDialog (null,
                                           "Impossible de créer le répertoire pour l'image : \"" + repHtml + "\"",
                                           "Erreur lors de l'exportation de la carte",
                                           JOptionPane.ERROR_MESSAGE) ;
            return null ;
        }


        
        // Récupérer et nettoyer le texte du paragraphe
        String texteParagraphe = (paragraphe == null ?
                                     "" :
                                     paragraphe.codeHtmlSimple()) ;
        // (supprimer les balises des autres liens)
        texteParagraphe = saufEntreMarqueurs (texteParagraphe, "<a ", ">") ;
        texteParagraphe = texteParagraphe.replace ("</a>", "") ;
        
        // Créer le texte de la page html
        String titrePageHtml = nomCivilisation + " " + annee ;
        String texteHtml = DEBUT_HTML.replace (MARQUEUR_TITRE, titrePageHtml) +
        					 "<div align = \"center\">\n" +
                               "<img src=\"./images/" + nomFicImage + "\" alt=\"Carte\"><br>\n" +
                               "<center><i>" + nomCivilisation + " "
                                             + ptCentre.enChaineElaboree() + " "
                                             + "en " + annee +
                                       "</i>" +
                               "</center>\n" +
                             "</div>\n" +
                             "<br>" +
                             "<div align = \"justify\">\n"  + texteParagraphe + "\n</div>\n"  +    		
                           FIN_HTML ;
        
        // Créer la page html
        String chemFicHtml = repHtml.getAbsolutePath() + File.separator + nomFicHtml ;
        Fichiers.sauverContenuTexte (texteHtml, chemFicHtml, ENCODAGE_HTML) ;
        
        // Créer l'image de la carte
        String chemFicImages = repImages.getAbsolutePath() + File.separator + nomFicImage ;
        new ExporteurPNG().exporter (imageCarte, chemFicImages) ;
        

        // Renvoyer le chemin du fichier généré
        return chemFicHtml ;
    }
    
}
