/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.utiles.es.fichiers;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.IOException;



/**
 * Fonctions utilitaires de lecture de valeurs telles que gérées par les processeurs x86
 *   (les poids faibles devant).
 */
public class LectureX86
{

    /**
     * Lit un octet dans le flux de données et renvoie la valeur.<br>
     * Note : Simplement par homogénéité lors de l'utilisation des méthodes de lecture, rien de différent de
     *        readByte()
     * @param lect Flux de lecture des données.
     * @return La valeur lue.
     * @throws IOException En cas de problème de lecture du flux.
     */
    public static byte lireOctet (DataInputStream lect) throws IOException
    {
        return lect.readByte() ;
    }

    
    /**
     * Lit un booléen sur un octet dans le flux de données et renvoie la valeur.<br>
     * (lecture normale de booléen)
     * @param lect Flux de lecture des données.
     * @return La valeur lue.
     * @throws IOException En cas de problème de lecture du flux.
     */
    public static boolean lireBooleen1 (DataInputStream lect) throws IOException
    {
        return lect.readBoolean() ;
    }
    
    
    /**
     * Lit un booléen sur deux octets dans le flux de données et renvoie la valeur.<br>
     * (lecture d'un booléen sur 1 octets, puis sauf d'un octet : les fichiers VB semblent contenir des
     *  booléens sur 2 octets de valeur identique, 0 ou 255 (-1))
     * @param lect Flux de lecture des données.
     * @return La valeur lue.
     * @throws IOException En cas de problème de lecture du flux.
     */
    public static boolean lireBooleen2 (DataInputStream lect) throws IOException
    {
        boolean b = lect.readBoolean() ;
        lect.skip (1) ;
        return b ;
    }

    
    /**
     * Lit un entier sur deux octets dans le flux de données et renvoie la valeur.<br>
     * @param lect Flux de lecture des données.
     * @return La valeur lue.
     * @throws IOException En cas de problème de lecture du flux.
     */
    public static short lireEntier2 (DataInputStream lect) throws IOException
    {
        // Lire les octets
        byte o1 = lect.readByte() ;
        byte o2 = lect.readByte() ;
        
        // Réorganiser les octets
        byte[] octets = {o2, o1} ;
        
        // Relire le nombre dans l'ordre
        DataInputStream lect2 = new DataInputStream (new ByteArrayInputStream (octets)) ;
        short val = lect2.readShort() ;
        lect2.close() ;
        
        // Renvoyer le nombre
        return val ;
    }
    
    
    /**
     * Lit un entier sur quatre octets dans le flux de données et renvoie la valeur.<br>
     * @param lect Flux de lecture des données.
     * @return La valeur lue.
     * @throws IOException En cas de problème de lecture du flux.
     */
    public static int lireEntier4 (DataInputStream lect) throws IOException
    {
        // Lire les octets
        byte o1 = lect.readByte() ;
        byte o2 = lect.readByte() ;
        byte o3 = lect.readByte() ;
        byte o4 = lect.readByte() ;
        
        // Réorganiser les octets
        byte[] octets = {o4, o3, o2, o1} ;
        
        // Relire le nombre dans l'ordre
        DataInputStream lect2 = new DataInputStream (new ByteArrayInputStream (octets)) ;
        int val = lect2.readInt() ;
        lect2.close() ;
        
        // Renvoyer le nombre
        return val ;
    }
    
    
    /**
     * Lit un entier sur huit octets dans le flux de données et renvoie la valeur.<br>
     * @param lect Flux de lecture des données.
     * @return La valeur lue.
     * @throws IOException En cas de problème de lecture du flux.
     */
    public static long lireEntier8 (DataInputStream lect) throws IOException
    {
        // Lire les octets
        byte o1 = lect.readByte() ;
        byte o2 = lect.readByte() ;
        byte o3 = lect.readByte() ;
        byte o4 = lect.readByte() ;
        byte o5 = lect.readByte() ;
        byte o6 = lect.readByte() ;
        byte o7 = lect.readByte() ;
        byte o8 = lect.readByte() ;

        // Réorganiser les octets
        byte[] octets = {o8, o7, o6, o5, o4, o3, o2, o1} ;
        

        // Relire le nombre dans l'ordre
        DataInputStream lect2 = new DataInputStream (new ByteArrayInputStream (octets)) ;
        long val = lect2.readLong() ;
        lect2.close() ;
        
        // Renvoyer le nombre
        return val ;
    }
    

    /**
     * Lit un flottant sur quatre octets dans le flux de données et renvoie la valeur.<br>
     * @param lect Flux de lecture des données.
     * @return La valeur lue.
     * @throws IOException En cas de problème de lecture du flux.
     */
    public static float lireFlottant4 (DataInputStream lect) throws IOException
    {
        // Lire les octets
        byte o1 = lect.readByte() ;
        byte o2 = lect.readByte() ;
        byte o3 = lect.readByte() ;
        byte o4 = lect.readByte() ;
        
        // Réorganiser les octets
        byte[] octets = {o4, o3, o2, o1} ;
        
        // Relire le nombre dans l'ordre
        DataInputStream lect2 = new DataInputStream (new ByteArrayInputStream (octets)) ;
        float val = lect2.readFloat() ;
        lect2.close() ;
        
        // Renvoyer le nombre
        return val ;
    }
    

}
