/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.utiles.exttypes;



/**
 * Fonctions utiles sur les nombres flottants.
 */
public class Flottants
{

    /**
     * Arrondit un nombre flottant à la nième décimale.<br>
     * Les valeurs du système décimal ne peuvent pas être stockées de manière exacte en binaire en virgule
     *   flottante (0,1 n'a pas de valeur exacte en binaire). La valeur binaire la plus proche est utilisée.
     *   Les opérations sur les nombres en virgule flottante (additions, ...) introduisent donc des dérives et
     *   l'addition 0,1 + 0,1 + 0,1 peut donner des choses comme 0,300000000001 par exemple.<br>
     * Lorsque des opérations en virgules flottantes sont effectuées, il est parfois utile de les corriger
     *   (pour les afficher par exemple).
     * @param nombre      Le nombre à arrondir.
     * @param nbDecimales Le nombre de décimales à conserver.
     * @return La valeur arrondie.
     */
    public static double arrondir (double nombre, int nbDecimales)
    {
        long multiplicateur = (long) Math.pow (10, nbDecimales) ;
        return (double) Math.round (nombre * multiplicateur) / multiplicateur ;
    }
    
    
    /**
     * Indique si un nombre flottant est nul (&lt;=&gt; assez proche de zéro).
     * @param nombre Le nombre à tester.
     * @return Vrai si le nombre est assez proche de 0 pour être considéré comme nul, faux sinon.
     */
    public static boolean estNul (double nombre)
    {
        return Math.abs(nombre) < 0.00000001 ;
    }
    
    
    /**
     * Indique si deux nombre flottants sont égaux (&lt;=&gt; assez proches l'un de l'autre).
     * @param nb1 Le premier  nombre.
     * @param nb2 Le deuxième nombre.
     * @return Vrai si les nombres sont égaux, faux sinon.
     */
    public static boolean egaux (double nb1, double nb2)
    {
        return estNul (nb1 - nb2) ;
    }
    
    
    /**
     * Affiche une valeur réelle (2 chiffres après la virgule, séparateur des milliers et des décimales
     *   corrects).<br>
     * (séparateur de milliers : aucun ; séparateur décimal : virgule)
     * @param valeur La valeur à afficher.
     * @return La chaîne représentant la valeur.
     */
    public static String formater (double valeur)
    {
        String chValeur = "" + arrondir (valeur, 2) ;
        chValeur = chValeur.replaceAll ("\\.", ",") ;
        return chValeur ;
    }
    /**
     * Affiche une valeur réelle (2 chiffres après la virgule, séparateur des milliers et des décimales
     *   corrects).<br>
     * (séparateur de milliers : aucun ; séparateur décimal : virgule)<br>
     * Si le nombre se termine par ",0" ces deux caractères sont supprimés.
     * @param valeur La valeur à afficher.
     * @return La chaîne représentant la valeur.
     */
    public static String formaterSimplifier (double valeur)
    {
        String chValeur = formater (valeur) ;
        return (chValeur.endsWith (",0") ? chValeur.substring (0, chValeur.length() - 2) : chValeur) ;
    }
     /**
     * Affiche une valeur entière (séparateur des milliers correct).<br>
     * (séparateur de milliers : espace ; séparateur décimal : aucun [valeurs arrondies])<br>
     * Si le nombre se termine par ",0" ces deux caractères sont supprimés.
     * @param valeur La valeur à afficher.
     * @return La chaîne représentant la valeur.
     */
    // 
    public static String formater2 (double valeur)
    {
        String chValeur ;                   // Valeur transformée en chaîne
        String chFormatee ;                 // Valeur formatée
        
        // Récupérer la valeur
        chValeur = "" + Math.round (valeur) ;
        chFormatee = "" ;
        
        // Ajouter les espaces
        int cpt = 0 ;
        for (int i = chValeur.length()-1 ; i >= 0 ; i--)
        {
            chFormatee = chValeur.charAt (i) + chFormatee ;
            if (((++cpt % 3) == 0) && (i != 0))
                chFormatee = " " + chFormatee ;
        }
        
        // Renvoyer le résultat
        return chFormatee ;
    }
    
}
