/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.utiles.math;

import static java.lang.Math.sqrt ;



import java.awt.Point;
import java.util.ArrayList;
import java.util.List ;



/**
 * Fonctions de calcul dans le plan.
 */
public class CalculsPlan
{
    
    
    /**
     * Crée et renvoie un point du plan à partir de coordonnées angulaire.<br>
     * L'angle est en radians.
     * @param r     Rayon.
     * @param angle Angle.
     * @return Le point correspondant en coordonnées cartésiennes.
     */
    public static PointPlan creerPointCoordAngulaires (double r, double angle)
    {
        double x = Math.floor (r * Math.cos (angle)) ;
        double y = Math.floor (r * Math.sin (angle)) ;
        return new PointPlan (x, y) ;
    }
    
    
    /**
     * Renvoie l'angle à l'origine OI,OP (en degrés décimaux).<br>
     *   - O est le centre du repère (0,0)<br>
     *   - I est sur l'axe des abscisses (100,0)<br>
     *   - P le point lui-même<br>
     * L'angle est en radians.
     * @param p Point dans le plan.
     * @return L'angle calculé.
     */
    public static double angleOrigine (PointPlan p)
    {
        PointPlan o = new PointPlan (  0, 0) ;
        PointPlan i = new PointPlan (100, 0) ;
        return angle (o, i, p) ;
    }
    

    /**
     * Calcule l'angle en radian OA,OB dans cet ordre.<br>
     * Si O, A et B sont alignés alors l'angle est nul ou plat (cela intègre les cas ou O=A, O=B ou A=B).<br>
     * L'angle est compris entre [0,2Pi[.
     * @param o Point du plan.
     * @param a Point du plan.
     * @param b Point du plan.
     * @return L'angle calculé.
     */
    public static double angle (PointPlan o, PointPlan a, PointPlan b)
    {
        double teta = angleRelatif (o, a, b) ;
        return (teta >= 0 ? teta : 2 * Math.PI + teta) ;
    }
    
    /**
     * Calcule l'angle en radian OA,OB dans cet ordre.<br>
     * Si O, A et B sont alignés alors l'angle est nul ou plat (cela intègre les cas ou O=A, O=B ou A=B).<br>
     * L'angle est compris entre ]-2Pi;+2Pi[.
     * @param o Point du plan.
     * @param a Point du plan.
     * @param b Point du plan.
     * @return L'angle calculé.
     */
    public static double angleRelatif (PointPlan o, PointPlan a, PointPlan b)
    {
        int posRelative = positionRelative (o, a, b) ;
        
        // O, A et B alignés
        if (posRelative == 0)
        {
            // Si O, A et B sont sur une parallèle à l'axe des Y
            if (a.x() == o.x())
            {
                return ((b.y() - o.y()) * (a.y()- o.y()) < 0) ? Math.PI : 0 ;
            }
            else
            {
                // Si B n'est pas du même côté que O alors l'angle vaut pi
                return ((b.x() - o.x()) * (a.x() - o.x()) < 0) ? Math.PI : 0 ;
            }
        }
        
        double teta = Math.acos (produitScalaire (o, a, b) / (o.distanceA (a) * o.distanceA (b))) ;
        if (posRelative < 0)
            teta = -teta ;
        return teta ;
    }
    
    /**
     * -Renvoie -1, 0 ou 1 suivant la position de C par rapport à (AB).<br>
     * Soit A et B deux points distincts et C un troisième point.<br>
     * Le déterminant des vecteurs AC et BC vaut 0 si C est sur la droite AB et
     *   garde un signe constant dans chaque demi-plan défini par la droite (AB).<br>
     * Positif si A,B,C en sens direct (sens inverse des aiguilles d'une montre
     *   ou sens trigonométrique), négatif sinon.
     * @param c Point du plan.
     * @param a Point du plan.
     * @param b Point du plan.
     * @return -1, 0 ou +1 selon les positions relatives des points.
     */
    public static int positionRelative (PointPlan c, PointPlan a, PointPlan b)
    {
        double determinant = (a.x() - c.x()) * (b.y() - c.y()) - (a.y() - c.y()) * (b.x() - c.x()) ;
        if (Math.abs(determinant) < 1.0E-10) determinant = 0;
        return (int) Math.signum (determinant) ;
    }
    
    /**
     * Renvoie le produit scalaire des vecteurs OA et OB.
     * @param o Point du plan.
     * @param a Point du plan.
     * @param b Point du plan.
     * @return Le produit scalaire.
     */
    public static double produitScalaire (PointPlan o, PointPlan a, PointPlan b)
    {
        return (a.x() - o.x()) * (b.x() - o.x()) +
               (a.y() - o.y()) * (b.y() - o.y()) ;
    }
    
    /**
     * Renvoie le rayon du cercle circonscrit au rectangle dont on passe la largeur et la hauteur.
     * @param largeur Largeur du rectangle.
     * @param hauteur Hauteur du rectangle.
     * @return Le rayon calculé.
     */
    public static double rayonCercleCirconscrit (int largeur, int hauteur)
    {
        return sqrt ((largeur * largeur + hauteur * hauteur) / 4) ;
    }
    
    
    /**
     * Calcule la somme des angles CA(i),CA(i+1), où C est le point central et A(i) les points de la liste,
     *   qui constituent un contour fermé.<br>
     * La somme est en radian.<br>
     * Si le point central est extérieur au contour, la somme vaut 0.<br>
     * Si le point central est à l'intérieur du contour, la somme vaut 2*pi (contour dans le sens direct) ou
     *   -2*pi (contour dans le sens inverse).<br>
     * Note : attention aux imprécisions dans la somme de nombres flottants (il se peut que la somme ne
     *   vaille pas très précisément 0 ou +/-2*pi).
     * @param ptCentre      Point central.
     * @param pointsContour Points du contour.
     * @return La somme des angles.
     */
    public static double sommeAngles (PointPlan ptCentre, List<PointPlan> pointsContour)
    {
        double somme = 0 ;
        for (int i = 0 ; i < pointsContour.size() ; i++)
        {
            somme += angleRelatif (ptCentre,
                                   pointsContour.get (i),
                                   pointsContour.get ((i + 1) % pointsContour.size())) ;
        }
        return somme ;
    }
 
    /**
     * On calcule l'intersection entre le segment [AB] et le cercle de centre O et de 
     * rayon R. A est dans le cercle et B est hors du cercle. Le point recherché 
     * est de type t*A + (1-t)*B avec t entre 0 et 1 et vérifiant l'équation du second degré
     * at²+bt+c = 0 avec
     * a = (Ax-Bx)²+(Ay-By)²
     * b = Bx*(Ax-Bx)+By*(Ay-By)
     * c = Bx²+By²-R²
     * Il faut calculer le discriminant et discuter suivant son signe
     * @param A le premier point
     * @param B le deuxième point
     * @param rayon le rayon du cercle
     * @return l'intersection
     */
    public static PointPlan intersectionSegmentCercle(PointPlan A, PointPlan B, double rayon)
    {
        double Ax = A.x (),
               Ay = A.y (),
               Bx = B.x (),
               By = B.y ();
        double a = (Ax-Bx)*(Ax-Bx)+(Ay-By)*(Ay-By),
               b = 2*(Bx*(Ax-Bx)+By*(Ay-By)),
               c = Bx*Bx + By*By - rayon*rayon; 
        double delta = b*b - 4*a*c;
        double t = (-b + Math.sqrt(delta))/(2*a);
        if ((t < 0)||(t>1))
                t = (-b - Math.sqrt(delta))/(2*a);
        if ((t<0)||(t>1))
        {
        	System.out.println("ce cas ne devrait jamais se produire");
            System.out.println("determinerPointSurLeCercle de rayon : alerte " + rayon);
            System.out.println("t = " + t);
            System.out.println("A = " + A.toString());
            System.out.println("B = " + B.toString());
            
            return (t<0 ? new PointPlan(Double.NaN,0) : new PointPlan(0,Double.NaN));
        }
        else
        {
            double x = t*Ax + (1-t)*Bx,
                   y = t*Ay + (1-t)*By;
            return new PointPlan(x,y);
        }
    }
    
    /**
     * changement de repère : au début l'origine est au centre de la carte on veut la 
     * ramener en haut à gauche de l'écran, ordonnées vers le bas
     * @param P le point dont on cherche les nouvelles coordonnées
     * @param largeur la largeur de la carte
     * @param hauteur la hauteur de la carte
     * @return les nouvelles coordonnées
     */
    public static PointPlan changementOrigineCentreVersCoinHautGauche(PointPlan P, int largeur, int hauteur)
    {
        return new PointPlan (largeur / 2 + P.x(), hauteur / 2 - P.y()) ;
    }

    /**
     * application de la méthode de changement de repères pour toute une liste de points
     * @param ligne la liste de points dont on cherche les nouvelles coordonnées 
     * @param largeur la largeur de l'image
     * @param hauteur la hauteur de l'image
     * @return la liste des nouvelles coordonnées
     */
    public static List<PointPlan> changementOrigineCentreVersCoinHautGauche(List<PointPlan> ligne, int largeur, int hauteur)
    {
        List<PointPlan> ligneAjustee = new ArrayList<>();
        for (PointPlan P : ligne) ligneAjustee.add(changementOrigineCentreVersCoinHautGauche(P, largeur, hauteur)) ;
        return ligneAjustee;
    }

    /**
     * application de la méthode de changement de repères pour une liste à un ensemble
     * de listes 
     * @param contours la liste de listes dont on veut les nouvelles coordonnées
     * @param largeur la largeur de l'image
     * @param hauteur la hauteur de l'image
     * @return la liste des listes de nouvelles coordonnées
     */
    public static List<List<PointPlan>> changementOrigineCentreVersCoinHautGauchePourTout(List<List<PointPlan>> contours, int largeur, int hauteur) 
    {
        List<List<PointPlan>> contoursAjustes = new ArrayList<List<PointPlan>>() ;
        for (List<PointPlan> contour : contours)
        {
            List<PointPlan> contourAjuste = changementOrigineCentreVersCoinHautGauche(contour, largeur, hauteur);
            contoursAjustes.add(contourAjuste);
        }
        return contoursAjustes;
    }

    
}
