/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.stockage;

import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;

import fr.histoiremondiale.histoire.utiles.exceptions.ExceptionConnexionBaseDonnees;



/**
 * Classe générale d'accès aux données de l'application.
 */
public class AccesDonneesBase
{

    // [Notes sur l'accès à la base H2]
    //
    //     - Avec l'option file_lock=no, la base de données n'est pas verrouillée. Le seul risque que je
    //       vois est qu'on modifie la base pendant qu'elle est ouverte par l'appli, un développeur avec
    //       SquirrelSQL par exemple. Et dans le pire des cas improbables l'appli pourrait planter à cause
    //       de cette manoeuvre, mais ce n'est pas très grave, on ne risque pas de corruption des données.
    //       L'appli peut ainsi être ouverte plusieurs fois, par un utilisateur ou par des utilisateurs
    //       différents.
    //     - Le fait qu'on ne puisse pas écrire dans le répertoire contenant la base (fichier de traces)
    //       quand l'appli est installée chez un utilisateur ne semble pas choquer H2. A la limite c'est un
    //       peu dommage de perdre le fichier des journaux, en cas de problème lié à la base chez un
    //       utilisateur. Notons que les exceptions devraient se retrouver dans la console, s'il y en a une.
    //       Si on veut récupérer les erreurs de manière plus fiable, mieux vaudrait récupérer les traces de
    //       la console / envoyer les exceptions ailleurs que s'attaquer spécifiquement au problème de la
    //       base de données.
    //     ! Le nom d'utilisateur et le mot de passe sont en clair dans le code : de toutes façons la base
    //       ne contient que des données standards (pas propres à l'utilisateur) et se trouve dans une zone
    //       en lecture seule une fois installée (et de toutes manières l'utilisateur a les mêmes droits sur
    //       le fichier de la base qu'à travers l'application)

    
    // Définition de la connexion (pas la peine de faire un truc générique pour l'instant)
    private static final String NOM_CLASSE_PILOTE_JBDC = "org.h2.Driver" ;
    private static final String PROTOCOLE_BASE         = "jdbc:h2:" ;
    private static final String NOM_BASE               = "histoire" ;
    private static final String OPTIONS_BASE           = "file_lock=no;access_mode_data=r" ;    // Base en lecture seule (au lieu de rw, rws ou rwd)
    private static final String NOM_UTILISATEUR        = "sa" ;
    private static final String MOT_DE_PASSE           = "" ;
    
    // Informations de connexion à la base de données
    private String nomClassePiloteJDBC ;
    private String urlBase ;
    private String nomUtilisateur ;
    private String motDePasse ;

    // Une seule connexion pour toute l'appli
    private static Connection connexionCourante ;
    
    
    
    /**
     * Constructeur.
     * @param chemRepDonnees Chemin du répertoire des données de l'appli.
     */
   public AccesDonneesBase (String chemRepDonnees)
    {
        // Mémoriser les paramètres
        this.nomClassePiloteJDBC = NOM_CLASSE_PILOTE_JBDC ;
        this.urlBase             = PROTOCOLE_BASE
                                 + new File(chemRepDonnees).getAbsolutePath() + File.separator + "histoire" + File.separator + "base"
                                 + "/"
                                 + NOM_BASE
                                 + ";"
                                 + OPTIONS_BASE ;
        this.nomUtilisateur      = NOM_UTILISATEUR ;
        this.motDePasse          = MOT_DE_PASSE ;
        
        // Vérifier que le pilote JDBC est bien disponible
        try
        {
            Class.forName (this.nomClassePiloteJDBC) ;
        }
        catch (ClassNotFoundException e)
        {
            throw new RuntimeException ("Pilote JDBC non trouvé : " + this.nomClassePiloteJDBC) ;
        }
        
        // Vérifier que la connexion à la base est bien initialisée
        try
        {
        	//urlBase = PROTOCOLE_BASE ("jdbc:h2:") + 
        	//
            creerConnexion (this.urlBase, this.nomUtilisateur, this.motDePasse) ;
        }
        catch (SQLException e)
        {
            throw new ExceptionConnexionBaseDonnees ("Impossible de créer la connexion à la base de données", e) ;
        }
    }
    
    // Créee la connexion à la base
    private static synchronized void creerConnexion (String urlBase, String nomUtilisateur, String motDePasse) throws SQLException
    {
        if (connexionCourante == null)
            connexionCourante = DriverManager.getConnection (urlBase, nomUtilisateur, motDePasse) ;
    }
    
    /**
     * Ferme la connexion à la base de données.
     * @throws SQLException Si une exception survient.
     */
    public static synchronized void fermerConnexion () throws SQLException
    {
        if (connexionCourante != null)
            connexionCourante.close() ;
    }
    
    
    
    /**
     * Renvoie la connexion à la base de données.
     * @return La connexion à la base de données.
     */
    protected static synchronized Connection connexionCourante ()
    {
        return connexionCourante ;
    }

    /**
     * Pour déterminer les livres liés à un paragraphe
     */
	public static String obtenirBiblio(int id_par) throws SQLException 
	{
		String sql = "SELECT LIVRES.TITRE, LIVRES.AUTEUR "
				+ "FROM PARAGRAPHES_LIVRES "
				+ "JOIN LIVRES ON PARAGRAPHES_LIVRES.LIVRE = LIVRES.ID "
				+ "WHERE PARAGRAPHE = " + id_par + ";";
		
		ResultSet resultat = connexionCourante.prepareStatement(sql).executeQuery();
		String biblio = "<html><head>"
				+ "<style type='text/css'>body {background-color:#E0FCFE;color:#000000;margin:10px;font-family:Arial;text-align:justify;}>"
				+ "</style></head><body><table width = '100%' border = 1>"
				+ "<tr><th width = '55%'>Titre</th><th width = '45%'>Auteur (ou éditeur)</th></tr>";
		boolean trouveBiblio = resultat.next();
		if (trouveBiblio)
		{
			do
			{
				String titre = resultat.getString("TITRE");
				String auteur = resultat.getString("AUTEUR");
				biblio += "<tr><td>" + titre + "</td><td>" + auteur + "</td></tr>";
				
			}while (resultat.next());
			return biblio + "</table></body></html>";
		}
		return null;
	}
}
