/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.utiles.es.fichiers;

import static fr.histoiremondiale.histoire.utiles.exttypes.Chaines.sousChaine;

import java.io.BufferedInputStream ;
import java.io.BufferedOutputStream ;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream ;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream ;
import java.io.InputStreamReader;
import java.io.OutputStream ;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;



/**
 * Fonctions utilitaires diverses pour la lecture et l'écriture dans les fichiers.
 */
public class Fichiers
{
    
    
    /**
     * Renvoie l'URL (locale) désignant le fichier (ou le répertoire) dont le chemin est passé en paramètre.
     * @param cheminFichier Chemin du fichier.
     * @return L'URL correspondant à ce chemin local.
     */
    public static String urlLocaleDeNomFichier (String cheminFichier)
    {
        return "file:///" + new File (cheminFichier).getAbsolutePath().replace (File.separator, "/") ;
    }
    
    
    /**
     * Renvoie la chaîne représentant le contenu du fichier texte.
     * @param chemFic     Le chemin du fichier.
     * @param encodageFic L'encodage du fichier.
     * @return Le contenu du fichier dans une chaîne.
     * @throws IOException En cas de problème d'accès au fichier.
     */
    public static String chargerContenuTexte (String chemFic, String encodageFic) throws IOException
    {
        return chargerContenuTexte(new File (chemFic),encodageFic);
    }

    /**
     * renvoie la chaine représentant le contenu du fichier texte avec un encodage par défaut (UTF8)
     * @param fichier le nom du fichier (avec son chemin d'accès)
     * @return le contenu du fichier
     * @throws IOException En cas de problème d'accès au fichier.
     */
    public static String chargerContenuTexte (String fichier) throws IOException
    {
    	return chargerContenuTexte(fichier, "UTF-8");
    }
    
    /**
     * renvoie la chaine représentant le contenu du fichier texte avec un encodage par défaut (UTF8)
     * @param fic le fichier
     * @return le contenu du fichier
     * @throws IOException En cas de problème d'accès au fichier.
     */

    public static String chargerContenuTexte (File fic) throws IOException
    {
    	return chargerContenuTexte(fic,"UTF-8");
    }
    
    /**
     * Renvoie la chaîne représentant le contenu du fichier texte.
     * @param fic         Le fichier.
     * @param encodageFic L'encodage du fichier.
     * @return Le contenu du fichier dans une chaîne.
     * @throws IOException En cas de problème d'accès au fichier.
     */
    public static String chargerContenuTexte (File fic, String encodageFic) throws IOException
    {
        StringBuilder  chTmp ;
        BufferedReader lect  = null ;
        
        try
        {
            // Initialisations
            chTmp = new StringBuilder() ;
            lect  = new BufferedReader (new InputStreamReader (new FileInputStream (fic), encodageFic)) ;
            
            // Lire le fichier
            String ligne ;
            while ((ligne = lect.readLine()) != null)
            {
                chTmp.append (ligne) ;
                chTmp.append ("\n") ;
            }
            
            // Fermer le fichier
            lect.close() ;
            lect = null ;
        }
        finally
        {
            if (lect != null)
            {
                try { lect.close() ; }
                catch (Throwable e) {}
            }
        }
        
        // Renvoyer le texte
        return chTmp.toString() ;
    }

    
    /**
     * Sauvegarde le texte dans le fichier dont le chemin est passé en paramètre.
     * @param texte       Le texte à sauvegarder.
     * @param chemFic     Le chemin du fichier dans lequel écrire.
     * @param encodageFic Encodage à utiliser.
     * @throws IOException En cas de problème d'accès au fichier.
     */
    public static void sauverContenuTexte (String texte, String chemFic, String encodageFic) throws IOException
    {
        FileOutputStream ecrBin = null ;
        PrintWriter      ecr    = null ;
        try
        {
            ecrBin = new FileOutputStream (chemFic) ;
            ecr    = new PrintWriter (new OutputStreamWriter (ecrBin, encodageFic)) ;
            ecr.write (texte) ;
            ecrBin.getFD().sync() ;
            ecr.close() ;
            ecr    = null ;
        }
        finally
        {
            if (ecr != null)
            {
                try { ecrBin.getFD().sync() ; }
                catch (Throwable e) {}
                try { ecr.close() ; }
                catch (Throwable e) {}
            }
        }
    }

    
    /**
     * Crée le répertoire dont le chemin est passé en paramètre. Les répertoires parents sont créés si nécessaire.<br>
     * Renvoie vrai si le répertoire existe à la fin.
     * @param chemRep Le chemin du répertoire à créer.
     * @return Vrai si el répertoire existe à la fin de l'opération, faux sinon.
     */
    public static boolean creerRepertoires (String chemRep)
    {
        return creerRepertoires (new File (chemRep)) ;
    }
    /**
     * Crée le répertoire passé en paramètre. Les répertoires parents sont créés si nécessaire.<br>
     * Renvoie vrai si le répertoire existe à la fin.
     * @param rep Le répertoire à créer.
     * @return Vrai si el répertoire existe à la fin de l'opération, faux sinon.
     */
    public static boolean creerRepertoires (File rep)
    {
        rep.mkdirs() ;      // (renvoie faux si le répertoire existait déjà)
        return rep.exists() && rep.isDirectory() ;
    }
    
    
    /**
     * Renomme un fichier.<br>
     * Un paramètre permet d'indiquer si on souhaite détruire le fichier destination s'il existe déjà.<br>
     * Si le fichier destination existe et que la fonction n'est pas autorisée à le détruire, une exception
     *   est levée.
     * @param ficSource                      Fichier à renommer.
     * @param ficDestination                 Fichier une fois renommé.
     * @param detruireFicDestinationSiExiste Indique si le fichier destination peut être détruit dans le cas
     *                                       où il existe déjà.
     */
    public static void renommer (File ficSource, File ficDestination, boolean detruireFicDestinationSiExiste)
    {
        // Vérifier que le fichier source existe
        if (! ficSource.exists())
            throw new RuntimeException ("Impossible de renommer " + ficSource.getAbsolutePath() + ": le fichier n'existe pas.") ;
        
        // Vérifier que le fichier destination n'existe pas
        if (ficDestination.exists())
        {
            if (detruireFicDestinationSiExiste)
                ficDestination.delete() ;
            else
                throw new RuntimeException ("Impossible de renommer " + ficSource.getAbsolutePath() + " en " + ficDestination.getAbsolutePath() + " : le fichier destination existe déjà.") ;
        }
        
        // Renommer le fichier
        ficSource.renameTo (ficDestination) ;
    }
    
    
    /**
     * Indique si le nom de fichier en paramètre se termine par l'extension indiquée (par ".extension").<br>
     * Le test ne différencie pas les majuscules des minuscules et png est donc égal à PNG.
     * @param nomFic    Nom du fichier à tester.
     * @param extension Extension (sans le ".").
     * @return vrai si le nom du fichier se termine par l'extension considérée, faux sinon.
     */
    public static boolean extensionEgale (String nomFic, String extension)
    {
        // Cas du nom de fichier trop court
        if (nomFic.length() < extension.length() + 1)
            return false ;
        
        // Comparer la fin du nom de fichier à l'extension indiquée.
        String finNomFic = sousChaine (nomFic, nomFic.length() - extension.length() - 1) ;
        return finNomFic.equalsIgnoreCase ("." + extension) ;
    }
  
    /**
     * sauver un contenu texte dans un fichier (encodage UTF-8)
     * @param fichier nom du fichier à écrire
     * @param contenu texte à sauver dans le fichier
     */
    public static void ecrireFichier(String fichier, String contenu)
    {
    	try
		{
			sauverContenuTexte(contenu, fichier, "UTF-8");
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
    }
    
    
    // Renvoie le contenu binaire d'un fichier sous forme d'un tableau d'octets.
    public static byte[] chargerContenuBinaire (String chemFic) throws IOException
    {
        // Charger les données
        InputStream           lectFic   = new BufferedInputStream (new FileInputStream (chemFic)) ;
        ByteArrayOutputStream ecrOctets = new ByteArrayOutputStream() ;
        OutputStream          ecrTampon = new BufferedOutputStream (ecrOctets) ;
        int                   octet ;

        // Copier les données octet par octet
        while ((octet = lectFic.read()) != -1)
            ecrOctets.write (octet) ;
        
        // Fermer les flux
        ecrTampon.close() ;
        lectFic.close() ;
        
        // Renvoyer le tableau d'octets
        return ecrOctets.toByteArray() ;
    }

}
