/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.utiles.es.fichiers;

import static fr.histoiremondiale.histoire.utiles.es.fichiers.Fichiers.renommer;

import java.io.BufferedInputStream ;
import java.io.BufferedOutputStream ;
import java.io.File ;
import java.io.FileInputStream ;
import java.io.FileOutputStream ;
import java.io.IOException ;
import java.io.InputStream ;
import java.io.InputStreamReader ;
import java.io.OutputStream ;
import java.io.OutputStreamWriter ;
import java.io.PrintWriter ;
import java.io.Reader ;
import java.util.Properties ;



/**
 * Classe permettant de simplifier la manipulation des fichiers de propriétés.
 */
public class Proprietes
{

    /**
     * Charge le contenu d'un fichier de propriétés.<br>
     * Si le fichier indiqué n'existe pas, un ensemble vide de propriétés est renvoyé (utile pour
     *   les fichiers de sauvegarde de paramètres, optionnels).
     * @param chemFicConfig Chemin du fichier de configuration à lire.
     * @param encodage      Encodage à utiliser.
     * @return Les propriétés lues dans le fichier.
     * @throws IOException En cas de problème de lecture du fichier.
     */
    public static Properties chargerProprietes (String chemFicConfig, String encodage) throws IOException
    {
        Properties props = new Properties() ;               // Liste de propriétés à remplir
        File ficConfig   = new File (chemFicConfig) ;       // Fichier source
        
        
        // Si le fichier existe, charger les données
        if (ficConfig.exists())
        {
            // Ouvrir le fichier
            InputStream lectBin = new BufferedInputStream (new FileInputStream (ficConfig)) ;
            Reader      lect    = new InputStreamReader   (lectBin, encodage) ;
            
            // Lire les propriétés
            try
            {
                props.load (lect) ;
                lect.close() ;
                lect = null ;
            }
            finally
            {
                if (lect != null)
                {
                    // Essayer de fermer le fichier, mais sans masquer l'erreur d'origine
                    try     { lect.close() ; }
                    catch (Throwable e) {}
                }
            }
        }
        
        // Renvoyer les propriétés chargées
        return props ;
    }
    
    
    /**
     * Sauvegarde les propriétés dans le fichier indiqué.<br>
     * La fonction utilise un fichier temporaire pour éviter d'écraser l'ancien si quelque chose se passe mal
     *   pendant la sauvegarde.
     * @param proprietes    Propriétés à sauvegarder.
     * @param titre         Texte qui sera inscrit en commentaire dans le fichier de propriétés.
     * @param chemFicConfig Chemin du fichier dans lequel écrire.
     * @param encodage      Encodage à utiliser.
     * @throws IOException En cas de problème de lecture du fichier.
     */
    public static void sauverProprietes (Properties proprietes, String titre, String chemFicConfig, String encodage) throws IOException
    {
        // S'assurer que le répertoire de destination du fichier existe
        File ficConfig    = new File (chemFicConfig) ;
        File ficConfigTmp = new File (ficConfig.getAbsolutePath() + "~") ;
        ficConfig.getParentFile().mkdirs() ;
        
        // Ouvrir le fichier
        FileOutputStream ecrFic = new FileOutputStream     (ficConfigTmp) ;
        OutputStream     ecrBin = new BufferedOutputStream (ecrFic) ;
        PrintWriter      ecr    = new PrintWriter (new OutputStreamWriter (ecrBin, encodage), true) ;
        
        try
        {
            // Stocker les propriétés
            proprietes.store (ecr, titre) ;
            
            // Fermer le fichier
            ecr.flush() ;           // (vide le tampon des flux suivants aussi)
            ecrFic.getFD().sync() ;
            ecr.close() ;
            ecr = null ;
            renommer (ficConfigTmp, ficConfig, true) ;
        }
        catch (Throwable e)
        {
            // Essayer de fermer le fichier, mais sans masquer l'erreur d'origine
            if (ecr != null)
            {
                try   { ecr.close() ; }
                catch (Throwable e2) {}
            }
            
            // Faire suivre l'exception d'origine
            throw new IOException ("Erreur lors de la sauvegarde du fichier \"" + titre + "\" (" + ficConfig.getAbsolutePath() + ")", e) ;
        }
    }
    
}
