/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.utiles.es.fichiers;

import java.io.BufferedReader ;
import java.io.File;
import java.io.FileInputStream ;
import java.io.FileOutputStream ;
import java.io.IOException ;
import java.io.InputStreamReader ;
import java.io.OutputStreamWriter ;
import java.io.PrintWriter ;

import javax.swing.JFileChooser;
import javax.swing.JFrame;



// Classe utilitaire permettant de réencoder un fichier dans un autre encodage (pour des fichiers de Patrice).
public class Reencodeur
{

    // Réencode un fichier dans un nouvel encodage.
    // !!! ATTENTION, c'est un utilitaire vite fait :
    //      - Pas de validation du fichier sur le disque (tampon système) et réécriture dans le fichier
    //        lui-même : si un problème survient à ce moment, les données risquent d'être perdues ;
    //        globalement, les cas d'erreur ne sont pas correctement traités
    //      - Pour des caractères "exotiques", je ne sais pas ce que donne le fait de lire des char un par
    //        un (certains caractères devraient être étalés sur deux caractères ; enfin ce n'est sans doute
    //        pas gênant ici (?))
    //      - Le fichier est intégralement chargé puis écrasé, ça peut être gênant s'il est très gros
    
    // amélioration
    // les fichiers destination et source sont donnés par l'utilisateur (pas finalisé)
    // un test pour vérifier la validité du UTF-8. Mon vrai problème est de vérifier à quel endroit un fichier
    // ne respecte plus le codage utf-8 (la conversion utf-8 de PsPad semble bugguée)
    
    public static void reencoderFic (String source, String encodageSource, String destination, String encodageDestination) throws IOException
    {
        System.out.print ("Réencodage du fichier " + source + " (" + encodageSource + ") vers " +destination +"' ("+ encodageDestination + ") ... ") ;
        //on doit d'abord vérifier que le fichier source n'est pas déjà en utf-8
        if (testUTF8(source))
            {
                return;
            }
        
        // Lecture du fichier
        StringBuilder ch = new StringBuilder() ;
        BufferedReader lect = new BufferedReader (new InputStreamReader (new FileInputStream (source), encodageSource)) ;
        int c ;
        while ((c = lect.read()) != -1)
            ch.append ((char)c) ;
        lect.close() ;
        
        //test encodage
        //on affiche le texte et on regarde
        
        
        // Réécriture du fichier
        PrintWriter ecr = new PrintWriter (new OutputStreamWriter (new FileOutputStream (destination), encodageDestination)) ;
        ecr.print (ch.toString()) ;
        ecr.close() ;
        
        System.out.println ("terminé.") ;
    }
    
// Une fonction pour vérifier que le code UTF-8 est valide    
    public static boolean testUTF8 (String source)throws IOException
    {
/*
 * les séquences de caractères valides en utf-8 sont les suivantes (source : http://fr.wikipedia.org/wiki/UTF-8)
 * Les caractères UTF-8 sont codés sur 1 à 4 octets
 * Un octet peut à priori avoir les valeurs 00 à FF. La table suivante permet le décodage.
 * Elle s'écrit sous la forme 
 * <valeur premier octet> : <nombre d'octets composant le caractère>||interdit
 * on note octet0, le premier octet, octet1, octet2 et octet3 les autres
 * premier caractère
 * 00 à 7F : 1 octet (équivalent ASCII), l'octet suivant commence un nouveau caractère
 * 80 à C1 : interdit 
 * C2 à DF : 2 octets, octet1 entre 80 et BF
 * E0      : 3 octets, octet1 entre A0 et BF, octet2 entre 80 et BF
 * E1 à EC : 3 octets, octet1 et octet2 entre 80 et BF
 * ED      : 3 octets, octet1 entre 80 et 9F, octet2 entre 80 et BF
 * EE & EF : 3 octets, octet1 et octet2 entre 80 et BF
 * F0      : 3 octets, octet1 entre 90 et BF, octet2 et octet3 entre 80 et BF
 * F1 à F3 : 3 octets, octet1,octet2,octet3 entre 80 et BF
 * F4      : 3 octets, octet1 entre 80 et 8F, octet2 et octet3 entre 80 et BF     
 * F5 à FF : interdit
 * Tout autre suite de code est interdite       
 */
        System.out.println("On teste le fichier : " + source);
        FileInputStream lect = new FileInputStream (source);
        
        //BufferedReader lect = new BufferedReader (new InputStreamReader (new FileInputStream (source), "UTF-8")) ;
        int c ;
        int n = 0;
        boolean erreur=false;
        boolean sortie=false;

        while (!(sortie||erreur))
        {    
            
            c=lect.read ();
            sortie=(c==-1);
            if (!sortie)
            {//on teste toutes les valeurs de c (voir ci dessus)
                //System.out.print (hexa(c)+" ");
                System.out.print((char)c);
                n++; //position du caractère
                if (c<0x80)//caractère sur 1 octet
                {
                    //OK, octet ASCII
                }
                else if (c<0xC2)//80 à C1, valeurs interdites
                {
                    System.out.println("Erreur sur le caractère " + hexa (n) +" de valeur interdite : " + hexa (c));
                    erreur=true;
                }
                else if (c<0xE0)//C2 à DF, caractère sur 2 octets, octet suivant entre 80 et BF
                {
                    erreur=!lireSuivant1(lect,0x7F,0xC0);
                    if (erreur)
                    {
                        System.out.println("Erreur sur le caractère " + hexa (n)+" de valeur : " + hexa (c) + " (l'octet suivant est de valeur interdite)");
                    }
                    else n++;
                }
                else if (c<0xF0)//E0 à EF, caractère sur 3 octets,
                {
                    erreur=!lireSuivant2(lect,c);
                    if (erreur)
                    {
                        System.out.println("\nErreur sur le caractère " + hexa (n) +" de valeur : " + hexa (c) + " (l'un des 2 octets suivants est de valeur interdite)");
                    }
                    else n=n+2;
                }
                else if (c<0xF5)
                {
                    erreur=!lireSuivant3(lect,c);
                    if (erreur)
                    {
                        System.out.println("Erreur sur le caractère " + hexa (n) +" de valeur : " + hexa (c) + " (l'un des 3 octets suivants est de valeur interdite)");
                    }
                    else n=n+3;
                }
                else
                {
                    System.out.println("Erreur sur le caractère " + hexa (n) +" de valeur interdite : " + hexa (c));
                    erreur=true;
                }
            }
        }
        lect.close() ;
        return !(erreur);
    }

    private static boolean lireSuivant1(FileInputStream fichier, int mini, int maxi) throws IOException
//On lit un octet dans le fichier, il doit être strictement entre mini et maxi (]mini,maxi[)
//La fonction renvoie true si l'octet est bien entre mini et maxi et false sinon. 
//n est le numéro du caractère lu
    {
        int c1 ;
        c1=fichier.read ();
        return((c1>mini)&&(c1<maxi));
    }
    
    private static boolean lireSuivant2(FileInputStream fichier, int c) throws IOException
/*
 * On lit deux octets dans le fichier
 * c est le caractère précédent, en fonction de sa valeur certaines conditions supplémentaires
 * existent (voir le code)
 * La fonction renvoie true si les octets vérifient les conditions et false sinon.
 */      
    {
        switch(c)
        {
            case 0xE0 ://premier octet entre A0 et BF, suivant entre 80 et BF
                return (lireSuivant1(fichier,0x9F,0xC0)? lireSuivant1(fichier,0x7F,0xC0):false);
            case 0xED ://premier octet entre 80 et 9F, suivant entre 80 et BF
                return (lireSuivant1(fichier,0x7F,0xA0)? lireSuivant1(fichier,0x7F,0xC0):false);
            default:   //les deux octets entre 80 et BF 
                return lire2octets(fichier);
        }
    }

    private static boolean lireSuivant3(FileInputStream fichier, int c) throws IOException
/*
 * On lit trois octets dans le fichier
 * c est le caractère précédent, en fonction de sa valeur certaines conditions supplémentaires
 * existent (voir le code)
 * La fonction renvoie true si les octets vérifient les conditions et false sinon.
 */      
    {
        switch(c)
        {
            case 0xF0 ://premier octet entre 90 et BF, suivants entre 80 et BF
                return (lireSuivant1(fichier,0x8F,0xC0)? lire2octets(fichier):false);
            case 0xF4 ://premier octet entre 80 et 8F, suivant entre 80 et BF
                return (lireSuivant1(fichier,0x7F,0x90)? lire2octets(fichier):false);
            default:   //les trois octets entre 80 et BF 
                return (lireSuivant1(fichier,0x7F,0xC0)? lire2octets(fichier):false);
        }
    }

    private static boolean lire2octets(FileInputStream fichier) throws IOException
    //lire deux octets entre 80 et BF
    {
        return (lireSuivant1(fichier,0x7F,0xC0)? lireSuivant1(fichier,0x7F,0xC0):false);
        
    }
    
    private static String hexa(int n)
    {
        return Integer.toHexString (n).toUpperCase ();
    }
    
    // Méthode principale
    public static void main (String[] args) throws IOException
    {
    
        JFrame frame = new JFrame("choix d'un fichier");
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        JFileChooser fichier = new JFileChooser("D:" + File.separator + "www" + File.separator + "Histoire Mondiale" + File.separator + "java");
        //frame.add(fichier);
        //frame.pack();
        //frame.setVisible(true);
        int retour=fichier.showOpenDialog(frame);
        if (retour==JFileChooser.APPROVE_OPTION )
            {
                //System.out.println("Vous avez choisi" + fichier.getSelectedFile ().getName ());
                String nomFichier =  fichier.getSelectedFile ().getName ();   
                System.out.println("\n" + nomFichier + " est un fichier UTF-8 " +
                		((testUTF8("D:" + File.separator + "www" + File.separator + "Histoire Mondiale" + File.separator + "java" + File.separator + "" + nomFichier))? "valide":"non valide")); 
                     
            }
        //reencoderFic ("D:" + File.separator + "www" + File.separator + "Histoire Mondiale" + File.separator + "java" + File.separator + "index.html", "Cp1252", "D:" + File.separator + "www" + File.separator + "Histoire Mondiale" + File.separator + "java" + File.separator + "index.html.utf8", "UTF-8") ;
    }
    
}
