/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 * 
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou le modifier sous les
 * termes de la licence Affero GPL publiée par la Fondation pour le logiciel libre (Free
 * Software Foundation), en choisissant la version 3 de cette licence ou n'importe quelle
 * version ultérieure, à votre convenance.
 * 
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE D'AUCUNE SORTE
 * : y compris d'être vendable ou de pouvoir servir un but donné. Voir le texte de la
 * licence AGPL pour plus de détails.
 * 
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire. Si ce n'est pas le
 * cas, regardez à cette adresse : <http://www.gnu.org/licenses/>.
 * 
 */
package fr.histoiremondiale.histoire.utiles.igraphique.swing;

import java.awt.Component;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridLayout;

import javax.swing.Icon;
import javax.swing.JFrame;
import javax.swing.JLabel;

/**
 * TexteVertical écrit une chaine de caractère verticalement en tournant les caractères
 * de 90° à gauche ou à droite, ou en les écrivant l'un sous l'autre. Le texte ainsi
 * écrit est transformé en image.
 * On peut l'utiliser pour un JTabbedPane avec des onglets à droite ou à gauche ou 
 * encore avec n'importe quel composant acceptant des images (JLabel ou JButton par
 * exemple).
 */

public class TexteVertical implements Icon
{
	//les constantes publiques
	/**
	 * pas de rotation
	 */
	public static final int PAS_DE_ROTATION = 0;
	
	/**
	 * rotation de 90°
	 */
	public static final int ROTATION = 1;
	
	/**
	 * texte écrit de bas en haut
	 */
	public static final int ASCENDANT = -1;
	
	/**
	 * texte écrit de haut en bas
	 */
	public static final int DESCENDANT = 1;
	
	//les constantes privées
	private static final double ANGLE_DROIT = Math.PI/2;
	//pour mettre une marge au texte 
	private static final int MARGE = 2;
	
	//les attributs
	private String texte;				//le texte à écrire
	private Component composant;		//le composant qui recevra le texte écrit verticalement
	private int rotation;				//la rotation des caractères
	private int sens;					//le sens d'écriture des caractères (bas vers haut ou le contraire)
	private int hauteurFonte;			//la hauteur des caractères
	private String caracteres[];		//les caractères du texte
	private int largeur;				//la largeur du texte (celle du caractère le plus large)
	private int largeurCaracteres[];	//la largeur de chaque caractère
	private int descendante;			//la hauteur sous la ligne de base pour la fonte utilisée 
	private int ascendante;				//la hauteur au dessus de la ligne de base
	private int hauteur;				//la hauteur du texte
	
	/**
	 * Créer un TexteVertical à partir de la chaîne de caractères texte, pour le composant component,
	 * et d'orientaion donnée. 
	 * @param composant le composant concerné
	 * @param texte le texte à écrire
	 * @param rotation la rotation des caractères (0, pas de rotation, différent de 0, rotation)
	 * @param sens le sens d'écriture (1 du bas vers le haut, -1 du haut vers le bas) 
	 */
 	public TexteVertical(Component composant, String texte, int rotation, int sens) 
 	{
		this.texte = texte;
		this.composant = composant;
		this.rotation = rotation;
		if (sens == 0)
		{
			throw new IllegalArgumentException("le sens doit être non nul (\n" +
											   "\tnégatif pour une écriture du bas vers le haut\n" +
											   "\tposistif pour une écriture du haut evrs le bas") ;
		}
		this.sens = sens;
 		if (rotation == TexteVertical.PAS_DE_ROTATION) calculDimensionSansRotation();
 		else calculDimensionAvecRotation();
	}

	private void calculDimensionAvecRotation()
	{
		FontMetrics fm = this.composant.getFontMetrics(this.composant.getFont());
		//la hauteur maxi des caractères en pixels
		this.hauteurFonte = fm.getHeight();
		//le nombre de pixels sous la ligne de base
		this.descendante = fm.getDescent();
		//le nombre de pixels au dessus de la ligne de base 
		this.ascendante = fm.getAscent() + fm.getLeading();
		//en cas de rotation, la largeur est la hauteur de la fonte
		this.largeur = this.hauteurFonte;
		//et la hauteur est la largeur du texte + une marge avant et après
		this.hauteur = fm.stringWidth(this.texte);
	}

	private void calculDimensionSansRotation()
	{
		FontMetrics fm = this.composant.getFontMetrics(this.composant.getFont());
		//la hauteur maxi des caractères en pixels
		this.hauteurFonte = fm.getHeight();
		//le nombre de pixels sous la ligne de base
		this.descendante = fm.getDescent();
		//le nombre de pixels au dessus de la ligne de base 
		this.ascendante = fm.getAscent() + fm.getLeading();
		//sans rotation, les caratères sont écrits les uns en dessous des autres
		String s = (this.sens < 0 ? new StringBuffer(this.texte).reverse().toString() : this.texte);
		//le tableau des caractères de la chaine
		int nbChar = this.texte.length();
		char data[] = new char[nbChar];
		s.getChars(0, nbChar, data, 0);
		//sans rotation la largeur est la largeur du caractère le plus large
		this.largeur = 0;
		//pour le dessin du texte on aura besoin des caractères 
		this.caracteres = new String[nbChar];
		//et de leur largeur
		this.largeurCaracteres = new int[nbChar];
		//le caractère courant
		char ch;
		for (int i = 0; i < nbChar; i++) 
		{
			ch = data[i];
			this.largeurCaracteres[i] = fm.charWidth(ch);
			//la largeur c'est la largeur du plus grand caractère
			if (this.largeurCaracteres[i] > this.largeur) this.largeur = this.largeurCaracteres[i];
			this.caracteres[i] = new String(data, i, 1);				
		}
		//la hauteur c'est la hauteur d'un caractère multipliée par le nombre de caractères
		this.hauteur = this.hauteurFonte * nbChar;
	}

	/**
     * Peindre le texte (sous forme d'une image) à l'endroit spécifié
     */
    public void paintIcon(Component c, Graphics g, int x, int y) 
    {
    	//transformer g en Graphics2D
    	Graphics2D g2 = (Graphics2D) g;
	    //couleur d'écriture
    	g2.setColor(c.getForeground());
    	y += MARGE;
    	
    	if (this.rotation == TexteVertical.PAS_DE_ROTATION) peindreLettresSansRotation(g2,x,y);
    	else peindreLettresAvecRotation(g2,x,y);
    	
	}
 	
	private void peindreLettresAvecRotation(Graphics2D g2, int x, int y)
	{
		double angle;
		if (this.sens < 0)
		{
			x += this.ascendante;
			y += this.hauteur;
			angle = - TexteVertical.ANGLE_DROIT;
		}
		else
		{
			x += this.descendante;
			angle = TexteVertical.ANGLE_DROIT;
		}
		g2.rotate(angle, x, y);
	    g2.drawString(this.texte, x, y);
	    g2.rotate(-angle, x, y);

	}

	//les lettres sont l'une au dessous de l'autre, du haut vers le bas ou dans l'autre sens
	private void peindreLettresSansRotation(Graphics2D g2, int x, int y)
	{
		int yPos = y + this.ascendante ;
		for (int i = 0; i < this.caracteres.length; i++) 
		{
			//les caractères sont centrés.  
			g2.drawString(this.caracteres[i], x+((this.largeur-this.largeurCaracteres[i])/2), yPos);
			yPos += this.hauteurFonte;
		}
	}

	/**
	 * @see javax.swing.Icon#getIconHeight()
	 */
	public int getIconHeight() 
	{
		return (this.hauteur + 2 * MARGE) ;
	}

	/**
	 * @see javax.swing.Icon#getIconWidth()
	 */
	public int getIconWidth() 
	{
		return this.largeur;
	}
	
	
	/**
	 * une méthode de test
	 * @param args les argument éventuels
	 */
	public static void main(String[] args)
	{
		JFrame fenetre = new JFrame("essai de texte vertical");
		fenetre.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		fenetre.setLayout(new GridLayout(1,4));
		String s = "babcdefghijklmnop";
        //pas de rotation, haut en bas
		JLabel label1 = new JLabel();
		TexteVertical texte1 = new TexteVertical(label1,s,TexteVertical.PAS_DE_ROTATION,TexteVertical.DESCENDANT);
		label1.setIcon(texte1);
        fenetre.getContentPane().add(label1);

        //pas de rotation, bas en haut
		JLabel label2 = new JLabel();
		TexteVertical texte2 = new TexteVertical(label2,s,TexteVertical.PAS_DE_ROTATION,TexteVertical.ASCENDANT);
		label2.setIcon(texte2);
        fenetre.getContentPane().add(label2);

        //rotation, bas en haut
		JLabel label3 = new JLabel();
		TexteVertical texte3 = new TexteVertical(label3,s,TexteVertical.ROTATION,TexteVertical.DESCENDANT);
		label3.setIcon(texte3);
        fenetre.getContentPane().add(label3);

        //rotation, haut en bas
		JLabel label4 = new JLabel();
		TexteVertical texte4 = new TexteVertical(label4,s,TexteVertical.ROTATION,TexteVertical.ASCENDANT);
		label4.setIcon(texte4);
        fenetre.getContentPane().add(label4);
        
        fenetre.pack();
        fenetre.setVisible(true);
	}
}
