/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.utiles.igraphique.swing.gestplacement;

import java.awt.Component ;
import java.awt.Container ;
import java.awt.Dimension ;
import java.awt.LayoutManager ;
import java.awt.LayoutManager2 ;
import java.util.HashMap ;
import java.util.Map ;



/**
 * Gestionnaire de placement de composants gérant une liste horizontale ou verticale de composants.<br>
 * Ce gestionnaire de placement ressemble à ce que sont censées faire BoxLayout et le conteneur Box,
 *   mais en plus simple. Box et BoxLayout sont trop capricieux (par exemple sur une boîte verticale tant
 *   qu'elle contenait des JLabel ça allait (ils étaient alignés à gauche), mais quand a été ajouté un
 *   composant perso contenant une image et ayant une taille préférée, une taille minimale et une taille
 *   maximale les composant se sont tous trouvés alignés bizarrement (BoxLayout tente de faire des
 *   alignements sur je sais pas quoi apparemment). Et suivant si le composant avait ou non une taille
 *   minimale et avait ou non une taille maximale, le résultat était très différent. BoxLayout semble
 *   se baser sur cette information pour décider comment placer les composants, selon un rituel mystérieux.
 */
public class PlaceurListe implements LayoutManager, LayoutManager2
{
    
    // Constantes
    // (sens de placement)
    public static final int COLONNE = 1 ;
    public static final int LIGNE   = 2 ;
    // (alignements - note : des entiers longs surtout pour éviter que l'appel à Container.add ne choisisse la mauvaise méthode)
    public static final long ALIGN_CENTRE =  1 ;        // Centré
    public static final long ALIGN_GAUCHE =  2 ;        // A gauche (sens : colonne)
    public static final long ALIGN_DROITE =  4 ;        // A droite (sens : colonne)
    public static final long ALIGN_HAUT   =  8 ;        // En haut  (sens : ligne)
    public static final long ALIGN_BAS    = 16 ;        // En bas   (sens : ligne)

    
    // Données
    private int                 sensPlacement ;
    private long                alignementParDefaut ;   // Alignement par défaut des composants
    private Map<Component,Long> contraintes ;           // Contraintes sur les composants
    
    
    
    /**
     * Constructeur.
     * @param sensPlacement Le sens de placement (horizonal ou vertical).
     */
    public PlaceurListe (int sensPlacement)
    {
        this (sensPlacement, ALIGN_GAUCHE + ALIGN_HAUT) ;
    }
    
    /**
     * Constructeur.
     * @param sensPlacement       Le sens de placement (horizonal ou vertical, lignes ou colonnes).
     * @param alignementParDefaut Alignement des composants (haut, bas, gauche, droite ; combinaison possible).
     */
    public PlaceurListe (int sensPlacement, long alignementParDefaut)
    {
        modifSensPlacement (sensPlacement) ;
        this.alignementParDefaut = alignementParDefaut ;
        this.contraintes         = new HashMap<>() ;
    }
    
    
    /**
     * Modifie le sens de placement des composants.
     * @param sensPlacement Le sens de placement (horizonal ou vertical).
     */
    public void modifSensPlacement (int sensPlacement)
    {
        this.sensPlacement = sensPlacement ;
    }
    

    
    // Mémorise les contraintes associées à un composant
    // Note : les contraintes peuvent ne pas correspondre au sens. Par exemple on peut indiquer qu'un
    //   composant est aligné à gauche et en haut. Si l'organisation est en colonne, il sera aligné
    //   à gauche, si elle est en ligne il sera aligné en haut. Ce n'est pas grave (si on utilise une
    //   variable pour configurer un ensemble de valeurs) et ça peut servir si on moifie le sens du
    //   gestionnaire de composants
    public void addLayoutComponent (Component comp, Object contraintes)
    {
        if (contraintes != null)
            this.contraintes.put (comp, (Long)contraintes) ;
    }

    // Indique qu'un composant ne fait plus partie du conteneur
    public void removeLayoutComponent (Component comp)
    {
        this.contraintes.remove (comp) ;
    }
    
    
    // Organiser les composants du conteneur parent
    public void layoutContainer (Container parent)
    {
        // Composants en colonne
        if (this.sensPlacement == COLONNE)
        {
            int yCourant = 0 ;
            for (Component comp : parent.getComponents())
            {
                long contrainte = (this.contraintes.containsKey(comp) ? this.contraintes.get (comp) : this.alignementParDefaut) ;
                int xComp ;

                // Calculer la taille du composant
                comp.setSize (comp.getPreferredSize()) ;
                
                // Calculer l'alignement du composant
                // (par défaut à gauche (voir les notes : il se peut qu'on ait exprimé des contraintes que pour l'organsiation en ligne))
                if      ((contrainte & ALIGN_DROITE) != 0) xComp = parent.getWidth() - comp.getWidth() ;
                else if ((contrainte & ALIGN_CENTRE) != 0) xComp = (parent.getWidth() - comp.getWidth()) / 2 ;
                else                                       xComp = 0 ;
                
                // Positionner le composant
                comp.setLocation (xComp, yCourant) ;
                
                // Passer au suivant
                yCourant += comp.getHeight() ;
            }
        }
        // Composants en ligne
        else if (this.sensPlacement == LIGNE)
        {
            int xCourant = 0 ;
            for (Component comp : parent.getComponents())
            {
                long contrainte = (this.contraintes.containsKey(comp) ? this.contraintes.get (comp) : this.alignementParDefaut) ;
                int yComp ;
                
                // Calculer la taille du composant
                comp.setSize (comp.getPreferredSize()) ;

                // Calculer l'alignement du composant
                // (par défaut en haut (voir les notes : il se peut qu'on ait exprimé des contraintes que pour l'organsiation en ligne))
                if      ((contrainte & ALIGN_BAS)    != 0) yComp = parent.getHeight() - comp.getHeight() ;
                else if ((contrainte & ALIGN_CENTRE) != 0) yComp = (parent.getHeight() - comp.getHeight()) / 2 ;
                else                                       yComp = 0 ;
                
                // Positionner le composant
                comp.setLocation (xCourant, yComp) ;

                // Passer au suivant
                xCourant += comp.getWidth() ;
            }
        }
        else
        {
            exceptionSensInconnu() ;
        }
    }

    
    // Calculer la taille minimale du contenu du conteur parent
    public Dimension minimumLayoutSize (Container parent)
    {
        int largeurMin = 0 ;
        int hauteurMin = 0 ;

        
        // Composants en colonne
        if (this.sensPlacement == COLONNE)
        {
            for (Component comp : parent.getComponents())
            {
                Dimension tailleComp = comp.getMinimumSize() ;
                largeurMin  = (int) Math.max (largeurMin, tailleComp.getWidth()) ;
                hauteurMin += tailleComp.getHeight() ;
            }
        }
        // Composants en ligne
        else if (this.sensPlacement == LIGNE)
        {
            for (Component comp : parent.getComponents())
            {
                Dimension tailleComp = comp.getMinimumSize() ;
                largeurMin += tailleComp.getWidth() ;
                hauteurMin  = (int) Math.max (hauteurMin, tailleComp.getHeight()) ;
            }
        }
        else
        {
            exceptionSensInconnu() ;
        }
        
        // Renvoyer la taille calculée
        return new Dimension (largeurMin, hauteurMin) ;
    }
    
    
    // Calculer la taille maximale du contenu du conteur parent
    public Dimension maximumLayoutSize (Container parent)
    {
        int largeurMax = 0 ;
        int hauteurMax = 0 ;

        
        // Composants en colonne
        if (this.sensPlacement == COLONNE)
        {
            for (Component comp : parent.getComponents())
            {
                Dimension tailleComp = comp.getMaximumSize() ;
                largeurMax  = (int) Math.max (largeurMax, tailleComp.getWidth()) ;
                hauteurMax += tailleComp.getHeight() ;
            }
        }
        // Composants en ligne
        else if (this.sensPlacement == LIGNE)
        {
            for (Component comp : parent.getComponents())
            {
                Dimension tailleComp = comp.getMaximumSize() ;
                largeurMax += tailleComp.getWidth() ;
                hauteurMax  = (int) Math.max (hauteurMax, tailleComp.getHeight()) ;
            }
        }
        else
        {
            exceptionSensInconnu() ;
        }
        
        // Renvoyer la taille calculée
        return new Dimension (largeurMax, hauteurMax) ;
    }

    
    // Calculer la taille préférée du contenu du conteur parent
    public Dimension preferredLayoutSize (Container parent)
    {
        int largeurPref = 0 ;
        int hauteurPref = 0 ;

        
        // Composants en colonne
        if (this.sensPlacement == COLONNE)
        {
            for (Component comp : parent.getComponents())
            {
                Dimension tailleComp = comp.getPreferredSize() ;
                largeurPref  = (int) Math.max (largeurPref, tailleComp.getWidth()) ;
                hauteurPref += tailleComp.getHeight() ;
            }
        }
        // Composants en ligne
        else if (this.sensPlacement == LIGNE)
        {
            for (Component comp : parent.getComponents())
            {
                Dimension tailleComp = comp.getPreferredSize() ;
                largeurPref += tailleComp.getWidth() ;
                hauteurPref  = (int) Math.max (hauteurPref, tailleComp.getHeight()) ;
            }
        }
        else
        {
            exceptionSensInconnu() ;
        }
        
        // Renvoyer la taille calculée
        return new Dimension (largeurPref, hauteurPref) ;
    }
    
    
    // Le conteneur de ce placeur souhaiterait être aligné à gauche et en haut
    public float getLayoutAlignmentX (Container target)
    {
        return 0 ;
    }
    // Le conteneur de ce placeur souhaiterait être aligné à gauche
    public float getLayoutAlignmentY (Container target)
    {
        return 0 ;
    }

    
    
    // Lève une exception indiquant que le sens de placement n'est pas reconnu
    private void exceptionSensInconnu ()
    {
        throw new IllegalStateException ("Le sens de placement n'est pas reconnu : " + this.sensPlacement) ;
    }

    
    
    // Méthodes non implémentées
    // (ne rien faire : le placeur n'est pas intéressé par ces informations)
    public void addLayoutComponent (String name, Component comp)
    {}
    public void invalidateLayout (Container target)
    {}
    
}
