/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique;

import java.awt.BorderLayout ;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.event.ActionEvent ;
import java.awt.event.ActionListener ;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;

import javax.swing.JButton ;
import javax.swing.JCheckBox;
import javax.swing.JDialog ;
import javax.swing.JFrame ;
import javax.swing.JPanel ;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import fr.histoiremondiale.histoire.utiles.igraphique.swing.Fenetres;
import fr.histoiremondiale.histoire.utiles.math.CoordSphere;
import fr.histoiremondiale.histoire.utiles.math.Matrice;
import fr.histoiremondiale.histoire.utiles.math.PointPlan;
import fr.histoiremondiale.histoire.utiles.math.PointSphere;
import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.donnees.Territoire;
import fr.histoiremondiale.histoire.igraphique.actions.ActionsEspaceTemps;
import fr.histoiremondiale.histoire.igraphique.composants.BarreLoupe;
//import fr.histoiremondiale.histoire.utiles.math.CoordSphere;
//import fr.histoiremondiale.histoire.utiles.math.Matrice;
//import fr.histoiremondiale.histoire.utiles.math.PointPlan;
//import fr.histoiremondiale.histoire.utiles.math.PointSphere;



/**
 * Représente une boîte de dialogue modale permettant de modifier la valeur de loupe à partir
 *   de laquelle les infos d'un territoire s'affichent.
 */
public class DialLimiteVisibiliteTerritoire extends JDialog implements ActionListener, WindowListener, ChangeListener
{

    private Territoire  territoire ;                // Territoire pour lequel on effectue le réglage
    private BarreLoupe  barre ;                     // La barre
    private JCheckBox   cc_territoireTjsVisible ;   // Coche permettant d'activer/désactiver la visibilité permanent des infos du territoire
    private JPanel      sud ;                       // Le panneau sud
    private JPanel      boutons ;                   // Le panneau contenant les boutons
    private JButton     btnOK ;                     // Le bouton pour valider
    private JButton     btnAnnul ;                  // Le bouton pour annuler
    private double      valeurInitiale ;            // Valeur de la loupe limite avant l'ouverture de la boîte de dialogue
    private double      loupe ;                     // La valeur d'agrandissement de la carte
    
    
    
    /**
     * Constructeur.
     * @param parent          Fenêtre parent.
     * @param territoireParam Territoire concerné.
     * @param mini            Valeur minimale.
     * @param maxi            Valeur maximale.
     * @param valeur          Valeur initiale.
     * @param loupe           Valeur de la loupe.
     */
    public DialLimiteVisibiliteTerritoire (JFrame parent, Territoire territoireParam, double mini, double maxi, double valeur, double loupe)
    {
        super (parent, territoireParam.nomNormalise(), true) ;
        
        this.valeurInitiale = valeur ;
        this.territoire     = territoireParam ;
        this.loupe          = loupe ;
        
        //les panneaux
        this.sud     = new JPanel() ;
        this.boutons = new JPanel() ;
        
        //les composants
        //La barre
        this.barre=new BarreLoupe(mini, maxi, valeur, loupe);  
        //ajouter une case à cocher pour rendre l'info toujours visible
        this.cc_territoireTjsVisible = new JCheckBox ("Toujours visible") ;
        this.btnOK = new JButton("OK") ;
        this.btnAnnul = new JButton("Annuler") ;
        
        //Construction du panneau boutons par ajout des deux boutons
        this.boutons.add (this.btnOK);
        this.boutons.add (this.btnAnnul) ;
        
        //Construction du bas du composant par ajout de la case à cocher et des boutons
        this.sud.setLayout (new BorderLayout()) ;
        this.sud.add (this.cc_territoireTjsVisible,BorderLayout.NORTH);
        this.sud.add (this.boutons,BorderLayout.SOUTH);
        
        //Construction du composant par ajout de la barre et du bas du composant.
        setLayout (new BorderLayout() ) ;        
        add (this.barre,BorderLayout.CENTER);
        add (this.sud, BorderLayout.SOUTH) ;
        
        //Initialisation de la case à cocher
        this.cc_territoireTjsVisible.setSelected (valeur == 0) ;
        
        //Les écouteurs
        this.barre.addChangeListener(this) ;        
        this.btnOK.addActionListener (this) ;
        this.btnAnnul.addActionListener(this) ;
        this.cc_territoireTjsVisible.addActionListener(this) ;
        addWindowListener(this) ;
        
        // Ajuster la taille de la boîte de dialogue pour permettre à tous les composants de s'afficher correctement
        Dimension dim = this.getMinimumSize() ;
        int largeur = Math.max (dim.width, largeur(this.getTitle()) + 80 ) ;
        largeur = (int) Math.max (largeur, this.cc_territoireTjsVisible.getMinimumSize().getWidth()) ;
        largeur = (int) Math.max (largeur, this.btnOK.getMinimumSize ().getWidth () + this.btnAnnul.getMinimumSize ().getWidth () + 30) ;
        
        int hauteur = dim.height ;
        //La taille minimum doit permettre l'affichage du titre
        setMinimumSize(new Dimension(largeur, hauteur) );
        this.setPreferredSize (getMinimumSize());
        //Le layout s'occupe de placer tous les composants internes de ce JDialog et de dimensionner celui-ci
        pack () ;
        
		Fenetres.ajouterEcouteurEchapPourMasquer(this);
		Fenetres.centrer(this,parent);
    }
    
    
    /**
     * Positionne la fenêtre permettant de choisir l'agrandissement limite d'un territoire.
     */
    public void positionner ()
    {
        //le centre du texte (coordonnées géographiques)
        PointSphere ptCentreTexteSphere = HistoireMondiale.instance ().donneesIGraphique ().centreTexteTerritoire (this.territoire) ;        
        //le centre de la carte (coordonnées de l'écran)
        PointSphere ptCentrePlan = HistoireMondiale.instance ().etat ().ptCentralCarte ();
        
        PanCarte carte  = HistoireMondiale.instance ().fenPrincipale ().p_carte ;
        //top et left, largeur et hauteur
        int hautCarte   = carte.getLocationOnScreen ().y ,
            gaucheCarte = carte.getLocationOnScreen ().x ,
            largeurCarte = carte.getWidth (),
            hauteurCarte = carte.getHeight (),
            a = Math.max (hauteurCarte, largeurCarte);
        Matrice matrice = CoordSphere.matChgtSphereVersPlan2 (ptCentrePlan) ;
        PointPlan ptPlan = CoordSphere.projection (ptCentreTexteSphere, matrice, this.loupe, a);
        //Il reste à déterminer où placer le JDialog pour ne pas gêner
        int gauche = (ptPlan.x () > 0) ? gaucheCarte : gaucheCarte + largeurCarte - this.getWidth (),
            haut   = (ptPlan.y () > 0) ? hautCarte + hauteurCarte - this.getHeight () : hautCarte  ;
        this.setLocation (gauche, haut) ;
    }


    private int largeur (String titre)
    {
        FontMetrics fm = getFontMetrics(getFont()) ;
        return fm.stringWidth(titre) ;
    }

    /**
     * Traitement des actions sur la boîte de dialogue.
     */
    public void actionPerformed (ActionEvent e)
    {
        if (e.getSource() == this.btnOK)
        {
            double valeur = (this.cc_territoireTjsVisible.isSelected () ? 0 : this.barre.getValeur ()) ;
            ActionsEspaceTemps.modifierLoupeTerritoire (this.territoire, valeur) ;
            dispose();
         }
        else if (e.getSource() == this.btnAnnul)
        {
            //La valeur retournée est la valeur initiale
            //cc_territoireTjsVisible.setSelected (valeur == 0) ;
            ActionsEspaceTemps.modifierLoupeTerritoire (this.territoire, this.valeurInitiale) ;
            dispose() ;
        }
        else if (e.getSource () == this.cc_territoireTjsVisible)
        {
            if (this.cc_territoireTjsVisible.isSelected())
            {
                this.barre.setValeur (0) ;
                ActionsEspaceTemps.modifierLoupeTerritoire (this.territoire, 0) ;
            }
         }
        else   
        { 
            throw new IllegalStateException ("Source inconnue : " + e.getSource()) ; 
        }

    }


    /**
     * Traitements effectués quand la boîte de dialogue est fermée.
     */
    public void windowClosing (WindowEvent arg0)
    {
        ActionsEspaceTemps.modifierLoupeTerritoire (this.territoire, this.valeurInitiale) ;
        dispose() ;
    }
    /* 
     * Méthodes non implémentées
     */
    public void windowOpened        (WindowEvent arg0)  {}
    public void windowIconified     (WindowEvent arg0)  {}
    public void windowDeiconified   (WindowEvent arg0)  {}
    public void windowDeactivated   (WindowEvent arg0)  {}
    public void windowActivated     (WindowEvent arg0)  {}
    public void windowClosed        (WindowEvent arg0)  {}


    /* (non-Javadoc)
     * @see javax.swing.event.ChangeListener#stateChanged(javax.swing.event.ChangeEvent)
     */
    public void stateChanged (ChangeEvent e)
    {
        this.cc_territoireTjsVisible.setSelected (false);
        ActionsEspaceTemps.modifierLoupeTerritoire (this.territoire, this.barre.getValeur ()) ;
    }


}


