/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique.composants;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.GradientPaint;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.geom.Rectangle2D;
import java.text.DecimalFormat;

import javax.swing.BorderFactory;
import javax.swing.JFormattedTextField;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.border.EtchedBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.plaf.metal.MetalLookAndFeel;

import fr.histoiremondiale.histoire.utiles.exttypes.Flottants;
import static fr.histoiremondiale.histoire.utiles.math.PlagesNombres.dansPlage;


//import gestionbase.outils.math.Flottants;
//import static gestionbase.outils.math.PlagesNombres.dansPlage;

/**
 * Une barre qui représente une valeur v comprise entre deux bornes mini et maxi.
 * Cette barre est un JPanel<br>
 * 
 * On dispose d'une représentation graphique sous la forme d'une barre verticale
 * bicolore (par exemple rouge et blanc), la partie rouge est proportionnelle à  
 * valeur-mini (toute la barre représente maxi-mini). On a aussi un texte qui donne
 * la valeur de v en clair.<br>
 *
 * v peut prendre n'importe quelle valeur entre mini et maxi.<br> 
 * On peut faire varier v par une action de la souris sur la barre (soit
 * par clic, soit par dragged) ou en écrivant une nouvelle valeur pour
 * le texte. Quelle que soit la méthode utilisée pour faire varier v
 * (barre ou texte) la modification doit se répercuter à  l'autre (texte
 * ou barre).<br>
 * 
 * Si on donne à  v une valeur hors limite on donne à v la valeur limite dépassée
 * (maxi si v>maxi et mini si v<mini)<br>
 */

public class Barre extends JPanel implements MouseListener, MouseMotionListener, ActionListener, FocusListener
{
    //les propriétés de base
    // la valeur minimale : mini
    // la valeur maximale : maxi
    // la valeur courante, celle qui est amenée à changer : valeur
    // la valeur de référence : ce peut être la valeur initiale au moment de l'appel ou une autre valeur
    private double mini, maxi, valeur, reference ; 
    private int largeur = 60, hauteur = 300, largeurParDefaut = 60, hauteurParDefaut = 300;
    
    private JFormattedTextField texte ;  
    // le positionnement de la barre. La barre est un rectangle
    // (xHautGauche, yHautGauche) le coin à gauche et en haut de la barre
    // (xBasDroite, yBasDroite) le coin opposé
    // hauteurBarre est la hauteur de la barre qui permet de 
    // déterminer l'ordonnée sur la barre correspondant à la valeur courante
    // hauteurBarre = yBasDroite-yHautGauche
     private int xHautGauche, yHautGauche, yBasDroite, xBasDroite, hauteurBarre  ;

    private final static int LARGEUR_BARRE = 10 ; 

    private final static int MARGE_HAUT = 10 ;

    //les couleurs tirées du Look and Feel Metal du JSlider
    private final static Color BLANC = MetalLookAndFeel.getWhite();
    private final static Color SOMBRE = MetalLookAndFeel.getPrimaryControlDarkShadow();
    private final static Color FOND = new Color(238, 238, 238) ;    
    
    
    
    /**
     * Constructeur.
     * @param mini   Valeur minimale.
     * @param maxi   Valeur maximale.
     * @param valeur Valeur initiale.
     */
    public Barre (double mini, double maxi, double valeur)
    {
        this(mini, maxi, valeur, valeur) ;
    }
    
    
    /**
     * Constructeur.
     * @param mini      Valeur minimale.
     * @param maxi      Valeur maximale.
     * @param valeur    Valeur initiale.
     * @param reference Valeur de référence.
     */
    public Barre (double mini, double maxi, double valeur, double reference)
    {
        if (mini > maxi)
            throw new IllegalArgumentException ("La valeur minimale devrait être inférieure à la valeur maximale.") ;
        
        this.mini      = mini ; 
        this.maxi      = maxi ;

        this.valeur = valeurFormatee(dansPlage (valeur, mini, maxi));
        this.reference = valeurFormatee(dansPlage (reference, mini, maxi));

        
        //l'apect graphique
        setBorder (BorderFactory.createEtchedBorder ()) ; 
        setBackground (Color.LIGHT_GRAY) ; 
        setLayout (new BorderLayout ()) ; 

        //le texte formaté
        DecimalFormat df = new DecimalFormat("##0.00") ;
        this.texte = new JFormattedTextField (df) ; 
        this.texte.setHorizontalAlignment (JTextField.CENTER ) ; 
        this.texte.setValue (valeur) ; 
        this.texte.setEditable (false) ;
        this.texte.getCaret ().setVisible (false);
        
        add (this.texte, BorderLayout.SOUTH) ;
        
        //les listeners
        this.addMouseListener        (this) ; 
        this.addMouseMotionListener  (this) ; 
        this.texte.addActionListener (this) ; 
        this.texte.addMouseListener  (this) ;
        this.texte.addFocusListener  (this) ;
    }
    
    public double reference()
    {
    	return this.reference;
    }
    
    public Dimension getMinimumSize ()
    {
        return new Dimension (largeurParDefaut, hauteurParDefaut) ;
    }

    /**
     * on peint le composant 
     */
    public void paintComponent (Graphics g)
    {    
    	Graphics2D g2 = (Graphics2D)g ; 
        
        //fond gris clair
        g2.setPaint (FOND) ; 
        g2.fill3DRect (0, 0, this.getWidth (), this.getHeight (), true) ; 
        
        // Les calculs sont faits avec le texte en bas. 
        this.hauteurBarre = this.getHeight ()-2*MARGE_HAUT - this.texte.getHeight () ; 
        this.xHautGauche = (this.getWidth () - LARGEUR_BARRE)/2 ; 
        this.yHautGauche = MARGE_HAUT ; 
        this.yBasDroite = this.yHautGauche + this.hauteurBarre ;
        this.xBasDroite = this.xHautGauche + LARGEUR_BARRE ;
        
        // la hauteur colorée proportionnelle à la valeur
        int h = (int) (this.hauteurBarre * (this.maxi - this.valeur) / (this.maxi - this.mini)) ; 
        int hRef = (int) (this.hauteurBarre * (this.maxi - this.reference) / (this.maxi - this.mini)) ;
       
        g2.setColor (SOMBRE);
        Shape r = new Rectangle2D.Float (this.xHautGauche, this.yHautGauche,  LARGEUR_BARRE, this.hauteurBarre) ;
        g2.draw (r) ;
        
        //g2.drawRect (xHautGauche, yHautGauche,  LARGEUR_BARRE, hauteurBarre) ;
        
        // la partie basse
        Shape bas = new Rectangle2D.Float(this.xHautGauche + 1, this.yHautGauche + h, LARGEUR_BARRE-2, this.hauteurBarre- h) ;
        g2.setPaint (new GradientPaint(this.xHautGauche+1, this.yBasDroite + h, BLANC, this.xBasDroite-1 , this.yBasDroite + h, SOMBRE)) ;
        g2.fill (bas) ;
        
        //la partie haute
        Shape haut = new Rectangle2D.Float(this.xHautGauche + 1, this.yHautGauche + 1, LARGEUR_BARRE-2, h - 1) ;
        g2.setPaint (new GradientPaint(this.xHautGauche+1, this.yHautGauche + 1, SOMBRE, this.xBasDroite-1 , this.yHautGauche + 1, BLANC)) ;
        g2.fill (haut) ;
        //Une trace pour la valeur de référence
        g2.setColor (SOMBRE);
        g2.fill3DRect(this.xHautGauche -5, this.yHautGauche + hRef -1,LARGEUR_BARRE + 10, 3, true );
    }

    /**
     * Change la valeur du composant.
     * La valeur passée en paramètre est ajustée.
     * on repeint la barre et affiche la valeur que si celle-ci change.
     */
     public void setValeur (double d)
     {
        double nouvValeur = valeurFormatee(dansPlage (d, this.mini, this.maxi)) ;
        if (nouvValeur != this.valeur)
        {
            this.valeur = nouvValeur ; 
            repaint() ;
            this.texte.setValue (nouvValeur) ; 
            fireValeurChangee(new ChangeEvent(this)) ; 
        }
    }
    
     
     public void setReference (double d)
     {
    	double nouvRef = valeurFormatee(dansPlage(d, this.mini, this.maxi)) ;
    	this.reference = nouvRef ;
     }
     
    //MouseMotionListener
    /**
     * si on fait glisser la souris sur la barre on la modifie
     */
    public void mouseDragged (MouseEvent evt)
    {  
        //on modifie la valeur en déplaçant le curseur
        setValeur (this.maxi - (evt.getY () - this.yHautGauche) * (this.maxi-this.mini) / this.hauteurBarre) ; 
    }
    
    //ActionListener
    /**
     * Quand on a validé du texte
     */
    public void actionPerformed (ActionEvent evt)
    {
        //on modifie la valeur à  la main
        setValeur (Double.parseDouble (this.texte.getText ().replace (',', '.'))) ;
        //le texte n'est plus éditable et on l'indique visuellement
        this.texte.setEditable (false) ;
        this.texte.getCaret ().setVisible (false);
        this.texte.setBorder (new EtchedBorder (EtchedBorder.LOWERED)) ;
    }    
    
    //MouseListener
    /**
     * On presse sur le bouton de la souris
     */
    public void mousePressed (MouseEvent evt) 
    {
        //si on a cliqué sur le texte, il devient éditable
        this.texte.setEditable(evt.getSource() == this.texte) ;
        //selon que le texte est éditable ou non, on rend le curseur visible ou invisible
        this.texte.getCaret ().setVisible (this.texte.isEditable ()) ;
        //selon que le texte est éditable ou non, la bordure est en relief ou en creux
        EtchedBorder eb = new EtchedBorder((this.texte.isEditable () ? EtchedBorder.RAISED : EtchedBorder.LOWERED)) ;
        this.texte.setBorder (eb) ;
        if (this.texte.isEditable ())
        {
            // ici les actions pour mettre en évidence que le texte devient éditable.
        }
        else
        {
            // la modification de valeur en fonction de l'endroit où l'on a cliqué sur la barre.
            setValeur (this.maxi - (evt.getY () - this.yHautGauche) * (this.maxi-this.mini) / this.hauteurBarre) ; 
        }
    }
    //procédures non implémentées  
    public void mouseClicked (MouseEvent evt) {}
    public void mouseEntered (MouseEvent evt) {}
    public void mouseExited (MouseEvent evt){}
    public void mouseReleased (MouseEvent evt){}
    public void mouseMoved (MouseEvent evt){}

    public double getValeur ()
    {
        return this.valeur ;
    }

    // Pour informer les écouteurs du changement de valeur     
    protected void fireValeurChangee(ChangeEvent evt) 
    {
            for(ChangeListener listener : getChangeValeurListeners())
            {
                //listener.toString ();
                listener.stateChanged (evt);
            }
    }
    
    public void addChangeListener(ChangeListener l) 
    {
        this.listenerList.add(ChangeListener.class, l);
    }
    public void removeChangeListener(ChangeListener l) 
    {
        this.listenerList.remove(ChangeListener.class, l);
    }
    private ChangeListener[] getChangeValeurListeners() 
    {
        return this.listenerList.getListeners(ChangeListener.class);
    }


    /* (non-Javadoc)
     * @see java.awt.event.FocusListener#focusGained(java.awt.event.FocusEvent)
     */
    public void focusGained (FocusEvent evt){}


    /* (non-Javadoc)
     * @see java.awt.event.FocusListener#focusLost(java.awt.event.FocusEvent)
     */
    public void focusLost (FocusEvent evt)
    {
        //comme si on avait validé la valeur
        if (evt.getComponent ()==this.texte)
        {
            //on modifie la valeur à  la main
            setValeur (Double.parseDouble (this.texte.getText ().replace (',', '.'))) ;
            //le texte n'est plus éditable et on l'indique visuellement
            this.texte.setEditable (false) ;
            this.texte.getCaret ().setVisible (false) ;
            this.texte.setBorder (new EtchedBorder (EtchedBorder.LOWERED)) ;
        }
    }
    
    private double valeurFormatee(double valeur)
    {
        return Flottants.arrondir (valeur, 2);
    }

    public String toString()
    {
        return "Barre : mini = " + this.mini + " ; maxi = " + this.maxi + " ; valeur = " + this.valeur + " ; référence = " + this.reference ; 
    }

	public void setLargeur(int largeur) 
	{
		setPreferredSize(largeur,hauteur);
	}

	public void setHauteur(int hauteur) 
	{
		setPreferredSize(largeur,hauteur);
	}

	public void setPreferredSize(int l,int h)
	{
		setPreferredSize(new Dimension(l,h));
		largeur = l;
		hauteur = h;
	}
}

