/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * This file is part of Histoire.
 *
 * Histoire is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Histoire is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Histoire. If not, see <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique.composants;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;

import javax.swing.JPanel;
import javax.swing.SwingUtilities;



/**
 * La barreAB doit représenter un intervalle entre deux valeurs mini et maxi
 * Elle comporte une barre horizontale avec deux curseurs. Au dessus de chaque
 * curseur on indique la valeur correspondante.
 * 
 * On peut modifier ces valeurs par déplacement des curseurs.
 * 
 * Les propriétés nécessaires sont donc : mini, maxi, A et B comprises entre mini et maxi
 * avec A &lt; B
 */

public class BarreAB extends JPanel implements MouseListener, MouseMotionListener
{
    /* les attributs *******************************************************************/
    
    /* Les valeurs de base *************************************************************/
    private double mini, // la valeur minimale que peut prendre A
                   maxi, // la valeur maximale que peut prendre B
                   A,    // la valeur minimale de l'intervalle
                   B;    // la valeur maximale de l'intervalle
            
    
    /* Les variables graphiques ********************************************************/
    private int xHautGauche, // l'abscisse du coin en haut à gauche
                yHautGauche, // l'ordonnée du coin en haut à gauche
                xBasDroite,  // l'abscisse du coin en bas à droite
                largeurBarre,// la largeur de la barre 
                xAGauche,    // l'abscisse gauche du curseur de A 
                xADroite,    // l'abscisse droite du curseur de A   
                xBGauche,    // l'abscisse gauche du curseur de B
                xBDroite,    // l'abscisse droite du curseur de B
                yHautCurseur,// l'ordonnée du haut des curseurs
                yBasCurseur; // l'ordonnée du bas des curseurs
    
    /** Une variable ternaire ***********************************************************
     *  indiquant si on a pressé la souris alors qu'elle se trouve dans un des curseurs
     *  1 pour A, -1 pour B, 0 pour aucun
     */
    private int curseur;
    
    /* Les constantes graphiques *******************************************************/
    // la barre a une hauteur de 6 pixels
    private final static int      HAUTEUR_BARRE = 6 ;
    // on laisse une marge de 10 pixels à gauche et à droite, en haut et en bas du dessin
    private final static int      MARGE = 10 ;
    // les coordonnées des points dessinant le curseur
    private final static int [][] CURSEUR = {{0, 3, 3, -3, -3},{0, -3, -10, -10, -3}};
    
    /**
     * Le constructeur
     * @param mini  : la valeur minimale de A
     * @param maxi  : la valeur maximale de B
     * @param A     : la valeur minimale de l'intervalle
     * @param B     : la valeur maximale de l'intervalle
     * 
     * On doit avoir mini &lt;= A &lt; B &lt;= maxi
     */
    
    public BarreAB (int mini, int maxi, int A, int B)
    {
        this.mini = Math.min (mini, maxi);
        this.maxi = Math.max (mini, maxi);
        
        this.A = Math.max (this.mini,Math.min(A,B)); 
        this.B = Math.min (this.maxi,Math.max(A,B)); 
        
        //les listeners
        addMouseListener        (this) ; 
        addMouseMotionListener  (this) ; 
    }
    
    // Les acesseurs qui renvoient les valeurs extrèmes de l'intervalle   
    
    public double getInf(){return this.A ;}
    public double getSup(){return this.B ;}
    
    // les dimensions du composant
    public Dimension getPreferredSize()
    {
        return new Dimension(500,55);
    }

    /**
     * on peint le composant 
     */
    public void paintComponent (Graphics g)
    {    
        Graphics2D g2 = (Graphics2D)g ; 
         //fond gris clair
        g2.setPaint (Color.lightGray) ; 
        g2.fillRect ( 0, 0, this.getWidth (), this.getHeight ()) ; 
        // Les calculs pour le positionnement de la barre.
        // le coin en haut à gauche
        // l'ordonnée du coin en haut à gauche de la barre
        this.yHautGauche = (getHeight () - HAUTEUR_BARRE) / 2 ;
        // l'abscisse du coin en haut à gauche de la barre
        this.xHautGauche = MARGE ;
        // l'abscisse du coin en bas à droite de la barre
        this.xBasDroite = getWidth () - MARGE ;
        // la largeur de la barre (liée à la largeur du composant, xBasDroite - xHautGauche)
        this.largeurBarre = this.xBasDroite - this.xHautGauche ; 
        
        // l'abscisse sur la barre proportionnelle à la valeur de A
        double gauche = Math.max (this.mini, this.A - 10);
        double droite = Math.min (this.maxi, this.B + 10);
        int a = (int) (MARGE + (this.largeurBarre * (this.A - gauche) /(droite - gauche))) ; 
        // l'abscisse sur la barre proportionnelle à la valeur de B
        int b = (int) (MARGE + (this.largeurBarre * (this.B - gauche) /(droite - gauche))) ; 
        
        // dessin de la barre en blanc
        g2.setColor(Color.white);
        g2.fillRect (this.xHautGauche, this.yHautGauche, this.largeurBarre, HAUTEUR_BARRE);
        // La partie représentant l'intervalle [A,B] en rouge
        g2.setColor (Color.red);
        g2.fillRect (a, this.yHautGauche, b-a, HAUTEUR_BARRE);

        //Les curseurs (le troisième paramètre vaut true pour A et false pour B)
        dessineCurseur(g2, a, true) ;
        dessineCurseur(g2, b, false);
        
        // dessin des marques (le troisième paramètre vaut true pour A et false pour B)
        dessineMarque(g2, a, true) ;
        dessineMarque(g2, b, false) ;
    }

    /**
     * Dessine le curseur
     * @param g2       le contexte graphique
     * @param v        l'abscisse sur la barre correspondant à A ou B
     * @param curseurA vrai si c'est le curseur A, faux si c'est le B 
     */
    private void dessineCurseur(Graphics2D g2, int v, boolean curseurA)
    {
        //c'est un pentagone
        int [] x = new int[5] ;
        int [] y = new int[5] ;
        for (int i=0 ; i < 5 ; i++)
        {
            x[i] = v + CURSEUR[0][i] ;
            y[i] = this.yHautGauche + CURSEUR[1][i] ;
        }
        this.yHautCurseur = y[2];
        this.yBasCurseur = y[0];
        // On dessine le pourtour du curseur
        g2.setColor (Color.darkGray) ; 
        g2.fillPolygon (x, y, 5) ;
        if (curseurA)
        // il s'agit de A
        {
            this.xAGauche = x[4] ;
            this.xADroite = x[1] ;
        }
        else
        // il s'agit de B
        {
            this.xBGauche = x[4] ;
            this.xBDroite = x[1] ;
        }
    }
    
    /**
     * Dessine la valeur du curseur 
     * @param g2 le contexte graphique
     * @param v  l'abscisse sur la barre correspondant à A ou B
     * @param curseurA vrai si c'est le curseur A, faux si c'est le B 
     */
    private void dessineMarque(Graphics2D g2, int v, boolean curseurA)
    {
        g2.setColor (Color.BLACK);
        //on écrit la valeur du curseur
        
        String s = "" + (curseurA ? this.A : this.B) ;
        
        setFont (getFont().deriveFont (9f));
        FontMetrics fm = g2.getFontMetrics ();
        
        int h = fm.getHeight () ;
        int x = v - fm.stringWidth (s)/2 ;
        int y = this.yHautGauche - h ;
        g2.drawString (s, x, y) ;
    }
    
    /* MouseListener : procédures implémentées *****************************************/
    
    /* On presse sur le bouton de la souris ********************************************/
    public void mousePressed (MouseEvent e)
    {
        this.curseur = 0; // à priori on n'est pas sur un des curseurs
        if (! SwingUtilities.isLeftMouseButton (e)) return ;
        // ce n'est pas le bouton gauche
        if (e.getY()<this.yHautCurseur ||        // l'ordonnée est au-dessus des curseurs
            e.getY()>this.yBasCurseur  ||        // l'ordonnée est en-dessous des curseurs
            e.getX ()<this.xAGauche    ||        // l'abscisse est à gauche des curseurs
            e.getX ()> this.xBDroite) return;    // l'abscisse est à droite des curseurs
        if (e.getX () > this.xADroite  && e.getX () < this.xBGauche) return;
        // l'abscisse est entre les deux curseurs
        if (e.getX () < this.xADroite) this.curseur = 1; // c'est le curseur A
        if (e.getX () > this.xBGauche) this.curseur = -1;// c'est le curseur B 
    }
    
    /* On relache le bouton de la souris ***********************************************/
    public void mouseReleased (MouseEvent evt)
    {
        // Ne traiter que les clics gauche
        if (! SwingUtilities.isLeftMouseButton (evt)) return ;
        // sinon, le clic gauche est relâché, plus de curseur à déplacer
        this.curseur=0 ;
    }
    
    /* MouseListener : procédures non implémentées *************************************/ 
    public void mouseClicked (MouseEvent evt) {}
    public void mouseEntered (MouseEvent evt) {}
    public void mouseExited (MouseEvent evt){}

    /* MouseMotionListener : procédures implémentées ***********************************/
    
    /* on fait glisser la souris sur la barre ******************************************/
    public void mouseDragged (MouseEvent evt)
    {  
        // Ne traiter que le bouton gauche
        if (! SwingUtilities.isLeftMouseButton (evt)) return ;
        int gauche = (int) Math.max (this.mini, this.A - 10);
        int droite = (int) Math.min (this.maxi, this.B + 10);
        
        int v = (gauche + (evt.getX () - MARGE) * (droite-gauche) / this.largeurBarre) ;
        // le curseur A
        if (this.curseur == 1) 
        {
            setValeur(v,true);
        }
        // le curseur B
        if (this.curseur == -1)
        {
            setValeur(v, false);
        }
    }
    

    /* MouseMotionListener : procédures non implémentées *******************************/
    public void mouseMoved (MouseEvent evt){}
    
    
    /**
     * Change la valeur du composant.
     * La valeur passée en paramètre est ajustée.
     * on repeint la barre et affiche la valeur que si celle-ci change.
     * A priori c'est le composant lui-même qui change sa valeur, rien d'autre.
     * @param v la nouvelle valeur
     * @param curseurA quel curseur doit être modifié
     */
    private void setValeur (double v, boolean curseurA)
     {
        if (v > this.maxi) v = this.maxi ; 
        if (v < this.mini) v = this.mini ; 
        
        if (curseurA)
        // s'il s'agit du curseur A
        {
            if (v != this.A && v < this.B)
            // si la valeur change et qu'elle est inférieure à B
            {
                this.A = v;
                repaint();
            }
        }
        else
        /// il s'agit du curseur B
        {
            if (v != this.B && v > this.A)
            // si la valeur change et qu'elle est supérieure à A
            {
                this.B = v;
                repaint();
            }
        }
    }

    public String toString()
    {
        return "BarreAB : mini = " + this.mini + " ; maxi = " + this.maxi + " ; A = " + this.A + " ; B = " + this.B ; 
    }

//     public static void main(String[] args)
//    {
//        // Lancer l'interface graphique
//        JFrame f = new JFrame("Test de barre AB");
//        //BarreTemps barre = new BarreTemps(-180,180,22, 45-22);
//        BarreAB barre=new BarreAB(-180,180,-75,85);
//        f.getContentPane ().add (barre);
//        f.setSize (500, 100);
//        //f.setExtendedState(JFrame.MAXIMIZED_BOTH);
//        f.setDefaultCloseOperation (JFrame.EXIT_ON_CLOSE);
//        f.setVisible (true);
//    }
}
