/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique.composants;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.Graphics ;
import java.awt.Graphics2D;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;

import javax.swing.JComponent;
import javax.swing.SwingUtilities ;
import javax.swing.Timer;
import javax.swing.UIManager;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.plaf.metal.MetalLookAndFeel;



/**
 * La barre temps doit représenter la flèche du temps de -3100 à 2005
 * Elle doit comporter une barre horizontale avec un curseur, deux flèches
 * aux extrémités, des dates en dessous.
 * 
 * On doit pouvoir modifier la date par déplacement du curseur, clic sur les flèches
 * pour avancer ou reculer d'une année, clic entre le curseur et les flèches pour
 * avancer ou reculer de plusieurs années d'un coup.
 * 
 * Les propriétés nécessaires sont donc : mini (-3100), maxi (2005), date comprise
 * entre mini et maxi, pas de déplacement.
 *  
 * Seule la propriété valeur (la date) peut être changée par le composant, les modifications 
 * des trois autres propriétés sont extérieures au composant (esssentiellement le pas).
 * 
 *  Par conséquent il n'est pas nécessaire d'écouter ces changements.
 */
public class BarreTemps extends JComponent implements MouseListener, MouseMotionListener
{
    
    // les propriétés de base
    // on doit avoir, 
    // mini <= valeur <= maxi
    //    0 <=  pas   <= (maxi - mini)/2
    private int mini, maxi, valeur, pas ;

    // le positionnement de la barre et du curseur. La barre est un rectangle
    // (xHautGauche, yHautGauche) le coin à gauche et en haut de la barre (sans les flèches)
    // (xBasDroite, yBasDroite) le coin opposé
    // largeur_Barre est la largeur de la barre (uniquement le rectangle) qui permet de 
    // déterminer l'abscisse sur la barre correspondant à la valeur courante
    // largeur_Barre = BasDroiteX-HautGaucheX
    // x_curseur_gauche et x_curseur_droite déterminent la zone où se trouve le curseur
    private int xHautGauche, yHautGauche, xBasDroite, yBasDroite, largeurBarre, xGaucheCurseur, xDroiteCurseur ;

    // les constantes graphiques
    // la barre a une hauteur de 6 pixels
    private final static int      HAUTEUR_BARRE  = 6 ;
    // on laisse une marge de 20 pixels à gauche et à droite, en haut et en bas du dessin
    private final static int      MARGE          = 20 ;
    // les coordonnées des points dessinant la flèche à gauche
    private final static int [][] FLECHE_GAUCHE  = {{0, 0, -7, -7, 0, 0},{0, -5, 2, 3, 10, 5}};
    // les coordonnées des points dessinant la flèche à droite
    private final static int [][] FLECHE_DROITE  = {{0, 0, 7, 7, 0, 0},{5, 10, 3, 2, -5, 0}};
    // les coordonnées des points dessinant le curseur
    private final static int [][] CURSEUR        = {{-6, 0, 6, 6, - 6},{3, 9, 3, -5, -5}};
    // les marques sous la glissière
    private final static int []   MARQUES        = {-3000, -2000, -1000, 0, 1000, 2000} ;
    
    //les couleurs tirées du Look and Feel Metal du JSlider
    private final static Color BLANC = MetalLookAndFeel.getWhite();
    private final static Color GRIS_CLAIR = (Color)UIManager.get("Slider.altTrackColor"); 
    private final static Color GRIS = MetalLookAndFeel.getControlShadow();
    private final static Color GRIS_FONCE = MetalLookAndFeel.getPrimaryControlShadow();
    private final static Color SOMBRE = MetalLookAndFeel.getPrimaryControlDarkShadow();
    private final static Color FOND = new Color(238, 238, 238) ;    

    // un booléen indiquant si on a pressé la souris alors qu'elle se trouve dans le curseur
    private boolean SourisPresseeDansCurseur ;
    // un timer qui indique si le bouton de la souris est resté abaissé
    private Timer timer = createTimer() ;
    //l'abscisse de la souris au mement où le bouton est abaissé
    private int xSouris ;
    // une tempo pour que le timer ne démarre qu'après un certain temps
    private long tempo ; 
    
    
    
    /**
     * Constructeur.
     * @param mini   Valeur minimale.    
     * @param maxi   Valeur maximale.    
     * @param valeur Valeur courante.    
     * @param pas    Pas de variation.    
     */
    public BarreTemps (int mini, int maxi, int valeur, int pas)
    {
        // le modèle
        if (mini<maxi)
        {
            this.mini = mini ;
            this.maxi = maxi ;
         }
        else
        {
            this.mini = maxi ;
            this.maxi = mini ;
        }
        this.valeur=valeur ;
        if (valeur < this.mini) this.valeur = this.mini ;
        if (valeur > this.maxi) this.valeur = this.maxi ;

        this.pas = pas ;
        if (pas > (maxi-mini)/2) this.pas = ((maxi-mini)/2) ;
        
        //le composant écoute la souris
        addMouseListener (this) ; 
        addMouseMotionListener (this) ; 
    }

     


    // Les acesseurs    
    public int getValeur(){
        return this.valeur ;
    }
    public int getMini(){
        return this.mini ;
    }
    public int getMaxi(){
        return this.maxi ;
    }
    public int getPas (){
        return this.pas ;
    }
    
    public Dimension getPreferredSize()
    {
        return new Dimension(1200,55);
    }

    private ChangeListener[] getChangeValeurListeners() {
        return this.listenerList.getListeners(ChangeListener.class);
    }
    
    
    /**
     * on peint le composant 
     */
    public void paintComponent (Graphics g)
    {    
        Graphics2D g2 = (Graphics2D)g ; 
         //fond gris clair
        g2.setPaint (FOND) ; 
        g2.fill3DRect ( 0, 0, this.getWidth (), this.getHeight (), true) ; 
        // Les calculs pour le positionnement.
        // le haut de la barre
        this.yHautGauche = ( this.getHeight () - HAUTEUR_BARRE) / 2 ;
        this.yBasDroite = this.yHautGauche ;
        // l'abscisse du coin en haut à gauche de la barre
        this.xHautGauche = MARGE ;
        // l'abscisse du coin opposé de la barre
        this.xBasDroite = this.getWidth () - MARGE ;
        // la largeur de la barre (liée à la largeur du composant, xBasDroite - xHautGauche)
        this.largeurBarre = this.getWidth () - 2 * MARGE ; 
        
        // l'abscisse sur la barre proportionnelle à la valeur courante du composant
        int v = Math.round (this.largeurBarre * ( this.getValeur () - this.getMini ()) / 
                                                (this.getMaxi () - this.getMini ())) ; 
        // le pourtour du composant
        int [] x = new int[12] ;
        int [] y = new int[12] ;
        for (int i=0 ; i < 6 ; i++)
        {
            x[i]=this.xHautGauche - 6 + FLECHE_GAUCHE[0][i] ;
            y[i]=this.yHautGauche + FLECHE_GAUCHE[1][i] ;
        }
        for (int i=6 ; i < 12 ; i++)
        {
            x[i]=this.xBasDroite + 6 + FLECHE_DROITE[0][i - 6] ;
            y[i]=this.yBasDroite + FLECHE_DROITE[1][i - 6] ;
        }
        // le dessin de la glissière
        g2.setColor (SOMBRE);
        g2.drawPolygon (x, y, 12);
        
        // dessin de la barre
        dessineBarre(g2, v) ;
        // dessin des fleches
        dessineFlecheGauche(g2, x[1], y[1]) ;
        dessineFlecheDroite(g2, x[10], y[10]) ;
        // dessin des marques
        dessineMarques(g2, v) ;
        //Le curseur
        dessineCurseur(g2, v) ;
        
    }
    /**
     * Dessine juste la barre dans un look Metal
     * La barre est rallongée de la moitié de la largeur du curseur (10) pour 
     * représenter toutes les valeurs possibles.
     * @param g2 : le contexte graphique
     * @param v  : l'abscisse sur la barre proportionnelle à la valeur courante
     */
    private void dessineBarre(Graphics2D g2, int v)
    {
        // la partie gauche
        // 4 lignes de couleurs de clarté décroissantes du haut vers le bas
        g2.setColor(BLANC) ;
        g2.drawLine(this.xHautGauche - 5, this.yHautGauche + 1, this.xHautGauche + v, this.yHautGauche + 1) ;
        g2.setColor(GRIS_CLAIR) ;
        g2.drawLine(this.xHautGauche - 5, this.yHautGauche + 2, this.xHautGauche + v, this.yHautGauche + 2) ;
        g2.setColor(GRIS) ;
        g2.drawLine(this.xHautGauche - 5, this.yHautGauche + 3, this.xHautGauche + v, this.yHautGauche + 3) ;
        g2.setColor(GRIS_FONCE) ;
        g2.drawLine(this.xHautGauche - 5, this.yHautGauche + 4, this.xHautGauche + v, this.yHautGauche + 4) ;
        // la partie droite
        // 1 ligne plus foncée en haut
        g2.setColor(GRIS_FONCE) ;
        g2.drawLine(this.xHautGauche + v + 1, this.yHautGauche + 1, this.xBasDroite + 5, this.yHautGauche + 1) ;
    }

    /**
     * dessine la flèche gauche dans un look Metal
     * @param g2 le context graphique
     * @param x l'abscisse du point le plus haut de la flèche gauche
     * @param y l'ordonnée du point le plus haut de la flèche gauche
     */
    private void dessineFlecheGauche(Graphics2D g2, int x, int y)
    {
        g2.setColor (BLANC) ;
        // 5 lignes blanches en haut qui colorent l'intérieur de la flèche
        g2.drawLine(x - 1, y + 2, x - 1, y + 2) ;
        g2.drawLine(x - 1, y + 3, x - 2, y + 3) ;
        g2.drawLine(x - 1, y + 4, x - 3, y + 4) ;
        g2.drawLine(x - 1, y + 5, x - 4, y + 5) ;
        g2.drawLine(x , y + 6, x - 5, y + 6) ;
        // puis une ligne gris clair
        g2.setColor (GRIS_CLAIR) ;
        g2.drawLine(x , y + 7, x - 6, y + 7) ;
        // une autre gris
        g2.setColor (GRIS) ;
        g2.drawLine(x , y + 8, x - 6, y + 8) ;
        // le reste en gris foncé
        g2.setColor (GRIS_FONCE) ;
        g2.drawLine(x , y + 9, x - 5, y + 9) ;
        g2.drawLine(x - 1, y + 10, x - 4, y + 10) ;
        g2.drawLine(x - 1, y + 11, x - 3, y + 11) ;
        g2.drawLine(x - 1, y + 12, x - 2, y + 12) ;
        g2.drawLine(x - 1, y + 13, x - 1, y + 13) ;
    }

    /**
     * dessine la flèche droite dans un look Metal
     * @param g2 le context graphique
     * @param x l'abscisse du point le plus haut de la flèche droite
     * @param y l'ordonnée du point le plus haut de la flèche droite
     */
    private void dessineFlecheDroite(Graphics2D g2, int x, int y)
    {
        g2.setColor (GRIS_FONCE) ;
        // 5 lignes en gris foncé en haut
        g2.drawLine(x + 1, y + 2, x + 1, y + 2) ;
        g2.drawLine(x + 1, y + 3, x + 2, y + 3) ;
        g2.drawLine(x + 1, y + 4, x + 3, y + 4) ;
        g2.drawLine(x + 1, y + 5, x + 4, y + 5) ;
        g2.drawLine(x , y + 6, x + 5, y + 6) ;
    }

    /**
     * Dessine la valeur du curseur et les marques sous la barre 
     * @param g2 : le contexte graphique
     * @param v  : l'abscisse sur la barre, proportionnelle à la valeur courante
     */
    private void dessineMarques(Graphics2D g2, int v)
    {
        g2.setColor (Color.BLACK);
        //on écrit la valeur du curseur
        String s = ((Integer) this.getValeur ()).toString () ;
        FontMetrics fontMetrics = g2.getFontMetrics ();
        int h = fontMetrics.getHeight () ;
        int x = this.xHautGauche + v - fontMetrics.stringWidth (s)/2 ;
        int y = this.yHautGauche - h/2 ;
        g2.drawString (s, x, y) ;
        // les repères
        int w ;
        for (int i=1 ; i <= MARQUES.length ; i++){
            s = ((MARQUES[i-1] == 0) ? "" :(Integer) MARQUES[i-1]).toString() ;
            w = Math.round (this.largeurBarre * ( MARQUES[i-1] - this.getMini ()) / (this.getMaxi () - this.getMini ())) ;
            x = Math.round(this.xHautGauche + w - fontMetrics.stringWidth (s)/2) ;
            y = this.yHautGauche + 3 * h/2  ;
            g2.drawLine (this.xHautGauche + w , this.yHautGauche + 6, this.xHautGauche + w, this.yHautGauche + 6 + h/2) ;
            g2.drawString (s, x, y) ;
        }
    }


    /**
     * Dessine le curseur dans un look Metal
     * @param g2 : le contexte graphique
     * @param v  : l'abscisse sur la barre proportionnelle � la valeur courante
     */
    private void dessineCurseur(Graphics2D g2, int v)
    {
        //c'est un pentagone
        int [] x = new int[5] ;
        int [] y = new int[5] ;
        for (int i=0 ; i < 5 ; i++){
            x[i] = this.xHautGauche + v + CURSEUR[0][i] ;
            y[i] = this.yHautGauche + CURSEUR[1][i] ;
        }
        // On dessine le pourtour du curseur
        g2.setColor (SOMBRE) ; 
        g2.drawPolygon (x, y, 5) ;
        // On dessine l'intérieur du curseur
        // 2 lignes grises en haut
        g2.setColor (GRIS) ;
        g2.drawLine(x[4] + 1, y[4] + 1, x[3] - 1, y[4] + 1) ;
        g2.setColor (GRIS) ;
        g2.drawLine(x[4] + 1, y[4] + 2, x[3] - 1, y[4] + 2) ;
        // 2 lignes gris clair ensuite
        g2.setColor (GRIS_CLAIR) ;
        g2.drawLine(x[4] + 1, y[4] + 3, x[3] - 1, y[4] + 3) ;
        g2.setColor (GRIS_CLAIR) ;
        g2.drawLine(x[4] + 1, y[4] + 4, x[3] - 1, y[4] + 4) ;
        // 3 lignes blanches
        g2.setColor (BLANC) ;
        g2.drawLine(x[4] + 1, y[4] + 5, x[3] - 1, y[4] + 5) ;
        g2.setColor (BLANC) ;
        g2.drawLine(x[4] + 1, y[4] + 6, x[3] - 1, y[4] + 6) ;
        g2.setColor (BLANC) ;
        g2.drawLine(x[4] + 1, y[4] + 7, x[3] - 1, y[4] + 7) ;
        // 2 lignes gris clair
        g2.setColor (GRIS_CLAIR) ;
        g2.drawLine(x[4] + 1, y[4] + 8, x[3] - 1, y[4] + 8) ;
        g2.setColor (GRIS_CLAIR) ;
        g2.drawLine(x[4] + 2, y[4] + 9, x[3] - 2, y[4] + 9) ;
        // 2 lignes grises
        g2.setColor (GRIS) ;
        g2.drawLine(x[4] + 3, y[4] + 10, x[3] - 3, y[4] + 10) ;
        g2.setColor (GRIS) ;
        g2.drawLine(x[4] + 4, y[4] + 11, x[3] - 4, y[4] + 11) ;
        // 2 lignes gris foncé
        g2.setColor (GRIS_FONCE) ;
        g2.drawLine(x[4] + 5, y[4] + 12, x[3] - 5, y[4] + 12) ;
        g2.setColor (GRIS_FONCE) ;
        g2.drawLine(x[4] + 6, y[4] + 13, x[3] - 6, y[4] + 13) ;
        
        this.xGaucheCurseur = x[0] ;
        this.xDroiteCurseur = x[2] ;
    }

// Les modificateurs
    /**
     * Change la valeur du composant.
     * La valeur passée en paramètre est ajustée.
     * on ne repeint la barre et affiche la valeur que si celle-ci change.
     * @param v Nouvelle valeur.
     */
     public void setValeur (int v)
    {
        if (v > this.maxi) v = this.maxi ; 
        if (v < this.mini) v = this.mini ; 
        if (v != this.valeur)
        {
            if (v == 0) v = ((this.valeur < 0) ? 1 : -1) ;
            this.valeur = v ; 
            repaint () ; 
            fireValeurChangee(new ChangeEvent(this)) ; 
        }
    }
    /**
     * Change la valeur de inc
     * @param inc la nouvelle valeur
     */
     public void changeValeurDe(int inc)
    {
        if (inc != 0) setValeur(this.valeur + inc) ;
    }
     
    /**
     * Change la valeur du pas du composant.
     * La valeur passée en paramètre est ajustée.
     * @param p Nouvelle valeur pour le pas.
     */
     public void setPas (int p)
    {
        if (p > (this.maxi - this.mini)/2) p = (this.maxi - this.mini)/2 ; 
        if (p < 0) p = 0 ; 
        if (p != this.pas) this.pas = p ;
    }

    private Timer createTimer ()
    {
    // Création d'une instance de listener 
    // associée au timer
    ActionListener action = new ActionListener ()
      {
        // Méthode appelée à chaque tic du timer
        public void actionPerformed (ActionEvent event)
        {
           if ((System.currentTimeMillis ()- tempo) > 500 )
           {
               //clic sur la flèche gauche : on décrémente de 1
               if (xSouris < xHautGauche) changeValeurDe(- 1);  
               //clic entre la flèche gauche et le curseur : on décrémente de pas     
               else if (xSouris < xGaucheCurseur) changeValeurDe(- getPas());
               //clic sur le curseur : rien
               else if (xSouris < xDroiteCurseur) ; 
               //clic entre le curseur et la flèche droite  : on incrémente de pas
               else if (xSouris < xBasDroite) changeValeurDe(getPas());
               //clic sur la flèche droite : on incrémente de 1
               else changeValeurDe(1);
           }
        }
      };
         // Création d'un timer qui génère un tic
        // chaque 100 millième de seconde
        return new Timer (100, action);
      }  
    
     //MouseListener
     /**
     * Quand on clic, on modifie la valeur courante
     * sur la flèche à gauche, on décrémente valeur de 1 unité
     * sur la flèche droite, on incrémente valeur de 1 unité
     * entre la flèche gauche et le curseur, on décrémente valeur de pas
     * entre la flèche droite et le cursuer, on incrémente valeur de pas
     * sur le curseur : rien
     * Cela nécessite de connaître les dimensions de la barre et la position du curseur.
     */
    
    public void mouseClicked (MouseEvent e)
    {
        //timer.stop () ;
    } 

    public void mouseEntered (MouseEvent e) {}
    public void mouseExited (MouseEvent e){}
    
    
    
    public void mousePressed (MouseEvent e)
    {
        
        this.xSouris=e.getX () ;
        if (!this.SourisPresseeDansCurseur) 
            this.SourisPresseeDansCurseur = 
                ((this.xSouris >= this.xGaucheCurseur) && (this.xSouris <= this.xDroiteCurseur)) ;
        
        
        if (!this.SourisPresseeDansCurseur)
        {
            // Ne traiter que les clics gauche
            if (! SwingUtilities.isLeftMouseButton (e))
                return ;
            
            if (e.getX () < this.xHautGauche)   
                //clic sur la flèche gauche : on décrémente de 1
                this.changeValeurDe(- 1);
            else if (e.getX () < this.xGaucheCurseur) 
                //clic entre la flèche gauche et le curseur : on décrémente de pas
                this.changeValeurDe(- this.getPas());
            else if (e.getX () < this.xDroiteCurseur) 
                ; //clic sur le curseur : rien
            else if (e.getX () < this.xBasDroite)
                //clic entre le curseur et la flèche droite  : on incrémente de pas
                this.changeValeurDe(this.getPas());
            else 
            //clic sur la flèche droite : on incrémente de 1
            this.changeValeurDe(1);
            this.tempo = System.currentTimeMillis () ;
            this.timer.start () ;
        }
            
    }

    public void mouseReleased (MouseEvent e)
    {
        // Ne traiter que les clics gauche
        if (! SwingUtilities.isLeftMouseButton (e))
            return ;
        
        this.timer.stop () ;
        this.SourisPresseeDansCurseur = false ;
    }

    public void mouseDragged (MouseEvent evt)
    {
        // Ne traiter que lee bouton gauche
        if (! SwingUtilities.isLeftMouseButton (evt))
            return ;
            
       //if ((xSouris-evt.getX ()) > 10) timer.stop () ;
       //si en dehors du curseur on ne traite pas
       if (this.SourisPresseeDansCurseur)
           this.setValeur((int) (this.getMini () + (evt.getX() - MARGE) * 
                                  (double)(this.getMaxi () - this.getMini ())/ this.largeurBarre)) ;
    }

    public void mouseMoved (MouseEvent e){}
    
// Pour informer les écouteurs du changement de valeur
    
    protected void fireValeurChangee(ChangeEvent evt) {
            for(ChangeListener listener : getChangeValeurListeners()) {
                listener.stateChanged (evt);
            }
    }

    public void addChangeListener(ChangeListener l) {
        this.listenerList.add(ChangeListener.class, l);
    }
    public void removeChangeListener(ChangeListener l) {
        this.listenerList.remove(ChangeListener.class, l);
    }
}