/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique.donnees;

import java.util.ArrayList ;
import java.util.Collections ;
import java.util.List ;

import fr.histoiremondiale.histoire.donnees.Civilisation;
import fr.histoiremondiale.histoire.igraphique.donnees.comparateurs.ComparateurParagraphesParDate;



/**
 * Données d'un paragraphe du navigateur html. La période couverte par le paragraphe est [Debut-Fin[
 * L'année de fin n'est pas comprise dans la période du paragraphe.
 * 
 * Modif 29/04/2019 : l'ordre des paragraphes n'est plus donné par l'ordre des identifiants mais
 * par leur date début puis leur identifiant il faut modifier la méthode compareTo pour refléter
 * cet aspect des choses.
 * 
 */
public class InfosParagrapheNavig implements Comparable<Object>
{

    private Integer      id ;
    private Civilisation civilisation ;                 // Civilisation concernée par le paragraphe (optionnelle)
    private int          anneeDeb ;                     // Début/fin de la période couverte par le paragraphe (date de fin exclue)
    private int          anneeFin ;
    private String       titre ;                        // Titre du paragraphe
    
    private List<InfosParagrapheNavig> precedents ;     // Paragraphes précédant le paragraphe courant
    private List<InfosParagrapheNavig> suivants ;       // Paragraphes suivant le paragraphe courant
    
    private String       codeHtml ;                     // Code HTML représentant le paragraphe (code permettant de représenter une page html complète, en-têtes compris)
    private String       codeHtmlSimple ;               // Code HTML simple, sans liens, sans en-tête et pied de page, ... (pour l'insertion de ce code dans autre chose)


    
    /**
     * Constructeur.
     * @param id           Identifiant du paragraphe.
     * @param civilisation Civilisation concernée par le paragraphe.
     * @param anneeDeb     Début de la période couverte par le paragraphe.
     * @param anneeFin     Fin   de la période couverte par le paragraphe.
     * @param titre        Titre du paragraphe.
     */
    public InfosParagrapheNavig (Integer id, Civilisation civilisation, int anneeDeb, int anneeFin, String titre)
    {
        // Mémoriser les autres attributs
        this.id           = id ;
        this.civilisation = civilisation ;
        this.anneeDeb     = anneeDeb ;
        this.anneeFin     = anneeFin ;
        this.titre        = titre ;
        
        // Initialiser les autres attributs
        this.precedents = new ArrayList<>() ;
        this.suivants   = new ArrayList<>() ;
    }
    
    
    
    // Fonction de comparaison.
    // L'ordre des paragraphes est déterminé par son début et son identifiant
    // p1 < P2 ssi P1.debut < P2.debut ou (P1.debut = P2.debut et P1.id < P2.id)
    public int compareTo (Object obj)
    {
        // Premières comparaison
        if (this == obj)
            return 0 ;
        if (obj == null)
            return +1 ;
        
        // Vérification des types
        if (! (obj instanceof InfosParagrapheNavig))
            throw new IllegalArgumentException ("Type de paramètre invalide : " + obj) ;

        // Comparaison des valeurs
        InfosParagrapheNavig P2 = (InfosParagrapheNavig)obj;
        if (this.anneeDeb == P2.anneeDeb) return this.id - P2.id ;
        return this.anneeDeb - P2.anneeDeb;
    }
    

    
    /**
     * Indique si l'année en paramètre est dans la période couverte par ce paragraphe.
     * @param annee Année à comparer.
     * @return Vrai si l'année est comprise dans la période couverte par le paragraphe.
     */
    public boolean anneeEstDansPeriodeCouverte (int annee)
    {
        return (this.anneeDeb <= annee) && (annee < this.anneeFin) ;
    }

    
    /**
     * Renvoie l'année passée en paramètre si celle-ci est comprise entre
     *   l'année de début et l'année de fin du Paragraphe.
     *   Si l'année passée en paramètre est après la fin du paragraphe 
     *   on renvoie l'année de fin, sinon on renvoie l'année de début.
     * @param annee Année à ajuster.
     * @return L'année ajustée.
     */
    public int ajusterAnneeDansParagraphe (int annee)
    {
    	if (anneeEstDansPeriodeCouverte (annee)) return annee;
    	if (annee >= this.anneeFin) return this.anneeFin-1;
        return this.anneeDeb ;
    }
    
    
    // Modifie le code html du paragraphe
    public void modifCodeHtml (String codeHtml)
    {
        this.codeHtml = codeHtml ;
    }
    public void modifCodeHtmlSimple (String codeHtmlSimple)
    {
        this.codeHtmlSimple = codeHtmlSimple ;
    }
    
    
    /**
     * Renvoie l'ancre associée au paragraphe (sans échappement html appliqué).
     * @return L'ancre.
     */
    public String ancre ()
    {
        return (this.civilisation != null ? this.civilisation.nom() : "") +
               "&" +
               this.anneeDeb ;
    }
    

    /**
     * Renvoie les paragraphes précédents triés.
     * @return Les paragraphes précédents triés.
     */
    public List<InfosParagrapheNavig> precedentsTries ()
    {
        List<InfosParagrapheNavig> precedents = new ArrayList<> (this.precedents) ;
        Collections.sort (precedents, new ComparateurParagraphesParDate()) ;
        return precedents ;
    }

    /**
     * Renvoie les paragraphes suivants triés.
     * @return Les paragraphes suivants triés.
     */
    public List<InfosParagrapheNavig> suivantsTries ()
    {
        List<InfosParagrapheNavig> suivants = new ArrayList<> (this.suivants) ;
        Collections.sort (suivants, new ComparateurParagraphesParDate()) ;
        return suivants ;
    }


    
    // Autres accesseurs
    public Integer                    id             () { return this.id ; } ;
    public Civilisation               civilisation   () { return this.civilisation ; }
    public int                        anneeDeb       () { return this.anneeDeb ; }
    public int                        anneeFin       () { return this.anneeFin ; }
    public String                     titre          () { return this.titre ; }
    public List<InfosParagrapheNavig> precedents     () { return this.precedents ; }
    public List<InfosParagrapheNavig> suivants       () { return this.suivants ; }
    public String                     codeHtml       () { return this.codeHtml ; }
    public String                     codeHtmlSimple () { return this.codeHtmlSimple ; }



    public String afficheTitre ()
    {
        return this.civilisation.nom () + "(" + this.anneeDeb + " à " + this.anneeFin + ") : " + this.titre;
    }

}
