/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique.tracercartes;

import java.awt.Color ;
import java.awt.Font ;
import java.awt.Graphics2D ;

import javax.swing.plaf.metal.MetalLookAndFeel ;

import fr.histoiremondiale.histoire.utiles.igraphique.Polices;



/**
 * Classe représentant un texte à afficher sur la carte.
 */
public class TexteCarte
{
    // Le gris du Look and Feel Metal
    private final static Color GRIS = MetalLookAndFeel.getControlShadow() ;    
    
    private String[] lignes ;       // Texte (éventuellement sur plusieurs lignes)
    private Font     police ;       // Police de caractère à utiliser pour tracer le texte
    

    
    /**
     * Renvoie la hauteur cumulée de deux textes.
     * @param txtNom  Texte représentant le nom du territoire.
     * @param txtInfo Texte d'info supplémentaire sur le territoire.
     * @param g2      Pinceau.
     * @return La hauteur des deux textes cumulés.
     */
    public static int hauteurTextes (TexteCarte txtNom, TexteCarte txtInfo, Graphics2D g2)
    {
        return (txtNom.hauteur (g2) + txtInfo.hauteur (g2)) ;
    }
    
    
    /**
     * Constructeur.
     */
    public TexteCarte ()
    {
        this (new String[0], null) ;
    }
    
    /**
     * Constructeur.
     * @param lignes Les lignes du texte à afficher.
     * @param police Police de caractère à utiliser, null s'il n'y a rien à tracer...
     */
    public TexteCarte (String[] lignes, Font police)
    {
        this.lignes = lignes ;
        this.police = police ;
    }
    
    
    /**
     * Indique si le texte représenté par l'objet est affichable.<br>
     *   (&lt;=&gt; toutes les informations sont bien disponibles pour afficher quelque chose).<br>
     * Parce que certains textes peuvent ne pas être présents et que c'est plus facile de traiter
     *   tout ça uniformément.
     * @return Vrai si le texte est affichable, faux sinon.
     */
    public boolean estAffichable ()
    {
        return this.lignes != null && this.lignes.length > 0 && this.police != null ;
    }
    
    
    /**
     * Indique la hauteur du texte.
     * @param g Pinceau.
     * @return La hauteur du texte.
     */
    public int hauteur (Graphics2D g)
    {
        if (!estAffichable())
            return 0 ;
        
        // Sauver la police de l'outil de dessin
        Font policeInitPinceau = g.getFont() ;
        
        // Additionner les hauteurs de tous les textes
        g.setFont (this.police) ;
        int hauteur = 0 ;
        for (String txt : this.lignes)
            hauteur += Polices.hauteur (txt, g) ; 
        
        // Restaurer la police de l'outil de dessin
        g.setFont (policeInitPinceau) ;
        
        // Renvoyer la hauteur calculée + marge (espace entre le nom du territoire et l'info.
        // empiriquement la valeur 2 semble la bonne quand il n'y a pas de cadre et 4 s'il y un cadre
        // Cela m'inspire que le calcul du tracé devrait intégralement se faire dans cette classe à partir des
        // valeurs de nomTerritoire et infoTerritoire
        return hauteur ;
    }
    
    /**
     * Indique la largeur du texte.
     * @param g  Pinceau.
     * @return La largeur du texte.
     */
    public int largeur (Graphics2D g)
    {
        if (!estAffichable())
            return 0 ;
        
        // Sauver la police de l'outil de dessin
        Font policeInitPinceau = g.getFont() ;
        
        // Sélectionner la plus grande largeur
        g.setFont (this.police) ;
        int largeur = 0 ;
        for (String txt : this.lignes)
            largeur = (int) Math.max (largeur, Polices.largeur (txt, g)) ;
        
        // Restaurer la police de l'outil de dessin
        g.setFont (policeInitPinceau) ;
        
        // Renvoyer la largeur calculée
        return largeur ;
    }
    
    
    /**
     * Trace les lignes du texte avec le pinceau donné.
     * @param g2 Pinceau
     * @param x Abscisse de la position à laquelle tracer le texte.
     * @param y Ordonnée de la position à laquelle tracer le texte.
     * @param cadre Indique s'il faut tracer un cadre autour du texte.
     */
    public void tracerLignesCentrees (Graphics2D g2, int x, int y, boolean cadre)
    {
        //s'il n'y a rien à afficher
        if (!estAffichable()) return ;
        //On a quelquechose à afficher
        //On sauvegarde la police d'écriture antérieure
        Font policeInitPinceau = g2.getFont () ;
        //Les dimensions du texte
        int largeurTexte = largeur(g2) ;
        int hauteurTexte = hauteur(g2) ;
        // La police d'écriture
        g2.setFont (this.police) ;
        //Si on doit tracer un cadre il faut ajouter l'espace entre le nom du territoire et celui-ci.
        if (cadre) y=y+4 ;
        
        int yLigne = y ;
        //On écrit chaque ligne
        for (String ligne : this.lignes)
        {
            int largeurLigne = (int) Polices.largeur (ligne, g2) ;
            int xLigne       = x + (largeurTexte - largeurLigne) / 2 ;
            g2.drawString (ligne, xLigne, yLigne + (int)Polices.elevation (ligne, g2)) ;
            yLigne += Polices.hauteur (ligne, g2) ;
        }
        //Le cadre (c'est là que cela diffère)
        if (cadre)
        {
            Color  coulSauv  = g2.getColor() ;
            g2.setColor  (GRIS) ;
            g2.draw3DRect (x - 1, y - 1, largeurTexte + 2, hauteurTexte, false) ;
            g2.setColor  (coulSauv) ;
        }
        
        // Restaurer la police de l'outil de dessin
        g2.setFont (policeInitPinceau) ;
    }
    
}
