/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.utiles.exttypes;



/**
 * Utilitaires sur les chaînes.
 */
public class Chaines
{

    /**
     * Indique si la chaîne est vide (null ou ne contient que des espaces).
     * @param chaine Le chaîne à tester.
     * @return Vrai si la chaîne est vide, faux sinon.
     */
    public static boolean estVide (String chaine)
    {
        return chaine == null || chaine.trim().isEmpty() ;
    }
    
    
    /**
     * Renvoie tout ce qui précède un marqueur donné (la première occurrence du marqueur).<br>
     * Si le marqueur n'est pas trouvé, renvoie la chaîne complète.
     * @param chaine  La chaîne considérée.
     * @param marqueur Le marqueur à utiliser.
     * @return Ce qui précède le marqueur, toute la chaîne si le marqueur est absent.
     */
    public static String avantMarqueur (String chaine, String marqueur)
    {
        int posMarqueur = chaine.indexOf (marqueur) ;
        if (posMarqueur == -1)
            return chaine ;
        else
            return chaine.substring (0, posMarqueur) ;
    }
    
    
    /**
     * Renvoie tout ce qui suit un marqueur donné (la première occurrence du marqueur).<br>s
     * Si le marqueur n'est pas trouvé, renvoie une chaîne vide.
     * @param chaine  La chaîne considérée.
     * @param marqueur Le marqueur à utiliser.
     * @return Ce qui suit le marqueur, une chaîne vide si le marqueur est absent.
     */
    public static String apresMarqueur (String chaine, String marqueur)
    {
        int posMarqueur = chaine.indexOf (marqueur) ;
        int debSuite    = posMarqueur + marqueur.length() ;
        if (posMarqueur == -1 || debSuite >= chaine.length())
            return "" ;
        else
            return chaine.substring (debSuite) ;
    }
    
    
    /**
     * Renvoie tout ce qui se trouve entre deux marqueurs donnés.<br>
     * Si le marqueur de début n'est pas trouvé, renvoie une chaîne vide.<br>
     * Si le marqueur de fin n'est pas trouvé, renvoie tout ce qui suit le marqueur de début.
     * @param chaine      La chaîne considérée.
     * @param marqueurDeb Le marqueur de début à utiliser.
     * @param marqueurFin Le marqueur de fin   à utiliser.
     * @return Ce qui se trouve entre les deux marqueurs.
     */
    public static String entreMarqueurs (String chaine, String marqueurDeb, String marqueurFin)
    {
        return avantMarqueur (apresMarqueur (chaine, marqueurDeb), marqueurFin) ;
    }

    
    /**
     * Retire tout ce qui se trouve entre deux marqueurs donnés.
     * @param chaine      La chaîne considérée.
     * @param marqueurDeb Le marqueur de début à utiliser.
     * @param marqueurFin Le marqueur de fin   à utiliser.
     * @return La chaîne sauf ce qui se trouve entre les deux marqueurs.
     */
    public static String saufEntreMarqueurs (String chaine, String marqueurDeb, String marqueurFin)
    {
        String chTmp = chaine ;
        while (true)
        {
            String chTmp2 ;
            if (chTmp.contains (marqueurDeb) && chTmp.contains (marqueurFin))
            {
                String avantMarqueur = avantMarqueur (chTmp,        marqueurDeb) ;
                String apresMarqueur = apresMarqueur (chTmp,        marqueurDeb) ;
                chTmp2 = avantMarqueur + apresMarqueur (apresMarqueur, marqueurFin) ;
            }
            else
            {
                chTmp2 = chTmp ;
            }
            if (chTmp2.length() == chTmp.length())
                return chTmp2 ;
            chTmp = chTmp2 ;
        }
    }
    
    
    /**
     * Complète une chaîne en ajoutant à gauche le caractère donné jusqu'à ce que la chaîne atteigne la
     *   taille indiquée.<br>
     * Si la chaîne en paramètre est plus grande que la taille demandée, elle n'est pas modifiée.<br>
     * Si la chaîne en entrée vaut null, erreur.
     * @param chaine   La chaîne à traiter.
     * @param taille   La taille à atteindre.
     * @param carCompl Le caratère permettant de compléter la chaîne.
     * @return La chaîne complétée.
     */
    public static String completerGauche (String chaine, int taille, char carCompl)
    {
        String chRes = chaine ;
        while (chRes.length() < taille)
            chRes = carCompl + chRes ;
        return chRes ;
    }
    
    
     /**
     * Complète une chaîne en ajoutant à droite le caractère donné jusqu'à ce que la chaîne atteigne la
     *   taille indiquée.<br>
     * Si la chaîne en paramètre est plus grande que la taille demandée, elle n'est pas modifiée.<br>
     * Si la chaîne en entrée vaut null, erreur.
     * @param chaine   La chaîne à traiter.
     * @param taille   La taille à atteindre.
     * @param carCompl Le caractère permettant de compléter la chaîne.
     * @return La chaîne complétée.
     */
   public static String completerDroite (String chaine, int taille, char carCompl)
    {
        String chRes = chaine ;
        while (chRes.length() < taille)
            chRes += carCompl ;
        return chRes ;
    }
    
    
    /**
     * Renvoie une sous-chaîne commençant à l'indice de début donné et se terminant avant l'indice de fin
     *   donné (indice de fin exclu).<br>
     * Si l'indice de début est au delà de la fin de la chaîne, une chaine vide est renvoyée.
     * Si l'indice de fin est au delà de la fin de la chaîne, la fin de la chaîne est renvoyée.<br>
     * ! Contrairement à la méthode substring() de Java, rien ne garantit donc que la longueur de la chaîne
     *   renvoyée soit bien égale à (iFin - iDeb).
     * @param chaine La chaîne à traiter.
     * @param iDeb L'indice de début.
     * @param iFin L'indice de fin.
     * @return La sous-chaîne extraite.
     */
    public static String sousChaine (String chaine, int iDeb, int iFin)
    {
        if (iDeb >= chaine.length())
            return "" ;
        else if (iFin >= chaine.length() + 1)
            return chaine.substring (iDeb) ;
        else
            return chaine.substring (iDeb, iFin) ;
    }
    /**
     * Renvoie une sous-chaîne commençant à l'indice de début donné.<br>
     * Si l'indice de début est au delà de la fin de la chaîne, une chaine vide est renvoyée.
     * ! Contrairement à la méthode substring() de Java, rien ne garantit donc que la longueur de la chaîne
     *   renvoyée soit bien égale à (iFin - iDeb).
     * @param chaine La chaîne à traiter.
     * @param iDeb L'indice de début.
     * @return La sous-chaîne extraite.
     */
    public static String sousChaine (String chaine, int iDeb)
    {
        if (iDeb >= chaine.length())
            return "" ;
        else
            return chaine.substring (iDeb) ;
    }

    /**
     * Renvoie le dernier caractère de la chaîne. Si la chaîne est vide, renvoie le caractère de code 0.
     * @param chaine La chaîne à traiter.
     * @return Le dernier caractère.
     */
    public static char dernierCaractere(String chaine)
    {
        if (chaine == null || chaine.length ()== 0) return 0;
        return chaine.charAt (chaine.length () - 1);
    }
    
    /**
     * Renvoie la chaîne sans le dernier caractère. Si la chaîne est vide ou ne contient qu'un seul caractère,
     *   renvoie une chaîne vide.
     * @param chaine La chaîne à traiter.
     * @return La chaîne sans le dernier caractère.
     */
    public static String saufLeDernier(String chaine)
    {
        if (chaine.length () < 2) return ""; 
        return sousChaine(chaine,0,chaine.length ()-1) ;
    }
    
    /**
     * une fonction qui peut-être utile : renverser une chaine
     * @param chaine la chaine à traiter
     * @return la chaine écrite dans l'autre sens
     */
    public static String renverser(String chaine)
    {
    	if (chaine==null || chaine.length() < 2) return chaine;
    	return new StringBuffer(chaine).reverse().toString();
    }
    
    /**
     * une fonction qui vérifie si une chaine de caractères représente un entier
     * @param chaine la chaine à traiter
     * @return vrai si la chaine est un entier (entre -2 147 483 648 et 2 147 483 647)
     */
    public static boolean estUnEntier(String chaine) 
    {
        try 
        {
            Integer.parseInt(chaine);
        } 
        catch (NumberFormatException e)
        {
            return false;
        }
        return true;
    }   
    
    /**
     * La première lettre est en majuscule, les autres en minuscule 
     * @param value la chaine à traiter
     * @return      la chaine traitée
     */
	public static String PremiereLettreEnMajuscule(String value) 
	{
		if (value == null)  {return null;}
		switch (value.length())
		{
			case 0 : return value;
			case 1 : return value.toUpperCase();
			default : return value.substring(0,1).toUpperCase() + value.substring(1).toLowerCase();
		}
	}    
}
